<?php
// controllers/AuthController.php

/**
 * AuthController for Rehlko Customer Care application.
 * Handles user authentication processes like login and password reset.
 */
class AuthController {
    private $conn;
    private $userModel;
    private $mailer;
    private $notificationModel;

    /**
     * Constructor for AuthController.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
        $this->userModel = new User($db); // Instantiate User model
        $this->mailer = new Mailer();     // Instantiate Mailer utility
        $this->notificationModel = new Notification($db); // Instantiate Notification model
    }

    /**
     * Handles user login.
     * Expects 'email' (or username) and 'password' in the POST request body.
     */
    public function login() {
        // Decode JSON input from the request body
        $data = json_decode(file_get_contents("php://input"));

        // Validate required input fields
        if (empty($data->unique) || empty($data->password)) {
            sendJsonResponse(['message' => 'Email/Username and password are required.'], 400);
        }

        $unique = $data->unique;
        $password = $data->password;

        // Find user by email
        $user = $this->userModel->findByEmailorUsername($unique);

        // Check if user exists and password is correct
        if ($user && verifyPassword($password, $user['password']) && $user['is_active'] == 1) {
            // Generate JWT token upon successful login
            $token = generateJwt($user['id'], $user['role'], $user['email'], $user['fullname']);

            // Return success response with token and user details
            sendJsonResponse([
                'message' => 'Login successful.',
                'token' => $token,
                'success'=>true,
                'user' => [
                    'id' => $user['id'],
                    'username' => $user['username'],
                    'fullname' => $user['fullname'],
                    'email' => $user['email'],
                    'role' => $user['role'],
                    'country_id' => $user['country_id'] // Include country_id for frontend use
                ]
            ]);
        } else {
            // Return error for invalid credentials
            sendJsonResponse(['message' => 'Invalid email/username or password.Or Your account has been deleted.'], 401);
        }
    }

    /**
     * Handles the 'forgot password' request.
     * Expects 'email' in the POST request body.
     * Sends a password reset email (in a real app, this would be a link to set new password).
     */
    public function forgotPassword() {
        // Decode JSON input
        $data = json_decode(file_get_contents("php://input"));

        // Validate required input
        if (empty($data->email)) {
            sendJsonResponse(['message' => 'Email is required.'], 400);
        }

        $email = $data->email;

        // Find user by email
        $user = $this->userModel->findByEmailorUsername($email);

        if ($user) {
            // In a real application, you would generate a unique, time-limited token
            // and send a link to a password reset page (e.g., https://app.rehlkocustomercare.com/reset-password?token=XYZ)
            // For this simplified example, we'll generate a temporary password and email it.
            // THIS IS NOT RECOMMENDED FOR PRODUCTION. A secure reset link is preferred.

            $temporaryPassword = bin2hex(random_bytes(8)); // Generate a random 16-character hex password
            $hashedTemporaryPassword = hashPassword($temporaryPassword);

            // Update user's password in the database
            if ($this->userModel->updatePassword($user['id'], $hashedTemporaryPassword)) {
                // Prepare and send the email
                $emailBody = $this->mailer->getEmailTemplate('forgot_password', [
                    'fullname' => $user['fullname'],
                    'temporary_password' => $temporaryPassword,
                    'reset_link' => APP_BASE_URL . '/reset-password' // Placeholder for actual reset link
                ]);

                if ($this->mailer->sendMail($user['email'], $user['fullname'], 'Password Reset Request', $emailBody)) {
                    // Create notification for the user
                    $this->notificationModel->createNotification(
                        $user['id'],
                        'Password Reset Initiated',
                        'A password reset request was made for your account. Please check your email for instructions.',
                        'security'
                    );
                    sendJsonResponse(['message' => 'Password reset instructions sent to your email.'], 200);
                } else {
                    sendJsonResponse(['message' => 'Failed to send password reset email. Please try again later.'], 500);
                }
            } else {
                sendJsonResponse(['message' => 'Failed to update password in the database.'], 500);
            }
        } else {
            // It's often better practice not to disclose if an email exists for security reasons
            sendJsonResponse(['message' => 'If your email is registered, password reset instructions will be sent.'], 200);
        }
    }
}
?>
