<?php
// controllers/CountryController.php

/**
 * CountryController for Rehlko Customer Care application.
 * Manages CRUD operations for country records.
 * Access is restricted to 'superadmin' and 'admin' roles for create, update, delete.
 * Read access can be broader if needed for forms (e.g., user registration).
 */
class CountryController {
    private $conn;
    private $countryModel;

    /**
     * Constructor for CountryController.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
        $this->countryModel = new Country($db); // Instantiate Country model
    }

    /**
     * Creates a new country.
     * Only 'superadmin' and 'admin' roles are authorized.
     */
    public function create() {
        // Check authorization
        $currentUserRole = getCurrentUserRole();
        if (!in_array($currentUserRole, ['superadmin', 'admin'])) {
            sendJsonResponse(['message' => 'Unauthorized. Only superadmin or admin can create countries.'], 403);
        }

        $data = json_decode(file_get_contents("php://input"));

        // Validate required fields
        if (empty($data->country_name) || empty($data->country_name_code) || empty($data->country_phone_code)) {
            sendJsonResponse(['message' => 'Missing required fields: country_name, country_name_code, country_phone_code.'], 400);
        }

        // Set country properties
        $this->countryModel->country_name = $data->country_name;
        $this->countryModel->country_name_code = $data->country_name_code;
        $this->countryModel->country_phone_code = $data->country_phone_code;

        if ($this->countryModel->create()) {
            sendJsonResponse(['message' => 'Country created successfully.', 'countryId' => $this->countryModel->id], 201);
        } else {
            sendJsonResponse(['message' => 'Failed to create country.'], 500);
        }
    }

    /**
     * Reads all countries.
     * All authenticated users can read countries (e.g., for dropdowns in user forms).
     */
    public function readAll() {
        // No specific role check here for readAll, as countries are often needed for dropdowns.
        // If strict read access is needed, add:
        // $currentUserRole = getCurrentUserRole();
        // if (!$currentUserRole) { sendJsonResponse(['message' => 'Authentication required.'], 401); }

        $result = $this->countryModel->readAll();

        if ($result) {
            $countries_arr = [];
            while ($row = $result->fetch_assoc()) {
                $countries_arr[] = $row;
            }
            sendJsonResponse(['data'=>$countries_arr]);
        } else {
            sendJsonResponse(['message' => 'Failed to retrieve countries.'], 500);
        }
    }

    /**
     * Reads a single country by ID.
     * All authenticated users can read a specific country.
     *
     * @param int $id The ID of the country to retrieve.
     */
    public function read($id) {
        // No specific role check here for readOne, as countries are often needed for dropdowns.
        // If strict read access is needed, add:
        // $currentUserRole = getCurrentUserRole();
        // if (!$currentUserRole) { sendJsonResponse(['message' => 'Authentication required.'], 401); }

        $this->countryModel->id = $id;
        $country = $this->countryModel->readOne();

        if ($country) {
            sendJsonResponse($country);
        } else {
            sendJsonResponse(['message' => 'Country not found.'], 404);
        }
    }

    /**
     * Updates an existing country.
     * Only 'superadmin' and 'admin' roles are authorized.
     *
     * @param int $id The ID of the country to update.
     */
    public function update($id) {
        // Check authorization
        $currentUserRole = getCurrentUserRole();
        if (!in_array($currentUserRole, ['superadmin', 'admin'])) {
            sendJsonResponse(['message' => 'Unauthorized. Only superadmin or admin can update countries.'], 403);
        }

        $data = json_decode(file_get_contents("php://input"));

        // Validate required fields
        if (empty($data->country_name) || empty($data->country_name_code) || empty($data->country_phone_code)) {
            sendJsonResponse(['message' => 'Missing required fields.'], 400);
        }

        // Set country properties
        $this->countryModel->country_name = $data->country_name;
        $this->countryModel->country_name_code = $data->country_name_code;
        $this->countryModel->country_phone_code = $data->country_phone_code;
        $this->countryModel->id = $id;

        if ($this->countryModel->update()) {
            sendJsonResponse(['message' => 'Country updated successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to update country.'], 500);
        }
    }

    /**
     * Deletes a country.
     * Only 'superadmin' and 'admin' roles are authorized.
     *
     * @param int $id The ID of the country to delete.
     */
    public function delete($id) {
        // Check authorization
        $currentUserRole = getCurrentUserRole();
        if (!in_array($currentUserRole, ['superadmin', 'admin'])) {
            sendJsonResponse(['message' => 'Unauthorized. Only superadmin or admin can delete countries.'], 403);
        }

        $this->countryModel->id = $id;

        if ($this->countryModel->delete()) {
            sendJsonResponse(['message' => 'Country deleted successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to delete country or country not found.'], 500);
        }
    }
}
?>
