<?php
// controllers/EffectiveActionController.php
// This is a hypothetical file based on the problem description.
// Please apply these changes to your actual EffectiveActionController.php file.

/**
 * EffectiveActionController for Rehlko Customer Care application.
 * Manages CRUD operations for effective action entries linked to reports.
 */
class EffectiveActionController {
    private $conn;
    private $effectiveActionModel;
    private $reportModel; // To validate report_index and engineer assignment

    /**
     * Constructor for EffectiveActionController.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
        // Assuming you have an EffectiveAction model
        $this->effectiveActionModel = new EffectiveAction($db);
        $this->reportModel = new Report($db); // Assuming Report model exists
    }

    /**
     * Creates a new effective action for a report.
     */
    public function create() {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        if ($currentUserRole !== 'engineer') {
            sendJsonResponse(['message' => 'Unauthorized. Only engineers can create effective actions.'], 403);
            return;
        }

        $data = json_decode(file_get_contents("php://input"));
        $report_index = $_POST['report_index'] ?? ($data['report_index'] ?? null);
        $effective_action_title = $_POST['effective_action_title'] ?? ($data['effective_action_title'] ?? null);
        
        print_r($_POST);
        // Validate required fields
        if (empty($report_index) || empty($effective_action_title)) {
            sendJsonResponse(['message' => 'Missing required fields: report_index, effective_action_title. HAHAH'], 400);
            return;
        }

        // Verify report existence and engineer authorization
        $report = $this->reportModel->readOne($report_index, 'index');
        if (!$report) {
            sendJsonResponse(['message' => 'Report not found.'], 404);
            return;
        }
        if ($report['engineer_id'] != $currentUserId) {
            sendJsonResponse(['message' => 'Unauthorized. You are not assigned to this report.'], 403);
            return;
        }
        if ($report['status'] !== 'solving') {
            sendJsonResponse(['message' => 'Effective actions can only be added when report is in "solving" status.'], 400);
            return;
        }

        // Set model properties (adjust property names as per your EffectiveAction model)
        $this->effectiveActionModel->report_index = $report_index;
        $this->effectiveActionModel->effective_action_title = $effective_action_title;
        // Add other properties like media if applicable
        $this->effectiveActionModel->created_user = $currentUserId;

        if ($this->effectiveActionModel->create()) {
            sendJsonResponse(['success' => true, 'message' => 'Effective action created successfully.', 'id' => $this->effectiveActionModel->id], 201);
        } else {
            sendJsonResponse(['message' => 'Failed to create effective action.'], 500);
        }
    }

    /**
     * Reads effective action entries for a given report index.
     * This method is crucial for fixing the TypeError.
     *
     * @param string $reportIndex The report index to retrieve effective actions for.
     * @return void
     */
    public function readAllByReportIndex($reportIndex) {
        $currentUserRole = getCurrentUserRole();
        if (!$currentUserRole) {
            sendJsonResponse(['message' => 'Authentication required.'], 401);
            return;
        }

        // Verify user authorization to view this report first
        $report = $this->reportModel->readOne($reportIndex, 'index');
        if (!$report) {
            sendJsonResponse(['message' => 'Report not found or unauthorized to view.'], 404);
            return;
        }

        // Assuming your EffectiveAction model has a method like readByReportIndex
        $effectiveActions = $this->effectiveActionModel->readByReportIndex($reportIndex);

        if ($effectiveActions !== false) {
            // THIS IS THE CRITICAL FIX: Wrap the data in a 'success' and 'data' key
            sendJsonResponse(['success' => true, 'data' => $effectiveActions], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to retrieve effective actions.'], 500);
        }
    }

    /**
     * Reads a single effective action record by ID.
     *
     * @param int $id The ID of the effective action to read.
     */
    public function read($id) {
        $currentUserRole = getCurrentUserRole();
        if (!$currentUserRole) {
            sendJsonResponse(['message' => 'Authentication required.'], 401);
            return;
        }

        $this->effectiveActionModel->id = $id;
        $effectiveAction = $this->effectiveActionModel->readOne();

        if (!$effectiveAction) {
            sendJsonResponse(['message' => 'Effective action not found or unauthorized to view.'], 404);
            return;
        }

        // Verify user authorization to view this report first
        $report = $this->reportModel->readOne($effectiveAction['report_index'], 'index');
        if (!$report) {
            sendJsonResponse(['message' => 'Report not found or unauthorized to view.'], 404);
            return;
        }

        sendJsonResponse($effectiveAction, 200);
    }

    /**
     * Updates an existing effective action.
     *
     * @param int $id The ID of the effective action to update.
     */
    public function update($id) {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        if ($currentUserRole !== 'engineer') {
            sendJsonResponse(['message' => 'Unauthorized. Only engineers can update effective actions.'], 403);
            return;
        }

        $data = json_decode(file_get_contents("php://input"));

        if (empty($data->effective_action_title)) {
            sendJsonResponse(['message' => 'Missing required field: effective_action_title.'], 400);
            return;
        }

        // Fetch the effective action to get its report_index
        $this->effectiveActionModel->id = $id;
        $effectiveAction = $this->effectiveActionModel->readOne();
        if (!$effectiveAction) {
            sendJsonResponse(['message' => 'Effective action not found.'], 404);
            return;
        }

        // Verify engineer authorization for the associated report
        $report = $this->reportModel->readOne($effectiveAction['report_index'], 'index');
        if (!$report || $report['engineer_id'] != $currentUserId) {
            sendJsonResponse(['message' => 'Unauthorized. You are not assigned to the report associated with this effective action.'], 403);
            return;
        }
        if ($report['status'] !== 'solving') {
            sendJsonResponse(['message' => 'Effective actions can only be updated when report is in "solving" status.'], 400);
            return;
        }

        // Set model properties for update
        $this->effectiveActionModel->effective_action_title = $data->effective_action_title;
        // Add other properties like media if applicable
        $this->effectiveActionModel->id = $id;

        if ($this->effectiveActionModel->update()) {
            sendJsonResponse(['success' => true, 'message' => 'Effective action updated successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to update effective action.'], 500);
        }
    }

    /**
     * Deletes an effective action.
     *
     * @param int $id The ID of the effective action to delete.
     */
    public function delete($id) {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        // Fetch the effective action to get its report_index
        $this->effectiveActionModel->id = $id;
        $effectiveAction = $this->effectiveActionModel->readOne();
        if (!$effectiveAction) {
            sendJsonResponse(['message' => 'Effective action not found.'], 404);
            return;
        }

        // Authorize: Superadmin/Admin or the assigned engineer
        $authorized = false;
        if (in_array($currentUserRole, ['superadmin', 'admin'])) {
            $authorized = true;
        } else {
            $report = $this->reportModel->readOne($effectiveAction['report_index'], 'index');
            if ($report && $report['engineer_id'] == $currentUserId) {
                 $authorized = true;
            }
        }

        if (!$authorized) {
            sendJsonResponse(['message' => 'Unauthorized to delete this effective action.'], 403);
            return;
        }

        // Pass the ID to the delete method of the model
        if ($this->effectiveActionModel->delete()) { // Assuming model's delete method uses $this->id
            sendJsonResponse(['success' => true, 'message' => 'Effective action deleted successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to delete effective action.'], 500);
        }
    }
}
?>