<?php
// controllers/MaintenanceController.php

/**
 * MaintenanceController for Rehlko Customer Care application.
 * Manages maintenance record operations.
 * Update access for all roles except customers.
 * Read access is role-based and handled by the model.
 */
class MaintenanceController {
    private $conn;
    private $maintenanceModel;
    private $notificationModel;

    /**
     * Constructor for MaintenanceController.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
        $this->maintenanceModel = new Maintenance($db); // Instantiate Maintenance model
        $this->notificationModel = new Notification($db);
        $this->userModel = new User($db);// Instantiate Notification model
    }

    /**
     * Reads all maintenance records.
     * Access is restricted based on user role and team assignments, handled by the model.
     */
    public function readAll() {
        $currentUserRole = getCurrentUserRole();
        if (!$currentUserRole) {
            sendJsonResponse(['message' => 'Authentication required.'], 401);
        }

        $result = $this->maintenanceModel->readAll(); // Authorization is handled within the model

        if ($result) {
            $maintenances_arr = [];
            while ($row = $result->fetch_assoc()) {
                $maintenances_arr[] = $row;
            }
            sendJsonResponse(['data'=>$maintenances_arr]);
        } else {
            sendJsonResponse(['message' => 'Failed to retrieve maintenances or unauthorized.'], 403);
        }
    }

    /**
     * Reads a single maintenance record by ID.
     * Access is restricted based on user role and team assignments, handled by the model.
     *
     * @param int $id The ID of the maintenance record to retrieve.
     */
    public function read($id) {
        $currentUserRole = getCurrentUserRole();
        if (!$currentUserRole) {
            sendJsonResponse(['message' => 'Authentication required.'], 401);
        }

        $this->maintenanceModel->id = $id;
        $maintenance = $this->maintenanceModel->readOne(); // Authorization logic is inside readOne model method

        if ($maintenance) {
            sendJsonResponse(['data'=>$maintenance]);
        } else {
            sendJsonResponse(['message' => 'Maintenance record not found or unauthorized.'], 404);
        }
    }

    /**
     * Updates an existing maintenance record.
     * All roles except 'localcustomer' and 'globalcustomer' are authorized.
     * After update, notifies the product owner (customer).
     *
     * @param int $id The ID of the maintenance record to update.
     */
    public function update($id) {
        // Check authorization
        $currentUserRole = getCurrentUserRole();
        if (in_array($currentUserRole, ['localcustomer', 'globalcustomer'])) {
            sendJsonResponse(['message' => 'Unauthorized. Customers cannot update maintenance records.'], 403);
        }

        $data = json_decode(file_get_contents("php://input"));

        // Validate required fields for update
        if (empty($data->generator_serial_number) || empty($data->maintenance_title) || empty($data->maintenance_type) || empty($data->maintence_date)) {
            sendJsonResponse(['message' => 'Missing required fields for maintenance update.'], 400);
        }

        // Check if maintenance record exists
        $this->maintenanceModel->id = $id;
        $existingMaintenance = $this->maintenanceModel->readOne(); // This also handles auth.
        if (!$existingMaintenance) {
            sendJsonResponse(['message' => 'Maintenance record not found or unauthorized.'], 404);
        }

        // Set properties for update
        $this->maintenanceModel->generator_serial_number = $data->generator_serial_number;
        $this->maintenanceModel->maintenance_title = $data->maintenance_title;
        $this->maintenanceModel->maintenance_description = $data->maintenance_description ?? null;
        $this->maintenanceModel->maintenance_type = $data->maintenance_type;
        $this->maintenanceModel->maintenance_check_user = getCurrentUserId(); // User performing the update
        $this->maintenanceModel->maintence_date = $data->maintence_date;
        $this->maintenanceModel->id = $id;

        if ($this->maintenanceModel->update()) {
            // After successful update, notify the product owner (customer)
            $customerUser = $this->userModel->readById($existingMaintenance['customer_id']); // Get customer details
            if ($customerUser) {
                // Send email
                $emailBody = (new Mailer())->getEmailTemplate('customer_maintenance_update', [
                    'generator_serial_number' => $this->maintenanceModel->generator_serial_number,
                    'maintenance_title' => $this->maintenanceModel->maintenance_title,
                    'maintenance_date' => $this->maintenanceModel->maintence_date,
                    'customer_name' => $customerUser['fullname'],
                    'app_link' => APP_BASE_URL . '/app/maintenance' // Placeholder for app link
                ]);
                (new Mailer())->sendMail(
                    $customerUser['email'],
                    $customerUser['fullname'],
                    'Maintenance Update for Your Generator',
                    $emailBody
                );

                // Create in-app notification
                $this->notificationModel->createNotification(
                    $customerUser['id'],
                    'Maintenance Update',
                    'Maintenance for your generator ' . $this->maintenanceModel->generator_serial_number . ' has been updated.',
                    'maintenance'
                );
            }

            sendJsonResponse(['message' => 'Maintenance record updated successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to update maintenance record.'], 500);
        }
    }

    /**
     * Deletes a maintenance record.
     * All roles except 'localcustomer' and 'globalcustomer' are authorized.
     *
     * @param int $id The ID of the maintenance record to delete.
     */
    public function delete($id) {
        // Check authorization
        $currentUserRole = getCurrentUserRole();
        if (in_array($currentUserRole, ['localcustomer', 'globalcustomer'])) {
            sendJsonResponse(['message' => 'Unauthorized. Customers cannot delete maintenance records.'], 403);
        }

        $this->maintenanceModel->id = $id;

        if ($this->maintenanceModel->delete()) {
            sendJsonResponse(['message' => 'Maintenance record deleted successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to delete maintenance record or not found.'], 500);
        }
    }
}
?>
