<?php
// controllers/PreventionController.php

/**
 * PreventionController for Rehlko Customer Care application.
 * Manages CRUD operations for prevention entries linked to reports.
 * Only engineers assigned to a report can perform CRUD on its prevention steps.
 */
class PreventionController {
    private $conn;
    private $preventionModel;
    private $reportModel; // To validate report_index and engineer assignment

    /**
     * Constructor for PreventionController.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
        $this->preventionModel = new Prevention($db);
        $this->reportModel = new Report($db);
    }

    /**
     * Creates a new prevention record for a report.
     * Only engineers assigned to the report can create.
     */
    public function create() {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        if ($currentUserRole !== 'engineer') {
            sendJsonResponse(['message' => 'Unauthorized. Only engineers can create prevention records.'], 403);
        }

        $data = json_decode(file_get_contents("php://input"));

        // Validate required fields
        if (empty($data->report_index) || empty($data->prevention_title)) {
            sendJsonResponse(['message' => 'Missing required fields: report_index, prevention_title.'], 400);
        }

        // Verify report existence and engineer authorization
        $report = $this->reportModel->readOne($data->report_index, 'index');
        if (!$report) {
            sendJsonResponse(['message' => 'Report not found.'], 404);
        }
        if ($report['engineer_id'] != $currentUserId) {
            sendJsonResponse(['message' => 'Unauthorized. You are not assigned to this report.'], 403);
        }
        if ($report['status'] !== 'solving') {
            sendJsonResponse(['message' => 'Prevention steps can only be added when report is in "solving" status.'], 400);
        }

        // Set model properties
        $this->preventionModel->report_index = $data->report_index;
        $this->preventionModel->prevention_title = $data->prevention_title;
        $this->preventionModel->created_user = $currentUserId;

        if ($this->preventionModel->create()) {
            sendJsonResponse(['message' => 'Prevention record created successfully.', 'id' => $this->preventionModel->id], 201);
        } else {
            sendJsonResponse(['message' => 'Failed to create prevention record.'], 500);
        }
    }

    /**
     * Reads prevention entries for a given report index.
     * Access is based on report view permissions.
     *
     * @param string $reportIndex The report index to retrieve prevention steps for.
     */
    public function readByReportIndex($reportIndex) {
        $currentUserRole = getCurrentUserRole();
        if (!$currentUserRole) {
            sendJsonResponse(['message' => 'Authentication required.'], 401);
        }

        // Verify user authorization to view this report first
        $report = $this->reportModel->readOne($reportIndex, 'index');
        if (!$report) {
            sendJsonResponse(['message' => 'Report not found or unauthorized to view.'], 404);
        }

        $preventionActions = $this->preventionModel->readByReportIndex($reportIndex);

        if ($preventionActions !== false) {
            sendJsonResponse($preventionActions);
        } else {
            sendJsonResponse(['message' => 'Failed to retrieve prevention steps.'], 500);
        }
    }

    /**
     * Updates an existing prevention record.
     * Only the engineer assigned to the report can update.
     *
     * @param int $id The ID of the prevention record to update.
     */
    public function update($id) {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        if ($currentUserRole !== 'engineer') {
            sendJsonResponse(['message' => 'Unauthorized. Only engineers can update prevention records.'], 403);
        }

        $data = json_decode(file_get_contents("php://input"));

        if (empty($data->prevention_title)) {
            sendJsonResponse(['message' => 'Missing required field: prevention_title.'], 400);
        }

        // Fetch the prevention record to get its report_index
        $this->preventionModel->id = $id;
        $prevention = $this->preventionModel->readOne();
        if (!$prevention) {
            sendJsonResponse(['message' => 'Prevention record not found.'], 404);
        }

        // Verify engineer authorization for the associated report
        $report = $this->reportModel->readOne($prevention['report_index'], 'index');
        if (!$report || $report['engineer_id'] != $currentUserId) {
            sendJsonResponse(['message' => 'Unauthorized. You are not assigned to the report associated with this prevention record.'], 403);
        }
        if ($report['status'] !== 'solving') {
            sendJsonResponse(['message' => 'Prevention steps can only be updated when report is in "solving" status.'], 400);
        }

        // Set model properties for update
        $this->preventionModel->prevention_title = $data->prevention_title;
        $this->preventionModel->id = $id;

        if ($this->preventionModel->update()) {
            sendJsonResponse(['message' => 'Prevention record updated successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to update prevention record.'], 500);
        }
    }

    /**
     * Deletes a prevention record.
     * Only the engineer assigned to the report can delete, or superadmin/admin.
     *
     * @param int $id The ID of the prevention record to delete.
     */
    public function delete($id) {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        // Fetch the prevention record to get its report_index
        $this->preventionModel->id = $id;
        $prevention = $this->preventionModel->readOne();
        if (!$prevention) {
            sendJsonResponse(['message' => 'Prevention record not found.'], 404);
        }

        // Authorize: Superadmin/Admin or the assigned engineer
        $authorized = false;
        if (in_array($currentUserRole, ['superadmin', 'admin'])) {
            $authorized = true;
        } else {
            $report = $this->reportModel->readOne($prevention['report_index'], 'index');
            if ($report && $report['engineer_id'] == $currentUserId) {
                 $authorized = true;
            }
        }

        if (!$authorized) {
            sendJsonResponse(['message' => 'Unauthorized to delete this prevention record.'], 403);
        }

        if ($this->preventionModel->delete()) {
            sendJsonResponse(['message' => 'Prevention record deleted successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to delete prevention record.'], 500);
        }
    }
}
?>
