<?php
// controllers/ReportController.php

/**
 * ReportController for Rehlko Customer Care application.
 * Manages report lifecycle: creation, picking up, solving, checking, and completing.
 * Handles role-based access control for each action.
 */
class ReportController {
    private $conn;
    private $reportModel;
    private $userModel;
    private $supportTeamModel;
    private $notificationModel;
    private $reportMediaModel;
    private $mailer;
    private $pdfGenerator;

    /**
     * Constructor for ReportController.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
        $this->reportModel = new Report($db);
        $this->userModel = new User($db); // Needed for sending notifications to engineers/admins
        $this->supportTeamModel = new SupportTeam($db); // Needed to find assigned engineer
        $this->notificationModel = new Notification($db);
        $this->mailer = new Mailer();
        $this->pdfGenerator = new PdfGenerator(); // For report PDF generation
        $this->reportMediaModel = new ReportMedia($db);
    }

    /**
     * Creates a new report.
     * Only 'localcustomer' and 'globalcustomer' roles are authorized.
     */
    public function create(): void
    {
        // Get the current user's role and ID for authorization and data assignment.
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        // 1. Role Authorization: Ensure only customers can create reports.
        if (!in_array($currentUserRole, ['localcustomer', 'globalcustomer'])) {
            sendJsonResponse(['message' => 'Unauthorized. Only customers can create reports.'], 403);
            return;
        }

        // 2. Retrieve Data from $_POST and $_FILES (already correctly handled for multipart/form-data)
        $postData = $_POST;
        $filesData = $_FILES;

        // 3. Input Validation: Check for required fields from $_POST.
        if (empty($postData['problem_issue'])) {
            sendJsonResponse(['message' => 'Missing required fields: problem_issue.'], 400);
            return;
        }

        // 4. Populate Report Model: Assign data from $_POST to the report object.
        $this->reportModel->customer_id = $currentUserId;
        $this->reportModel->report_index = $this->reportModel->generateUniqueReportIndex();
        $this->reportModel->title = $postData['problem_issue'];
        $this->reportModel->description = $postData['problem_issue'];
        $this->reportModel->status = 'pending';

        // Optional fields, using null coalescing operator to set defaults if not provided in payload
        $this->reportModel->category_id = $postData['category_id'] ?? null;
        $this->reportModel->report_type = $postData['report_type'] ?? 'normal';
        $this->reportModel->generator_serial_number = $postData['generator_serial_number'] ?? null;
        $this->reportModel->running_hours = $postData['running_hours'] ?? null;
        $this->reportModel->load_test = $postData['load_test'] ?? null;
        $this->reportModel->load_hour = $postData['load_hour'] ?? null;
        $this->reportModel->load_amount = $postData['load_amount'] ?? null;
        $this->reportModel->used_for = $postData['used_for'] ?? 'stand by';
        $this->reportModel->error_code = $postData['error_code'] ?? null;
        $this->reportModel->test_run = $postData['test_run'] ?? null;
        $this->reportModel->location = $postData['location'] ?? null;
        $this->reportModel->layer = $postData['layer'] ?? null;
        $this->reportModel->remarks = $postData['remarks'] ?? null;

        // Important: These fields are now managed by ReportMedia model, set to null initially
        $this->reportModel->image_paths = null;
        $this->reportModel->video_path = null;


        // 5. Attempt to Create Report: Call the model's create method.
        if ($this->reportModel->create()) {
            $createdReportId = $this->reportModel->id; // Assuming the create method sets the ID of the new report
            $createdReportIndex = $this->reportModel->report_index;

            $uploadedImageUrls = [];
            $uploadedVideoUrl = null;

            // Handle multiple images (report_images[] from Flutter)
            if (isset($filesData['report_images']) && is_array($filesData['report_images']['name'])) {
                foreach ($filesData['report_images']['name'] as $key => $name) {
                    // Check for upload errors for each individual file
                    if ($filesData['report_images']['error'][$key] === UPLOAD_ERR_OK) {
                        // Create a single file entry array compatible with ReportMedia->uploadMedia
                        $fileEntry = [
                            'name' => $filesData['report_images']['name'][$key],
                            'type' => $filesData['report_images']['type'][$key],
                            'tmp_name' => $filesData['report_images']['tmp_name'][$key],
                            'error' => $filesData['report_images']['error'][$key],
                            'size' => $filesData['report_images']['size'][$key],
                        ];
                        $uploadedMedia = $this->reportMediaModel->uploadMedia(
                            $fileEntry,
                            $createdReportIndex,
                            'image' // media_type
                        );
                        if ($uploadedMedia) {
                            $uploadedImageUrls[] = $uploadedMedia['media_path']; // Store the path returned by the model
                        } else {
                            error_log("Failed to upload image for report " . $createdReportIndex . " - File: " . $name);
                        }
                    } else {
                        error_log("Image upload error for file " . $name . ": " . $filesData['report_images']['error'][$key]);
                    }
                }
            }

            // Handle single video (report_video from Flutter)
            if (isset($filesData['report_video']) && $filesData['report_video']['error'] === UPLOAD_ERR_OK) {
                $uploadedMedia = $this->reportMediaModel->uploadMedia(
                    $filesData['report_video'], // Pass the full $_FILES entry for the video
                    $createdReportIndex,
                    'video' // media_type
                );
                if ($uploadedMedia) {
                    $uploadedVideoUrl = $uploadedMedia['media_path']; // Store the path returned by the model
                } else {
                    error_log("Failed to upload video for report " . $createdReportIndex . " - File: " . $filesData['report_video']['name']);
                }
            } else if (isset($filesData['report_video'])) { // Log error if video file was sent but had an error
                error_log("Video upload error: " . $filesData['report_video']['error']);
            }

            // 6. Handle Success: Send notifications.
            $customer = $this->userModel->readById($currentUserId);
            if ($customer) {
                $this->notificationModel->createNotification(
                    $currentUserId,
                    'Report Created',
                    'Your report (' . $createdReportIndex . ') has been successfully created and is awaiting assignment.',
                    'reports'
                );
            }

            // Find the assigned engineer from the support team for this customer
            $supportTeam = $this->supportTeamModel->getByCustomerId($this->reportModel->customer_id);
            if ($supportTeam && !empty($supportTeam['engineer_id'])) {
                $engineer = $this->userModel->readById($supportTeam['engineer_id']);
                if ($engineer) {
                    // Send notification to the assigned engineer
                    $this->notificationModel->createNotification(
                        $engineer['id'],
                        'New Report Assigned',
                        'A new report (' . $createdReportIndex . ') has been assigned to you. Please pick it up.',
                        'reports'
                    );

                    // Send email to the assigned engineer
                    $this->mailer->sendMail(
                        $engineer['email'],
                        $engineer['fullname'],
                        'New Customer Care Report: ' . $createdReportIndex,
                        'Dear ' . $engineer['fullname'] . ',<br><br>' .
                        'A new customer care report (' . $createdReportIndex . ') has been created and assigned to you.' .
                        '<br><br>Please log in to the application to review and pick up the report.' .
                        '<br><br>Report Title: ' . $this->reportModel->title .
                        '<br>Description: ' . $this->reportModel->description .
                        '<br><br>Thank you,<br>Rehlko Customer Care Team',
                        null // Assuming no attachments
                    );
                }
            } else {
                // If no engineer is assigned or found for this customer, log for manual assignment or notify admin.
                error_log("CRON: No engineer assigned to customer_id " . $this->reportModel->customer_id . " for report " . $createdReportIndex . ". Manual assignment may be needed.");
                // Optionally, send a notification to a default admin/supervisor here if needed.
            }

            // Send success JSON response, including uploaded file URLs
            sendJsonResponse([
                'message' => 'Report created successfully.',
                'report_index' => $createdReportIndex,
                'image_paths' => $uploadedImageUrls, // Send URLs of uploaded images
                'video_path' => $uploadedVideoUrl // Send URL of uploaded video
            ], 201);
        } else {
            // 7. Handle Failure: Send server error JSON response.
            sendJsonResponse(['message' => 'Failed to create report.'], 500);
        }
    }
    /**
     * Allows an engineer to pick up a report.
     * Only 'engineer', 'admin', 'superadmin', 'supervisor' roles are authorized.
     */
    public function pickupReport($id) {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        if (!in_array($currentUserRole, ['engineer', 'admin', 'superadmin', 'supervisor'])) {
            sendJsonResponse(['message' => 'Unauthorized. Only engineers, admins, supervisors, or superadmins can pick up reports.'], 403);
            return;
        }

        $this->reportModel->id = $id;
        $report = $this->reportModel->readOne($this->reportModel->id, 'id');

        if (!$report) {
            sendJsonResponse(['message' => 'Report not found.'], 404);
            return;
        }

        if ($report['status'] !== 'pending') {
            sendJsonResponse(['message' => 'Report cannot be picked up. Current status is ' . $report['status'] . '.'], 400);
            return;
        }

        $this->reportModel->engineer_id = $currentUserId;
        $this->reportModel->status = 'solving';
        $this->reportModel->picked_up_time = date('Y-m-d H:i:s'); // Set picked up time
        $isPickedUp = true;

        if ($this->reportModel->updateStatus($id,$this->reportModel->status,$this->reportModel->engineer_id,null,true)) {
            // Notify the customer that the report has been picked up
            $customer = $this->userModel->readById($report['customer_id']);
            $engineer = $this->userModel->readById($this->reportModel->engineer_id);
            if ($customer) {
                $this->notificationModel->createNotification(
                    $customer['id'],
                    'Report Picked Up',
                    'Your report (' . $report['report_index'] . ') has been picked up by ' . $engineer['fullname'] . '.',
                    'reports'
                );
                // Optionally send an email as well
            }
            sendJsonResponse(['message' => 'Report picked up successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to pick up report.'], 500);
        }
    }

    /**
     * Allows an engineer to solve a report.
     * Only 'engineer', 'admin', 'superadmin', 'supervisor' roles are authorized.
     */
    public function solve($id) {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        if (!in_array($currentUserRole, ['engineer', 'admin', 'superadmin', 'supervisor'])) {
            sendJsonResponse(['message' => 'Unauthorized. Only engineers, admins, supervisors, or superadmins can solve reports.'], 403);
            return;
        }

        $data = json_decode(file_get_contents("php://input"));
        

        $this->reportModel->id = $id;
        $report = $this->reportModel->readOne($this->reportModel->id, 'id');

        if (!$report) {
            sendJsonResponse(['message' => 'Report not found.'], 404);
            return;
        }

        

        
        $this->reportModel->status = 'checking';
        $this->reportModel->solved_time = date('Y-m-d H:i:s'); // Set solved time

        if ($this->reportModel->updateStatus($id,$this->reportModel->status,$this->reportModel->engineer_id,null,true)) {
            // Notify the customer that the report has been solved
            $customer = $this->userModel->readById($report['customer_id']);
            $engineer = $this->userModel->readById($currentUserId);
            if ($customer) {
                $this->notificationModel->createNotification(
                    $customer['id'],
                    'Report Solved',
                    'Your report (' . $report['report_index'] . ') has been marked as solved by ' . $engineer['fullname'] . '. Please review the solution.',
                    'reports'
                );
                // Optionally send an email as well
            }
            sendJsonResponse(['message' => 'Report solved successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to solve report.'], 500);
        }
    }


    /**
     * Allows a supervisor or admin to complete a checked report.
     * Only 'supervisor', 'admin', 'superadmin' roles are authorized.
     */
    public function complete($id) {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();
        if (!in_array($currentUserRole, ['supervisor', 'admin', 'superadmin'])) {
            sendJsonResponse(['message' => 'Unauthorized. Only supervisors, admins, or superadmins can complete reports.'], 403);
            return;
        }

        $this->reportModel->id = $id;
        $report = $this->reportModel->readOne($this->reportModel->id, 'id');

        if (!$report) {
            sendJsonResponse(['message' => 'Report not found.'], 404);
            return;
        }

        if ($report['status'] !== 'checking') {
            sendJsonResponse(['message' => 'Report cannot be completed. Current status is ' . $report['status'] . '.'], 400);
            return;
        }

        $this->reportModel->status = 'completed';
        $this->reportModel->completed_time = date('Y-m-d H:i:s'); // Set completed time

        if ($this->reportModel->updateStatus($id,$this->reportModel->status,$this->reportModel->engineer_id,$currentUserId,true)) {
            // Notify the customer that the report has been completed
            $customer = $this->userModel->readById($report['customer_id']);
            if ($customer) {
                $this->notificationModel->createNotification(
                    $customer['id'],
                    'Report Completed',
                    'Your report (' . $report['report_index'] . ') has been completed. Thank you for your patience.',
                    'reports'
                );
                // Optionally generate PDF and send email with attachment
                $reportIndex = $report['report_index'] ;
                $reportData = $this->reportModel->getReportDetailsForPdf($reportIndex);
                $pdfPath = $this->pdfGenerator->generateReportPdf($id,$reportIndex,$reportData);
                if ($pdfPath && file_exists($pdfPath)) {
                    $this->mailer->sendMail(
                        $customer['email'],
                        $customer['fullname'],
                        'Your Report ' . $report['report_index'] . ' is Completed',
                        'Dear ' . $customer['fullname'] . ',<br><br>' .
                        'Your customer care report (' . $report['report_index'] . ') has been successfully completed.' .
                        '<br><br>Please find the attached PDF summary of the report details and solution.' .
                        '<br><br>Thank you,<br>Rehlko Customer Care Team',
                        $pdfPath // Attach the generated PDF
                    );
                } else {
                    error_log("Failed to generate or find PDF for completed report " . $report['report_index']);
                }
            }
            sendJsonResponse(['message' => 'Report completed successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to complete report.'], 500);
        }
    }

    /**
     * Reads a single report by its ID.
     * Access is restricted to users involved with the report (customer, engineer, champion, member)
     * and roles like 'admin', 'supervisor', 'superadmin'.
     *
     * @param int $id The ID of the report to retrieve.
     */
    public function readSingle($id) {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        $this->reportModel->id = $id;
        $report = $this->reportModel->readOne($this->reportModel->id, 'id');
        

        if (!$report) {
            sendJsonResponse(['message' => 'Report not found.'], 404);
            return;
        }
        $report_index = $report['report_index'];
        $report_media = $this->reportMediaModel->readByReportIndex($report_index);
        

        // Authorization check
        $authorized = false;
        if (in_array($currentUserRole, ['admin', 'supervisor', 'superadmin'])) {
            $authorized = true;
        } elseif ($report['customer_id'] == $currentUserId) { // Customer who created the report
            $authorized = true;
        } else {
            // Check if current user is the assigned engineer, champion, or member in the support team for this customer
            $supportTeam = $this->supportTeamModel->getByCustomerId($report['customer_id']); // Changed to getByCustomerId
            if ($supportTeam) {
                if ($supportTeam['engineer_id'] == $currentUserId ||
                    $supportTeam['champion_id'] == $currentUserId ||
                    $supportTeam['member_id'] == $currentUserId) {
                    $authorized = true;
                }
            }
        }

        if (!$authorized) {
            sendJsonResponse(['message' => 'Unauthorized to view this report.'], 403);
            return;
        }

        sendJsonResponse(['data'=>$report,'media'=>$report_media], 200);
    }

    /**
     * Reads all reports with optional filtering by status and search term.
     * Access is role-based.
     */
    public function readAll() {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        $filterStatus = $_GET['status'] ?? null;
        $searchTerm = $_GET['search'] ?? null;

        $reports = [];

        // Determine which reports the current user can see based on their role
        if (in_array($currentUserRole, ['superadmin', 'admin', 'supervisor'])) {
            // Superadmins, Admins, Supervisors can see all reports (filtered by status/search)
            $reports = $this->reportModel->readAll($filterStatus, $searchTerm);
        } elseif ($currentUserRole === 'engineer') {
            // Engineers can see reports assigned to them or unassigned reports they can pick up (status 'Pending')
            $reports = $this->reportModel->readByEngineerOrPending($currentUserId, $filterStatus, $searchTerm);
        } elseif (in_array($currentUserRole, ['localcustomer', 'globalcustomer'])) {
            // Customers can only see their own reports
            $reports = $this->reportModel->readByCustomer($currentUserId, $filterStatus, $searchTerm);
        } else {
            sendJsonResponse(['message' => 'Unauthorized to view reports.'], 403);
            return;
        }

        if ($reports !== false) {
            $reports_arr = [];
            while ($row = $reports->fetch_assoc()) {
                $reports_arr[] = $row;
            }
            sendJsonResponse(['data' => $reports_arr,'success'=>true]);
        } else {
            sendJsonResponse(['message' => 'Failed to retrieve reports.'], 500);
        }
    }
    
    public function recents() {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        $filterStatus = $_GET['status'] ?? null;
        $searchTerm = $_GET['search'] ?? null;

        $reports = [];

        // Determine which reports the current user can see based on their role
        if (in_array($currentUserRole, ['superadmin', 'admin', 'supervisor'])) {
            // Superadmins, Admins, Supervisors can see all reports (filtered by status/search)
            $reports = $this->reportModel->recents($filterStatus, $searchTerm);
        } elseif ($currentUserRole === 'engineer') {
            // Engineers can see reports assigned to them or unassigned reports they can pick up (status 'Pending')
            $reports = $this->reportModel->readByEngineerOrPending($currentUserId, $filterStatus, $searchTerm);
        } elseif (in_array($currentUserRole, ['localcustomer', 'globalcustomer'])) {
            // Customers can only see their own reports
            $reports = $this->reportModel->readByCustomer($currentUserId, $filterStatus, $searchTerm);
        } else {
            sendJsonResponse(['message' => 'Unauthorized to view reports.'], 403);
            return;
        }

        if ($reports !== false) {
            $reports_arr = [];
            while ($row = $reports->fetch_assoc()) {
                $reports_arr[] = $row;
            }
            sendJsonResponse(['data' => $reports_arr,'success'=>true]);
        } else {
            sendJsonResponse(['message' => 'Failed to retrieve reports.'], 500);
        }
    }

    /**
     * Generates a PDF for a specific report.
     * Access is restricted to 'superadmin', 'admin', 'supervisor', 'engineer' roles
     * and the customer who created the report.
     *
     * @param int $id The ID of the report to generate PDF for.
     */
    public function generatePdf($id) {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        $this->reportModel->id = $id;
        $report = $this->reportModel->readOne($this->reportModel->id, 'id');

        if (!$report) {
            sendJsonResponse(['message' => 'Report not found.'], 404);
            return;
        }

        // Authorization check for PDF generation
        $authorized = false;
        if (in_array($currentUserRole, ['superadmin', 'admin', 'supervisor', 'engineer'])) {
            $authorized = true;
        } elseif ($report['customer_id'] == $currentUserId) {
            $authorized = true;
        }

        if (!$authorized) {
            sendJsonResponse(['message' => 'Unauthorized to generate PDF for this report.'], 403);
            return;
        }

        $pdfPath = $this->pdfGenerator->generateReportPdf($id);

        if ($pdfPath && file_exists($pdfPath)) {
            header('Content-Type: application/pdf');
            header('Content-Disposition: attachment; filename="' . basename($pdfPath) . '"');
            readfile($pdfPath); // Output the file
            exit();
        } else {
            sendJsonResponse(['message' => 'PDF file not found or could not be generated.'], 500);
        }
    }

    /**
     * Deletes a report.
     * Only 'superadmin' and 'admin' roles are authorized.
     *
     * @param int $id The ID of the report to delete.
     */
    public function delete($id) {
        $currentUserRole = getCurrentUserRole();
        if (!in_array($currentUserRole, ['superadmin', 'admin'])) {
            sendJsonResponse(['message' => 'Unauthorized. Only superadmin or admin can delete reports.'], 403);
            return;
        }

        $this->reportModel->id = $id;

        // Optionally, add logic here to delete associated root_cause, corrective_action etc. records
        // If foreign key constraints are set with CASCADE DELETE in DB, this is handled automatically.

        if ($this->reportModel->delete()) {
            sendJsonResponse(['message' => 'Report deleted successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to delete report or report not found.'], 500);
        }
    }
}