<?php
// controllers/SuggestionController.php

/**
 * SuggestionController for Rehlko Customer Care application.
 * Manages CRUD operations for suggestion entries linked to reports.
 * Only engineers assigned to a report can perform CRUD on its suggestions.
 */
class SuggestionController {
    private $conn;
    private $suggestionModel;
    private $reportModel; // To validate report_index and engineer assignment

    /**
     * Constructor for SuggestionController.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
        $this->suggestionModel = new Suggestion($db);
        $this->reportModel = new Report($db);
    }

    /**
     * Creates a new suggestion for a report.
     * Only engineers assigned to the report can create.
     */
    public function create() {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        if ($currentUserRole !== 'engineer') {
            sendJsonResponse(['message' => 'Unauthorized. Only engineers can create suggestions.'], 403);
        }

        $data = json_decode(file_get_contents("php://input"));

        // Validate required fields
        if (empty($data->report_index) || empty($data->suggestion_title)) {
            sendJsonResponse(['message' => 'Missing required fields: report_index, suggestion_title.'], 400);
        }

        // Verify report existence and engineer authorization
        $report = $this->reportModel->readOne($data->report_index, 'index');
        if (!$report) {
            sendJsonResponse(['message' => 'Report not found.'], 404);
        }
        if ($report['engineer_id'] != $currentUserId) {
            sendJsonResponse(['message' => 'Unauthorized. You are not assigned to this report.'], 403);
        }
        if ($report['status'] !== 'solving') {
            sendJsonResponse(['message' => 'Suggestions can only be added when report is in "solving" status.'], 400);
        }

        // Set model properties
        $this->suggestionModel->report_index = $data->report_index;
        $this->suggestionModel->suggestion_title = $data->suggestion_title;
        $this->suggestionModel->created_user = $currentUserId;

        if ($this->suggestionModel->create()) {
            sendJsonResponse(['message' => 'Suggestion created successfully.', 'id' => $this->suggestionModel->id], 201);
        } else {
            sendJsonResponse(['message' => 'Failed to create suggestion.'], 500);
        }
    }

    /**
     * Reads suggestion entries for a given report index.
     * Access is based on report view permissions.
     *
     * @param string $reportIndex The report index to retrieve suggestions for.
     */
    public function readByReportIndex($reportIndex) {
        $currentUserRole = getCurrentUserRole();
        if (!$currentUserRole) {
            sendJsonResponse(['message' => 'Authentication required.'], 401);
        }

        // Verify user authorization to view this report first
        $report = $this->reportModel->readOne($reportIndex, 'index');
        if (!$report) {
            sendJsonResponse(['message' => 'Report not found or unauthorized to view.'], 404);
        }

        $suggestions = $this->suggestionModel->readByReportIndex($reportIndex);

        if ($suggestions !== false) {
            sendJsonResponse($suggestions);
        } else {
            sendJsonResponse(['message' => 'Failed to retrieve suggestions.'], 500);
        }
    }

    /**
     * Updates an existing suggestion.
     * Only the engineer assigned to the report can update.
     *
     * @param int $id The ID of the suggestion to update.
     */
    public function update($id) {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        if ($currentUserRole !== 'engineer') {
            sendJsonResponse(['message' => 'Unauthorized. Only engineers can update suggestions.'], 403);
        }

        $data = json_decode(file_get_contents("php://input"));

        if (empty($data->suggestion_title)) {
            sendJsonResponse(['message' => 'Missing required field: suggestion_title.'], 400);
        }

        // Fetch the suggestion to get its report_index
        $this->suggestionModel->id = $id;
        $suggestion = $this->suggestionModel->readOne();
        if (!$suggestion) {
            sendJsonResponse(['message' => 'Suggestion not found.'], 404);
        }

        // Verify engineer authorization for the associated report
        $report = $this->reportModel->readOne($suggestion['report_index'], 'index');
        if (!$report || $report['engineer_id'] != $currentUserId) {
            sendJsonResponse(['message' => 'Unauthorized. You are not assigned to the report associated with this suggestion.'], 403);
        }
        if ($report['status'] !== 'solving') {
            sendJsonResponse(['message' => 'Suggestions can only be updated when report is in "solving" status.'], 400);
        }

        // Set model properties for update
        $this->suggestionModel->suggestion_title = $data->suggestion_title;
        $this->suggestionModel->id = $id;

        if ($this->suggestionModel->update()) {
            sendJsonResponse(['message' => 'Suggestion updated successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to update suggestion.'], 500);
        }
    }

    /**
     * Deletes a suggestion.
     * Only the engineer assigned to the report can delete, or superadmin/admin.
     *
     * @param int $id The ID of the suggestion to delete.
     */
    public function delete($id) {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        // Fetch the suggestion to get its report_index
        $this->suggestionModel->id = $id;
        $suggestion = $this->suggestionModel->readOne();
        if (!$suggestion) {
            sendJsonResponse(['message' => 'Suggestion not found.'], 404);
        }

        // Authorize: Superadmin/Admin or the assigned engineer
        $authorized = false;
        if (in_array($currentUserRole, ['superadmin', 'admin'])) {
            $authorized = true;
        } else {
            $report = $this->reportModel->readOne($suggestion['report_index'], 'index');
            if ($report && $report['engineer_id'] == $currentUserId) {
                 $authorized = true;
            }
        }

        if (!$authorized) {
            sendJsonResponse(['message' => 'Unauthorized to delete this suggestion.'], 403);
        }

        if ($this->suggestionModel->delete()) {
            sendJsonResponse(['message' => 'Suggestion deleted successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to delete suggestion.'], 500);
        }
    }
}
?>
