<?php
// controllers/SupportTeamController.php

/**
 * SupportTeamController for Rehlko Customer Care application.
 * Manages CRUD operations for support team records.
 * Access is restricted for create, update, delete to 'superadmin', 'supervisor', 'admin'.
 * Read access is role-based and handled by the model.
 */
class SupportTeamController {
    private $conn;
    private $supportTeamModel;

    /**
     * Constructor for SupportTeamController.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
        $this->supportTeamModel = new SupportTeam($db); // Instantiate SupportTeam model
    }

    /**
     * Creates a new support team.
     * Only 'superadmin', 'supervisor', 'admin' roles are authorized.
     */
    public function create() {
        // Check authorization
        $currentUserRole = getCurrentUserRole();
        if (!in_array($currentUserRole, ['superadmin', 'supervisor', 'admin'])) {
            sendJsonResponse(['message' => 'Unauthorized. Only superadmin, supervisor, or admin can create support teams.'], 403);
            return;
        }

        $data = json_decode(file_get_contents("php://input"));

        // Validate required fields (customer_id is mandatory, others can be null depending on team structure)
        if (empty($data->customer_id)) {
            sendJsonResponse(['message' => 'Missing required field: customer_id.'], 400);
            return;
        }

        // Set support team properties
        $this->supportTeamModel->customer_id = $data->customer_id;
        $this->supportTeamModel->engineer_id = $data->engineer_id ?? null;
        $this->supportTeamModel->champion_id = $data->champion_id ?? null;
        $this->supportTeamModel->member_id = $data->member_id ?? null;

        if ($this->supportTeamModel->create()) {
            sendJsonResponse(['message' => 'Support team created successfully.', 'teamId' => $this->supportTeamModel->id], 201);
        } else {
            sendJsonResponse(['message' => 'Failed to create support team.'], 500);
        }
    }

    /**
     * Reads all support teams.
     * Access is restricted based on user role and inclusion in the team, handled by the model.
     */
    public function readAll() {
        $currentUserRole = getCurrentUserRole();
        if (!$currentUserRole) {
            sendJsonResponse(['message' => 'Authentication required.'], 401);
            return;
        }

        $result = $this->supportTeamModel->readAll(); // Authorization is handled within the model

        if ($result) {
            $teams_arr = [];
            while ($row = $result->fetch_assoc()) {
                $teams_arr[] = $row;
            }
            sendJsonResponse(['data'=>$teams_arr]);
        } else {
            sendJsonResponse(['message' => 'Failed to retrieve support teams or unauthorized.'], 403);
        }
    }

    /**
     * Reads a single support team by ID.
     * Access is restricted based on user role and inclusion in the team, handled by the model.
     *
     * @param int $id The ID of the support team to retrieve.
     */
    public function read($id) {
        $currentUserRole = getCurrentUserRole();
        if (!$currentUserRole) {
            sendJsonResponse(['message' => 'Authentication required.'], 401);
            return;
        }

        $this->supportTeamModel->id = $id;
        $team = $this->supportTeamModel->readOne(); // Authorization logic is inside readOne model method

        if ($team) {
            sendJsonResponse($team);
        } else {
            sendJsonResponse(['message' => 'Support team not found or unauthorized.'], 404);
        }
    }

    /**
     * Reads a single support team by customer ID.
     * Access is restricted to the customer themselves, or 'superadmin', 'supervisor', 'admin' roles.
     *
     * @param int $customerId The ID of the customer whose support team to retrieve.
     */
    public function readCustomerById($customerId) {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId(); // Assuming this function exists and returns the logged-in user's ID

        if (!$currentUserRole) {
            sendJsonResponse(['message' => 'Authentication required.'], 401);
            return;
        }

        // Fetch the support team by customer ID from the model
        $team = $this->supportTeamModel->readCustomerById($customerId);

        if (!$team) {
            sendJsonResponse(['message' => 'Support team not found for this customer.'], 404);
            return;
        }

        // Authorization check
        $authorized = false;
        if (in_array($currentUserRole, ['superadmin', 'supervisor', 'admin'])) {
            // Admins, supervisors, superadmins can view any customer's team
            $authorized = true;
        } elseif (in_array($currentUserRole, ['localcustomer', 'globalcustomer']) && $currentUserId == $customerId) {
            // Customers can view their own team
            $authorized = true;
        }

        if (!$authorized) {
            sendJsonResponse(['message' => 'Unauthorized to view this customer\'s support team.'], 403);
            return;
        }

        sendJsonResponse($team);
    }

    /**
     * Updates an existing support team.
     * Only 'superadmin', 'supervisor', 'admin' roles are authorized.
     *
     * @param int $id The ID of the support team to update.
     */
    public function update($id) {
        // Check authorization
        $currentUserRole = getCurrentUserRole();
        if (!in_array($currentUserRole, ['superadmin', 'supervisor', 'admin'])) {
            sendJsonResponse(['message' => 'Unauthorized. Only superadmin, supervisor, or admin can update support teams.'], 403);
            return;
        }

        $data = json_decode(file_get_contents("php://input"));

        // Validate required fields
        if (empty($data->customer_id)) { // Customer ID should generally not change for an existing team
            sendJsonResponse(['message' => 'Missing required field: customer_id.'], 400);
            return;
        }

        // Check if team exists
        $this->supportTeamModel->id = $id;
        if (!$this->supportTeamModel->readOne()) {
            sendJsonResponse(['message' => 'Support team not found.'], 404);
            return;
        }

        // Set properties for update
        $this->supportTeamModel->customer_id = $data->customer_id;
        $this->supportTeamModel->engineer_id = $data->engineer_id ?? null;
        $this->supportTeamModel->champion_id = $data->champion_id ?? null;
        $this->supportTeamModel->member_id = $data->member_id ?? null;
        $this->supportTeamModel->id = $id;

        if ($this->supportTeamModel->update()) {
            sendJsonResponse(['message' => 'Support team updated successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to update support team.'], 500);
        }
    }

    /**
     * Deletes a support team.
     * Only 'superadmin', 'supervisor', 'admin' roles are authorized.
     *
     * @param int $id The ID of the support team to delete.
     */
    public function delete($id) {
        // Check authorization
        $currentUserRole = getCurrentUserRole();
        if (!in_array($currentUserRole, ['superadmin', 'supervisor', 'admin'])) {
            sendJsonResponse(['message' => 'Unauthorized. Only superadmin or admin can delete support teams.'], 403);
            return;
        }

        $this->supportTeamModel->id = $id;

        if ($this->supportTeamModel->delete()) {
            sendJsonResponse(['message' => 'Support team deleted successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to delete support team or team not found.'], 500);
        }
    }
}
?>