<?php
// controllers/TermsController.php

/**
 * TermsController for Rehlko Customer Care application.
 * Manages CRUD operations for terms and policies content.
 * Access is restricted for create, update, delete to 'superadmin' and 'admin'.
 * Read access is public (or for all authenticated users).
 */
class TermsController {
    private $conn;
    private $termsModel;

    /**
     * Constructor for TermsController.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
        $this->termsModel = new Terms($db);
    }

    /**
     * Creates a new terms/policy record.
     * Only 'superadmin' and 'admin' roles are authorized.
     */
    public function create() {
        $currentUserRole = getCurrentUserRole();
        if (!in_array($currentUserRole, ['superadmin', 'admin'])) {
            sendJsonResponse(['message' => 'Unauthorized. Only superadmin or admin can create terms/policies.'], 403);
        }

        $data = json_decode(file_get_contents("php://input"));

        // Validate required fields
        if (empty($data->terms_category) || empty($data->terms_title) || empty($data->terms_content)) {
            sendJsonResponse(['message' => 'Missing required fields: terms_category, terms_title, terms_content.'], 400);
        }

        // Set model properties
        $this->termsModel->terms_category = $data->terms_category;
        $this->termsModel->terms_title = $data->terms_title;
        $this->termsModel->terms_content = $data->terms_content;

        if ($this->termsModel->create()) {
            sendJsonResponse(['message' => 'Terms/policy created successfully.', 'id' => $this->termsModel->id], 201);
        } else {
            sendJsonResponse(['message' => 'Failed to create terms/policy.'], 500);
        }
    }

    /**
     * Reads all terms/policy records.
     * Accessible by all authenticated users (or public if no auth check).
     */
    public function readAll() {
        // Decide if authentication is required for viewing terms.
        // For public-facing terms, no check is needed. For internal policies, maybe.
        // Assuming public/all authenticated can read.
        /*
        $currentUserRole = getCurrentUserRole();
        if (!$currentUserRole) {
            sendJsonResponse(['message' => 'Authentication required.'], 401);
        }
        */

        $result = $this->termsModel->readAll();

        if ($result) {
            $terms_arr = [];
            while ($row = $result->fetch_assoc()) {
                $terms_arr[] = $row;
            }
            sendJsonResponse(['data'=>$terms_arr]);
        } else {
            sendJsonResponse(['message' => 'Failed to retrieve terms/policies.'], 500);
        }
    }

    /**
     * Reads a single terms/policy record by ID.
     * Accessible by all authenticated users (or public if no auth check).
     *
     * @param int $id The ID of the terms/policy record to retrieve.
     */
    public function read($id) {
        // Same access consideration as readAll
        /*
        $currentUserRole = getCurrentUserRole();
        if (!$currentUserRole) {
            sendJsonResponse(['message' => 'Authentication required.'], 401);
        }
        */

        $this->termsModel->id = $id;
        $term = $this->termsModel->readOne();

        if ($term) {
            sendJsonResponse(['data'=>$term]);
        } else {
            sendJsonResponse(['message' => 'Terms/policy record not found.'], 404);
        }
    }

    /**
     * Updates an existing terms/policy record.
     * Only 'superadmin' and 'admin' roles are authorized.
     *
     * @param int $id The ID of the terms/policy record to update.
     */
    public function update($id) {
        $currentUserRole = getCurrentUserRole();
        if (!in_array($currentUserRole, ['superadmin', 'admin'])) {
            sendJsonResponse(['message' => 'Unauthorized. Only superadmin or admin can update terms/policies.'], 403);
        }

        $data = json_decode(file_get_contents("php://input"));

        // Validate required fields
        if (empty($data->terms_category) || empty($data->terms_title) || empty($data->terms_content)) {
            sendJsonResponse(['message' => 'Missing required fields.'], 400);
        }

        // Check if record exists
        $this->termsModel->id = $id;
        if (!$this->termsModel->readOne()) {
            sendJsonResponse(['message' => 'Terms/policy record not found.'], 404);
        }

        // Set model properties for update
        $this->termsModel->terms_category = $data->terms_category;
        $this->termsModel->terms_title = $data->terms_title;
        $this->termsModel->terms_content = $data->terms_content;
        $this->termsModel->id = $id;

        if ($this->termsModel->update()) {
            sendJsonResponse(['message' => 'Terms/policy updated successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to update terms/policy.'], 500);
        }
    }

    /**
     * Deletes a terms/policy record.
     * Only 'superadmin' and 'admin' roles are authorized.
     *
     * @param int $id The ID of the terms/policy record to delete.
     */
    public function delete($id) {
        $currentUserRole = getCurrentUserRole();
        if (!in_array($currentUserRole, ['superadmin', 'admin'])) {
            sendJsonResponse(['message' => 'Unauthorized. Only superadmin or admin can delete terms/policies.'], 403);
        }

        $this->termsModel->id = $id;

        if ($this->termsModel->delete()) {
            sendJsonResponse(['message' => 'Terms/policy deleted successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to delete terms/policy or not found.'], 500);
        }
    }
}
?>
