<?php
// controllers/UserController.php

/**
 * UserController for Rehlko Customer Care application.
 * Manages user-related operations, including CRUD by admins and profile updates by users themselves.
 */
class UserController {
    private $conn;
    private $userModel;
    private $mailer;
    private $notificationModel;

    /**
     * Constructor for UserController.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
        $this->userModel = new User($db);
        $this->mailer = new Mailer();
        $this->notificationModel = new Notification($db);
    }

    /**
     * Creates a new user.
     * Only 'superadmin' and 'admin' roles are authorized.
     */
    public function create() {
        // Check authorization
        $currentUserRole = getCurrentUserRole();
        if (!in_array($currentUserRole, ['superadmin', 'admin'])) {
            sendJsonResponse(['message' => 'Unauthorized. Only superadmin or admin can create users.'], 403);
        }

        $data = json_decode(file_get_contents("php://input"));

        // Validate required fields
        if (empty($data->username) || empty($data->email) || empty($data->password) || empty($data->role) || empty($data->country_id)) {
            sendJsonResponse(['message' => 'Missing required fields: username, email, password, role, country_id.'], 400);
        }

        // Check if email already exists
        if ($this->userModel->emailExists($data->email)) {
            $exist_email_user = $this->userModel->findByEmailorUsername($data->email);
            if($exist_email_user['is_active'] == 0){
                $id = $exist_email_user['id'];
                
                $this->userModel->id = $id; // Set ID for update
                $this->userModel->username = $data->username;
                $this->userModel->fullname = $data->fullname ?? $data->username;
                $this->userModel->email = $data->email;
                $this->userModel->password = $data->password; // Password will be hashed in the model's update
                $this->userModel->role = $data->role;
                $this->userModel->country_id = $data->country_id;
                $this->userModel->phone = $data->phone ?? null;
                $this->userModel->address = $data->address ?? null;
                
                if($this->userModel->reviveUser($id,true)){
                    $welcomeEmailBody = $this->mailer->getEmailTemplate('welcome_user', [
                'fullname' => $data->fullname,
                'username' => $data->username,
                'email' => $data->email,
                'temp_password' => $data->password, // Sending plain password here. For production, consider temporary password or reset link.
                'role' => $data->role,
                'login_link' => APP_BASE_URL . '/login' // Frontend login URL
            ]);
            $this->mailer->sendMail(
                $data->email,
                $data->fullname,
                'Welcome to Rehlko Customer Care',
                $welcomeEmailBody
            );

            // Create in-app notification for the new user
            $this->notificationModel->createNotification(
                $id, // Use the ID generated after creation
                'Welcome to Rehlko Customer Care!',
                'Your account has been successfully created. You can now log in.',
                'users'
            );

            sendJsonResponse(['message' => 'User created successfully.', 'userId' => $this->userModel->id], 201);
                }
            }else{
                sendJsonResponse(['message' => 'User with this email already exists.'], 409); 
            }
            // Conflict
        }

        // Set user properties from request data
        $this->userModel->username = $data->username;
        $this->userModel->fullname = $data->fullname ?? $data->username; // Fullname defaults to username
        $this->userModel->email = $data->email;
        $this->userModel->password = $data->password; // Password will be hashed in the model
        $this->userModel->role = $data->role;
        $this->userModel->country_id = $data->country_id;
        $this->userModel->phone = $data->phone ?? null;
        $this->userModel->address = $data->address ?? null;

        if ($this->userModel->create()) {
            // Send welcome email to the new user
            $welcomeEmailBody = $this->mailer->getEmailTemplate('welcome_user', [
                'fullname' => $this->userModel->fullname,
                'username' => $this->userModel->username,
                'email' => $this->userModel->email,
                'temp_password' => $data->password, // Sending plain password here. For production, consider temporary password or reset link.
                'role' => $this->userModel->role,
                'login_link' => APP_BASE_URL . '/login' // Frontend login URL
            ]);
            $this->mailer->sendMail(
                $this->userModel->email,
                $this->userModel->fullname,
                'Welcome to Rehlko Customer Care',
                $welcomeEmailBody
            );

            // Create in-app notification for the new user
            $this->notificationModel->createNotification(
                $this->userModel->id, // Use the ID generated after creation
                'Welcome to Rehlko Customer Care!',
                'Your account has been successfully created. You can now log in.',
                'users'
            );

            sendJsonResponse(['message' => 'User created successfully.', 'userId' => $this->userModel->id], 201);
        } else {
            sendJsonResponse(['message' => 'Failed to create user.'], 500);
        }
    }

    /**
     * Reads all users.
     * Access is restricted based on role (superadmin, admin can see all; others see limited sets).
     */
    public function readAll() {
        // Authorization is handled within the User model's readAll method for complexity
        $result = $this->userModel->readAll();

        if ($result) {
            $users_arr = [];
            while ($row = $result->fetch_assoc()) {
                // Exclude sensitive data like password hash from API response
                unset($row['password']);
                $users_arr[] = $row;
            }
            sendJsonResponse(['message' => 'Users fetched successfully.', 'data' => $users_arr], 200);
        } else {
            // If readAll returns false due to authorization in model or DB error
            sendJsonResponse(['message' => 'Failed to retrieve users or unauthorized.'], 403);
        }
    }

    /**
     * Reads a single user by ID.
     * Access is restricted based on role and relationship (self, support team).
     *
     * @param int $id The ID of the user to retrieve.
     */
    public function read($id) {
        $this->userModel->id = $id;
        $user = $this->userModel->readOne(); // Authorization logic is inside readOne model method

        if ($user) {
            unset($user['password']); // Don't expose password hash
            sendJsonResponse($user);
        } else {
            // readOne will send 403 response if unauthorized, or return false if not found.
            // If false, it means not found, as 403 is already handled by model.
            sendJsonResponse(['message' => 'User not found.'], 404);
        }
    }

    /**
     * Updates an existing user.
     * Only 'superadmin' and 'admin' roles can update other users.
     */
    // In your API endpoint file (e.g., api/user/update.php or similar)
// Assuming this code is within a method like 'update' in a UserController or similar class

    public function update($id) {
        // Check authorization
        $currentUserRole = getCurrentUserRole();
        if (!in_array($currentUserRole, ['superadmin', 'admin'])) {
            sendJsonResponse(['message' => 'Unauthorized. Only superadmin or admin can update users.'], 403);
        }

        $data = json_decode(file_get_contents("php://input"), true); // Add 'true' for associative array

        // Basic validation for essential fields that should ALWAYS be present
        // If these can also be optional, remove them from this initial check.
        // However, for update, typically username and email are required to identify what to update.
        if (!isset($data['username']) && !isset($data['email']) && !isset($data['fullname']) &&
        !isset($data['phone']) && !isset($data['address']) && !isset($data['country_id'])) {
            sendJsonResponse(['message' => 'No fields provided for update.'], 400);
        }


        // Check if user exists BEFORE attempting to set properties from $data
        $this->userModel->id = $id;
        // readOne now populates the object's properties as well
        $existingUserData = $this->userModel->readOne();
        if (!$existingUserData) {
            sendJsonResponse(['message' => 'User not found.'], 404);
        }

        // === Set properties for update, only if they are present in the request data ===
        // This is the key change here. Don't assign if not present in $data.

        if (isset($data['username'])) {
            $this->userModel->username = $data['username'];
        }
        if (isset($data['fullname'])) {
            $this->userModel->fullname = $data['fullname'];
        }
        if (isset($data['email'])) {
            // Only check email conflict if email is actually changed
            if ($data['email'] !== $existingUserData['email']) {
                if ($this->userModel->emailExists($data['email'], $id)) {
                    sendJsonResponse(['message' => 'Another user with this email already exists.'], 409); // Conflict
                }
            }
            $this->userModel->email = $data['email'];
        }
        if (isset($data['role'])) {
            $this->userModel->role = $data['role'];
        }
        // For optional fields, use isset. If they send "null", it will set to null in model.
        if (isset($data['phone'])) {
            $this->userModel->phone = $data['phone'];
        }
        if (isset($data['address'])) {
            $this->userModel->address = $data['address'];
        }
        // For country_id, also use isset. If client sends country_id: null, it will be null.
        // If client sends country_id: 123, it will be 123.
        // If client *doesn't send* country_id, it will NOT be set on the model, and thus NOT updated in DB.
        if (isset($data['country_id'])) {
            $this->userModel->country_id = $data['country_id'];
        }


        $this->userModel->id = $id; // Crucially, set the ID for the WHERE clause in the model's update method

        if ($this->userModel->update()) {
             error_log('User updated successfully. User Role: ' . $data['role'] . ', User Data : ' . $data['country_id']);
            sendJsonResponse(['message' => 'User updated successfully.'], 200);
        } else {
            // More granular error logging might be needed in model to differentiate
            // between "no fields to update" and "database error"
            sendJsonResponse(['message' => 'Failed to update user.'], 500);
        }
    }

    /**
     * Allows a user to update their own profile.
     * Authenticated user can update their own data.
     *
     * @param int $id The ID of the user whose profile is being updated.
     */
    public function updateProfile($id) {
    $currentUserId = getCurrentUserId();
        if ($currentUserId != $id) {
            sendJsonResponse(['message' => 'Unauthorized. You can only update your own profile.'], 403);
            return; // Terminate execution
        }

        $data = json_decode(file_get_contents("php://input"), true); // Returns associative array

        // Basic validation for profile fields: Check if any fields were provided at all
        if (empty($data)) {
            sendJsonResponse(['message' => 'No fields provided for update.'], 400);
            return; // Terminate execution
        }

        // Retrieve existing user data for comparison (e.g., email conflict, password verification)
        // Make sure readById is accessible and returns the full user data including password hash
        $existingUserData = $this->userModel->readById($id);
        if (!$existingUserData) {
            sendJsonResponse(['message' => 'User not found or an error occurred.'], 404);
            return; // Terminate execution
        }

        // Handle Password Update Separately (if new_password is provided)
        $passwordUpdated = false;
        if (isset($data['new_password']) && !empty($data['new_password'])) {
            if (isset($data['current_password']) && !empty($data['current_password'])) {
                // Verify current password
                if (verifyPassword($data['current_password'], $existingUserData['password'])) {
                    $newHashedPassword = hashPassword($data['new_password']);
                    if ($this->userModel->updatePassword($id, $newHashedPassword)) {
                        $passwordUpdated = true;
                        // Remove password fields from $data to prevent attempting to update them via general update
                        unset($data['current_password']);
                        unset($data['new_password']);
                    } else {
                        error_log("Failed to update password for user ID: $id");
                        sendJsonResponse(['message' => 'Failed to update password. Please try again.'], 500);
                        return; // Terminate execution
                    }
                } else {
                    sendJsonResponse(['message' => 'Current password incorrect.'], 401);
                    return; // Terminate execution
                }
            } else {
                sendJsonResponse(['message' => 'Current password is required to change password.'], 400);
                return; // Terminate execution
            }
        }

        // Prepare User Model for General Profile Update (non-password fields)
        $profileFieldsUpdated = false; // Flag to track if any non-password profile fields are present

        // Check for email conflict if email is changed
        if (isset($data['email'])) {
            if ($data['email'] !== $existingUserData['email']) {
                if ($this->userModel->emailExists($data['email'], $id)) {
                    sendJsonResponse(['message' => 'This email is already used by another account.'], 409); // Conflict
                    return; // Terminate execution
                }
            }
            $this->userModel->email = $data['email'];
            $profileFieldsUpdated = true;
        }

        // Check for username conflict if username is changed
        if (isset($data['username'])) {
            if ($data['username'] !== $existingUserData['username']) {
                // Assuming you have a usernameExists method, similar to emailExists
                if ($this->userModel->usernameExists($data['username'], $id)) {
                    sendJsonResponse(['message' => 'This username is already taken.'], 409); // Conflict
                    return; // Terminate execution
                }
            }
            $this->userModel->username = $data['username'];
            $profileFieldsUpdated = true;
        }

        // Assign other fields only if they exist in the request data
        if (isset($data['fullname'])) {
            $this->userModel->fullname = $data['fullname'];
            $profileFieldsUpdated = true;
        }
        if (isset($data['phone'])) {
            // Allow null for phone if client sends "phone": null
            $this->userModel->phone = $data['phone'];
            $profileFieldsUpdated = true;
        }
        if (isset($data['address'])) {
            // Allow null for address if client sends "address": null
            $this->userModel->address = $data['address'];
            $profileFieldsUpdated = true;
        }
        if (isset($data['country_id'])) {
            $countryId = $data['country_id'];

            // If country_id is null, check if the column allows null
            if ($countryId === null) {
                // Option 1: If 'country_id' in 'users' table is NULLABLE:
                $this->userModel->country_id = null;
                $profileFieldsUpdated = true;
            }
            // Option 2: If 'country_id' in 'users' table is NOT NULLABLE (and client sends null):
            // You should sendJsonResponse(['message' => 'Country ID cannot be null.'], 400); and return;
            // Or just let the foreign key constraint fail (less graceful)

            // If country_id is not null, validate it
            else {
            // Ensure it's an integer and not empty string/zero if not allowed
                if (!is_numeric($countryId) || $countryId <= 0) {
                    sendJsonResponse(['message' => 'Invalid Country ID format.'], 400);
                    return;
                }
                $countryId = (int)$countryId; // Cast to integer for safety

                // Check if the country exists in the country table
                // You'll need a method in your User model or a separate Country model for this.
                // Let's assume User model has countryExists($countryId)
                if (!$this->userModel->countryExists($countryId)) {
                    sendJsonResponse(['message' => 'Provided Country ID does not exist.'], 400);
                    return;
                }

                $this->userModel->country_id = $countryId;
                $profileFieldsUpdated = true;
            }
        }
       /** if (isset($data['country_id'])) {
            // ... validation as in point A ...
            // BUT DO NOT ALLOW $data['country_id'] === null here.
            if ($data['country_id'] === null) {
                sendJsonResponse(['message' => 'Country ID cannot be null. A country must be specified.'], 400);
                return;
            }
            // ... (rest of validation) ...
            $this->userModel->country_id = $data['country_id'];
            $profileFieldsUpdated = true;
        } else {
            // If it's mandatory and not provided in the request
            sendJsonResponse(['message' => 'Country ID is a required field.'], 400);
            return;
        }*/

        // Set the ID for the model's update method
        $this->userModel->id = $id;

        $profileGeneralUpdateSuccessful = false;
        if ($profileFieldsUpdated) { // Only call update if there are non-password fields to update
            if ($this->userModel->updateProfile()) {
                $profileGeneralUpdateSuccessful = true;
            } else {
                error_log("Failed to update user profile (non-password fields) for ID: $id");
                sendJsonResponse(['message' => 'Failed to update profile. Please try again.'], 500);
                return; // Terminate execution
            }
        }

        // Final Response based on what was updated
        if ($profileGeneralUpdateSuccessful || $passwordUpdated) {
            sendJsonResponse(['message' => 'Profile updated successfully.'], 200);
        } else {
            // If neither profile fields nor password was updated (e.g., only invalid fields sent, or no changes)
            sendJsonResponse(['message' => 'No valid fields provided for profile update or no changes detected.'], 200); // 200 OK if nothing changed but request was valid
        }
    }   


    /**
     * Deletes a user.
     * Only 'superadmin' and 'admin' roles are authorized.
     *
     * @param int $id The ID of the user to delete.
     */
    public function delete($id) {
        // Check authorization
        $currentUserRole = getCurrentUserRole();
        if (!in_array($currentUserRole, ['superadmin', 'admin'])) {
            sendJsonResponse(['message' => 'Unauthorized. Only superadmin or admin can delete users.'], 403);
        }

        // Prevent a user from deleting themselves (optional but good practice)
        $currentUserId = getCurrentUserId();
        if ($currentUserId == $id) {
            sendJsonResponse(['message' => 'You cannot delete your own account.'], 403);
        }

        $this->userModel->id = $id;

        if ($this->userModel->delete()) {
            sendJsonResponse(['message' => 'User deleted successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to delete user or user not found.'], 500);
        }
    }
}
?>
