<?php
// cron/maintenance_reminders.php

/**
 * Cron Job: Maintenance Reminders
 * This script is designed to be run daily by a server cron scheduler.
 * It checks for upcoming maintenance records and sends reminders to engineers.
 * It also notifies product owners (customers) when maintenance is updated.
 */

// Define the root directory if not already defined
if (!defined('ROOT_DIR')) {
    define('ROOT_DIR', realpath(__DIR__ . '/..'));
}

// Include necessary configuration and utility files
require_once ROOT_DIR . '/config/database.php';
require_once ROOT_DIR . '/config/mail.php'; // For APP_BASE_URL
require_once ROOT_DIR . '/utils/helpers.php'; // For email templating functions
require_once ROOT_DIR . '/utils/Mailer.php';

// Include all required models
require_once ROOT_DIR . '/models/Maintenance.php';
require_once ROOT_DIR . '/models/User.php';
require_once ROOT_DIR . '/models/Product.php'; // Needed for product details
require_once ROOT_DIR . '/models/CustomerProduct.php'; // Needed to link product to customer
require_once ROOT_DIR . '/models/Notification.php';

$maintenanceModel = new Maintenance($conn);
$userModel = new User($conn);
$productModel = new Product($conn);
$customerProductModel = new CustomerProduct($conn);
$notificationModel = new Notification($conn);
$mailer = new Mailer();

echo "Starting Maintenance Reminders Cron Job...\n";

// --- Step 1: Send Reminders for Upcoming Maintenance to Engineers ---
// Fetch maintenances due in the next 7 days (adjust as needed)
$upcomingMaintenances = $maintenanceModel->getUpcomingMaintenances(7);

$engineers = $userModel->getUsersByRole('engineer'); // Get all engineers to notify

foreach ($upcomingMaintenances as $maintenance) {
    // Check if a reminder for this maintenance has already been sent today
    // This requires a `last_reminder_sent_date` column in the `maintenance` table,
    // or similar logic using the `report_reminders` table (though its schema is for reports).
    // For simplicity, I'll assume `markMaintenanceReminderSent` updates a conceptual flag for now,
    // or you add a column to `maintenance` table like `last_engineer_reminder_sent_at`.

    // Prevent duplicate reminders for the same event in short period
    // If you add `last_engineer_reminder_sent_at` to `maintenance` table:
    // $lastSent = strtotime($maintenance['last_engineer_reminder_sent_at'] ?? '1970-01-01');
    // if (time() - $lastSent < (24 * 60 * 60)) { // If sent within last 24 hours
    //     continue;
    // }

    echo "Processing upcoming maintenance for " . $maintenance['generator_serial_number'] . "\n";

    // Get customer details for the product owner
    $customerProduct = $customerProductModel->getByGeneratorSerialNumber($maintenance['generator_serial_number']);
    $customer = null;
    if ($customerProduct) {
        $customer = $userModel->readById($customerProduct['customer_id']);
    }

    // Notify all engineers
    foreach ($engineers as $engineer) {
        // Create in-app notification for engineer
        $this->notificationModel->createNotification(
            $engineer['id'],
            'Upcoming Maintenance',
            'Maintenance for generator ' . $maintenance['generator_serial_number'] . ' is due (' . $maintenance['maintenance_type'] . ') on ' . date('Y-m-d', strtotime($maintenance['maintence_date'])) . '.',
            'maintenance'
        );

        // Send email to engineer
        $emailBody = $mailer->getEmailTemplate('engineer_maintenance_reminder', [
            'engineer_name' => $engineer['fullname'],
            'generator_serial_number' => $maintenance['generator_serial_number'],
            'maintenance_title' => $maintenance['maintenance_title'],
            'maintenance_type' => $maintenance['maintenance_type'],
            'due_date' => date('Y-m-d H:i', strtotime($maintenance['maintence_date'])),
            'customer_name' => $customer ? $customer['fullname'] : 'N/A',
            'app_link' => APP_BASE_URL . '/app/maintenance/' . $maintenance['id'] // Link to maintenance record
        ]);
        $mailer->sendMail(
            $engineer['email'],
            $engineer['fullname'],
            'Reminder: Upcoming Generator Maintenance',
            $emailBody
        );
    }
    // Mark that a reminder has been sent for this maintenance to engineers
    $maintenanceModel->markMaintenanceReminderSent($maintenance['id']);
}

// --- Step 2: Notify Customer After Maintenance Update ---
// Fetch maintenances that were recently updated and customer hasn't been notified yet.
// Assumes `modified_time` is updated on record changes and `last_customer_notified_at` track.
$recentlyUpdatedMaintenances = $maintenanceModel->getRecentlyUpdatedMaintenances(60); // Check for updates in last 60 minutes

foreach ($recentlyUpdatedMaintenances as $maintenance) {
    // Prevent duplicate customer notifications for the same update
    // If you add `last_customer_notified_at` to `maintenance` table:
    // $lastNotified = strtotime($maintenance['last_customer_notified_at'] ?? '1970-01-01');
    // if (strtotime($maintenance['modified_time']) <= $lastNotified) {
    //     continue; // Already notified for this update
    // }

    echo "Processing recently updated maintenance for customer notification: " . $maintenance['generator_serial_number'] . "\n";

    $customerProduct = $customerProductModel->getByGeneratorSerialNumber($maintenance['generator_serial_number']);
    if ($customerProduct) {
        $customer = $userModel->readById($customerProduct['customer_id']);
        if ($customer) {
            // Create in-app notification for customer
            $this->notificationModel->createNotification(
                $customer['id'],
                'Maintenance Update',
                'Maintenance for your generator ' . $maintenance['generator_serial_number'] . ' has been updated.',
                'maintenance'
            );

            // Send email to customer
            $emailBody = $mailer->getEmailTemplate('customer_maintenance_update', [
                'customer_name' => $customer['fullname'],
                'generator_serial_number' => $maintenance['generator_serial_number'],
                'maintenance_title' => $maintenance['maintenance_title'],
                'maintenance_date' => date('Y-m-d H:i', strtotime($maintenance['maintence_date'])),
                'app_link' => APP_BASE_URL . '/app/maintenance/' . $maintenance['id']
            ]);
            $mailer->sendMail(
                $customer['email'],
                $customer['fullname'],
                'Maintenance Update for Your Generator',
                $emailBody
            );
            // Mark that customer has been notified for this update
            $maintenanceModel->markCustomerNotified($maintenance['id']);
        }
    }
}

echo "Maintenance Reminders Cron Job Finished.\n";

$conn->close(); // Close DB connection
?>
