<?php
// models/Country.php

/**
 * Country Model for Rehlko Customer Care application.
 * Manages CRUD operations and data interaction for the 'country' table.
 */
class Country {
    private $conn;
    private $table_name = "country";

    // Object properties
    public $id;
    public $country_name;
    public $country_name_code;
    public $country_phone_code;
    public $created_time;
    public $modified_time;

    /**
     * Constructor for the Country model.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
    }

    /**
     * Creates a new country record.
     *
     * @return bool True on success, false on failure.
     */
    public function create() {
        $query = "INSERT INTO " . $this->table_name . "
                  SET country_name=?, country_name_code=?, country_phone_code=?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize inputs
        $this->country_name = htmlspecialchars(strip_tags($this->country_name));
        $this->country_name_code = htmlspecialchars(strip_tags($this->country_name_code));
        $this->country_phone_code = htmlspecialchars(strip_tags($this->country_phone_code));

        $stmt->bind_param("sss", $this->country_name, $this->country_name_code, $this->country_phone_code);

        if ($stmt->execute()) {
            $this->id = $this->conn->insert_id;
            return true;
        }
        error_log("Country creation failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Reads all country records.
     *
     * @return mysqli_result|false The result set on success, false on failure.
     */
    public function readAll() {
        $query = "SELECT id, country_name, country_name_code, country_phone_code, created_time, modified_time
                  FROM " . $this->table_name . "
                  ORDER BY country_name ASC";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        if ($stmt->execute()) {
            return $stmt->get_result();
        }
        error_log("Read all countries failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Reads a single country record by ID.
     *
     * @return array|false The country data as an associative array on success, false if not found.
     */
    public function readOne() {
        $query = "SELECT id, country_name, country_name_code, country_phone_code, created_time, modified_time
                  FROM " . $this->table_name . "
                  WHERE id = ? LIMIT 0,1";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $this->id = htmlspecialchars(strip_tags($this->id));
        $stmt->bind_param("i", $this->id);

        if ($stmt->execute()) {
            $result = $stmt->get_result();
            return $result->fetch_assoc();
        }
        error_log("Read one country failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Updates an existing country record.
     *
     * @return bool True on success, false on failure.
     */
    public function update() {
        $query = "UPDATE " . $this->table_name . "
                  SET country_name=?, country_name_code=?, country_phone_code=?
                  WHERE id=?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize inputs
        $this->country_name = htmlspecialchars(strip_tags($this->country_name));
        $this->country_name_code = htmlspecialchars(strip_tags($this->country_name_code));
        $this->country_phone_code = htmlspecialchars(strip_tags($this->country_phone_code));
        $this->id = htmlspecialchars(strip_tags($this->id));

        $stmt->bind_param("sssi", $this->country_name, $this->country_name_code, $this->country_phone_code, $this->id);

        if ($stmt->execute()) {
            return true;
        }
        error_log("Country update failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Deletes a country record by ID.
     *
     * @return bool True on success, false on failure.
     */
    public function delete() {
        $query = "DELETE FROM " . $this->table_name . " WHERE id = ?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $this->id = htmlspecialchars(strip_tags($this->id));
        $stmt->bind_param("i", $this->id);

        if ($stmt->execute()) {
            return true;
        }
        error_log("Country deletion failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }
}
?>
