<?php
// models/EffectiveAction.php

/**
 * EffectiveAction Model for Rehlko Customer Care application.
 * Manages CRUD operations and data interaction for the 'effective_action' table.
 */
class EffectiveAction {
    private $conn;
    private $table_name = "effective_action";

    // Object properties
    public $id;
    public $report_index;
    public $effective_action_title;
    public $created_user;
    public $created_datetime;
    public $modified_datetime;

    /**
     * Constructor for the EffectiveAction model.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
    }

    /**
     * Creates a new effective action record.
     *
     * @return bool True on success, false on failure.
     */
    public function create() {
        $query = "INSERT INTO " . $this->table_name . "
                  SET report_index=?, effective_action_title=?, created_user=?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize inputs
        $this->report_index = htmlspecialchars(strip_tags($this->report_index));
        $this->effective_action_title = htmlspecialchars(strip_tags($this->effective_action_title));
        $this->created_user = htmlspecialchars(strip_tags($this->created_user));

        $stmt->bind_param("ssi", $this->report_index, $this->effective_action_title, $this->created_user);

        if ($stmt->execute()) {
            $this->id = $this->conn->insert_id;
            return true;
        }
        error_log("Effective action creation failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Reads effective action records for a specific report index.
     *
     * @param string $reportIndex The report index to filter by.
     * @return array An array of effective action records.
     */
    public function readByReportIndex($reportIndex) {
        $query = "SELECT id, report_index, effective_action_title, created_user, created_datetime, modified_datetime
                  FROM " . $this->table_name . "
                  WHERE report_index = ?
                  ORDER BY created_datetime ASC";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return [];
        }

        $reportIndex = htmlspecialchars(strip_tags($reportIndex));
        $stmt->bind_param("s", $reportIndex);

        if ($stmt->execute()) {
            $result = $stmt->get_result();
            $effectiveActions = [];
            while ($row = $result->fetch_assoc()) {
                $effectiveActions[] = $row;
            }
            return $effectiveActions;
        }
        error_log("Read effective actions by report index failed: (" . $stmt->errno . ") " . $stmt->error);
        return [];
    }

    /**
     * Reads a single effective action record by ID.
     *
     * @return array|false The effective action data as an associative array on success, false if not found.
     */
    public function readOne() {
        $query = "SELECT id, report_index, effective_action_title, created_user, created_datetime, modified_datetime
                  FROM " . $this->table_name . "
                  WHERE id = ? LIMIT 0,1";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $this->id = htmlspecialchars(strip_tags($this->id));
        $stmt->bind_param("i", $this->id);

        if ($stmt->execute()) {
            $result = $stmt->get_result();
            return $result->fetch_assoc();
        }
        error_log("Read one effective action failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Updates an existing effective action record.
     *
     * @return bool True on success, false on failure.
     */
    public function update() {
        $query = "UPDATE " . $this->table_name . "
                  SET effective_action_title=?
                  WHERE id=?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize inputs
        $this->effective_action_title = htmlspecialchars(strip_tags($this->effective_action_title));
        $this->id = htmlspecialchars(strip_tags($this->id));

        $stmt->bind_param("si", $this->effective_action_title, $this->id);

        if ($stmt->execute()) {
            return true;
        }
        error_log("Effective action update failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Deletes an effective action record by ID.
     *
     * @return bool True on success, false on failure.
     */
    public function delete() {
        $query = "DELETE FROM " . $this->table_name . " WHERE id = ?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $this->id = htmlspecialchars(strip_tags($this->id));
        $stmt->bind_param("i", $this->id);

        if ($stmt->execute()) {
            return true;
        }
        error_log("Effective action deletion failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }
}
?>
