<?php
// models/Maintenance.php

/**
 * Maintenance Model for Rehlko Customer Care application.
 * Manages CRUD operations and data interaction for the 'maintenance' table.
 */
class Maintenance {
    private $conn;
    private $table_name = "maintenance";

    // Object properties
    public $id;
    public $generator_serial_number;
    public $maintenance_title;
    public $maintenance_description;
    public $maintenance_type;
    public $maintenance_check_user; // User who checked/performed the maintenance
    public $maintence_date; // Note: Typo 'maintence' from SQL schema
    public $is_reminder_sent; // Custom property to track reminder status (not in DB, managed internally)
    public $is_customer_notified; // Custom property to track customer notification (not in DB, managed internally)

    /**
     * Constructor for the Maintenance model.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
    }

    /**
     * Creates a new maintenance record.
     * This is primarily used by CustomerProduct model for auto-creation.
     *
     * @return bool True on success, false on failure.
     */
    public function create() {
        $query = "INSERT INTO " . $this->table_name . "
                  SET generator_serial_number=?, maintenance_title=?,
                      maintenance_description=?, maintenance_type=?,
                      maintenance_check_user=?, maintence_date=?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize inputs
        $this->generator_serial_number = htmlspecialchars(strip_tags($this->generator_serial_number));
        $this->maintenance_title = htmlspecialchars(strip_tags($this->maintenance_title));
        $this->maintenance_description = htmlspecialchars(strip_tags($this->maintenance_description));
        $this->maintenance_type = htmlspecialchars(strip_tags($this->maintenance_type));
        $this->maintenance_check_user = htmlspecialchars(strip_tags($this->maintenance_check_user));
        $this->maintence_date = htmlspecialchars(strip_tags($this->maintence_date));

        $stmt->bind_param("ssssis",
            $this->generator_serial_number,
            $this->maintenance_title,
            $this->maintenance_description,
            $this->maintenance_type,
            $this->maintenance_check_user,
            $this->maintence_date
        );

        if ($stmt->execute()) {
            $this->id = $this->conn->insert_id;
            return true;
        }
        error_log("Maintenance creation failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Reads all maintenance records with role-based filtering.
     *
     * @return mysqli_result|false The result set on success, false on failure.
     */
    public function readAll() {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        $query = "SELECT m.id, m.generator_serial_number, m.maintenance_title, m.maintenance_description,
                         m.maintenance_type, m.maintenance_check_user, m.maintence_date,
                         u.fullname AS check_user_name, cp.customer_id, c.fullname AS customer_name
                  FROM " . $this->table_name . " m
                  LEFT JOIN users u ON m.maintenance_check_user = u.id
                  LEFT JOIN customer_products cp ON m.generator_serial_number = cp.generator_serial_number
                  LEFT JOIN users c ON cp.customer_id = c.id";

        $whereClauses = [];
        $bindParams = [];
        $bindParamTypes = '';

        if (in_array($currentUserRole, ['superadmin', 'admin', 'supervisor', 'engineer', 'champion', 'member'])) {
            // These roles can read all maintenances.
            // For engineers, champions, members, it's implied they can read for their supported customers.
            // Let's refine this: engineers/champions/members should only see maintenance for their support team customers.
            if (in_array($currentUserRole, ['engineer', 'champion', 'member'])) {
                $query .= " JOIN support_team st ON cp.customer_id = st.customer_id";
                $whereClauses[] = "(st.engineer_id = ? OR st.champion_id = ? OR st.member_id = ?)";
                $bindParams[] = $currentUserId;
                $bindParams[] = $currentUserId;
                $bindParams[] = $currentUserId;
                $bindParamTypes .= 'iii';
            }
        } elseif (in_array($currentUserRole, ['localcustomer', 'globalcustomer'])) {
            // Customers can only read maintenances for their own products
            $whereClauses[] = "cp.customer_id = ?";
            $bindParams[] = $currentUserId;
            $bindParamTypes .= 'i';
        } else {
            sendJsonResponse(['message' => 'Unauthorized to read maintenances.'], 403);
        }

        if (!empty($whereClauses)) {
            $query .= " WHERE " . implode(" AND ", $whereClauses);
        }

        $query .= " ORDER BY m.maintence_date DESC";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        if (!empty($bindParams)) {
            $stmt->bind_param($bindParamTypes, ...$bindParams);
        }

        if ($stmt->execute()) {
            return $stmt->get_result();
        }
        error_log("Read all maintenances failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Reads a single maintenance record by ID.
     *
     * @return array|false The maintenance data as an associative array on success, false if not found.
     */
    public function readOne() {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        $query = "SELECT m.id, m.generator_serial_number, m.maintenance_title, m.maintenance_description,
                         m.maintenance_type, m.maintenance_check_user, m.maintence_date,
                         u.fullname AS check_user_name, cp.customer_id, c.fullname AS customer_name
                  FROM " . $this->table_name . " m
                  LEFT JOIN users u ON m.maintenance_check_user = u.id
                  LEFT JOIN customer_products cp ON m.generator_serial_number = cp.generator_serial_number
                  LEFT JOIN users c ON cp.customer_id = c.id
                  WHERE m.id = ? LIMIT 0,1";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $this->id = htmlspecialchars(strip_tags($this->id));
        $stmt->bind_param("i", $this->id);

        if ($stmt->execute()) {
            $result = $stmt->get_result();
            $maintenance = $result->fetch_assoc();

            if ($maintenance) {
                // Authorization check for single record
                if (in_array($currentUserRole, ['superadmin', 'admin', 'supervisor', 'engineer', 'champion', 'member'])) {
                    // Engineers, champions, members still need to check if customer is in their team
                    if (in_array($currentUserRole, ['engineer', 'champion', 'member'])) {
                         $supportTeamModel = new SupportTeam($this->conn);
                         $team = $supportTeamModel->getByCustomerId($maintenance['customer_id']);
                         if ($team && ($team['engineer_id'] == $currentUserId || $team['champion_id'] == $currentUserId || $team['member_id'] == $currentUserId)) {
                             return $maintenance;
                         } else {
                             sendJsonResponse(['message' => 'Unauthorized to view this maintenance record.'], 403);
                             return false;
                         }
                    }
                    return $maintenance;
                } elseif (in_array($currentUserRole, ['localcustomer', 'globalcustomer'])) {
                    // Customers can only view their own product's maintenance
                    if ($maintenance['customer_id'] == $currentUserId) {
                        return $maintenance;
                    } else {
                        sendJsonResponse(['message' => 'Unauthorized to view this maintenance record.'], 403);
                        return false;
                    }
                } else {
                    sendJsonResponse(['message' => 'Unauthorized to view this maintenance record.'], 403);
                    return false;
                }
            }
        }
        error_log("Read one maintenance failed or not found: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Retrieves upcoming maintenance records for reminder cron job.
     * Considers `maintence_date` and current date.
     * Adds `is_reminder_sent` and `is_customer_notified` flags for internal tracking.
     *
     * @param int $daysAdvance Number of days in advance to consider for upcoming maintenance.
     * @return array An array of upcoming maintenance records.
     */
    public function getUpcomingMaintenances($daysAdvance = 7) {
        $futureDate = (new DateTime())->modify("+$daysAdvance days")->format('Y-m-d H:i:s');
        $currentDate = (new DateTime())->format('Y-m-d H:i:s');

        // Note: The SQL table for maintenance does not have reminder flags.
        // For 'without composer' setup, we'll store and check these in the 'setting' table or in-memory
        // for the cron job, or ideally add new columns to the 'maintenance' table.
        // For now, let's assume `is_reminder_sent` and `is_customer_notified`
        // are conceptual or handled by the cron's own logic (e.g., checking last run time).
        // For a robust solution, consider adding columns like `last_reminder_sent_date`, `last_customer_notified_date` to `maintenance` table.

        $query = "SELECT m.id, m.generator_serial_number, m.maintenance_title,
                         m.maintenance_type, m.maintence_date, cp.customer_id
                  FROM " . $this->table_name . " m
                  JOIN customer_products cp ON m.generator_serial_number = cp.generator_serial_number
                  WHERE m.maintence_date BETWEEN ? AND ?
                  ORDER BY m.maintence_date ASC";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed for upcoming maintenances: (" . $this->conn->errno . ") " . $this->conn->error);
            return [];
        }

        $stmt->bind_param("ss", $currentDate, $futureDate);

        if ($stmt->execute()) {
            $result = $stmt->get_result();
            $maintenances = [];
            while ($row = $result->fetch_assoc()) {
                // Add conceptual flags for cron job
                $row['is_reminder_sent'] = false; // Need a mechanism to store this persistently
                $row['is_customer_notified'] = false; // Need a mechanism to store this persistently
                $maintenances[] = $row;
            }
            return $maintenances;
        }
        error_log("Get upcoming maintenances failed: (" . $stmt->errno . ") " . $stmt->error);
        return [];
    }

    /**
     * Retrieves recently updated maintenance records for customer notification.
     * Assumes a `modified_datetime` column (which exists) and fetches records updated recently.
     *
     * @param int $minutesAgo How many minutes ago to look for updates.
     * @return array An array of recently updated maintenance records.
     */
    public function getRecentlyUpdatedMaintenances($minutesAgo = 30) {
        $timeThreshold = (new DateTime())->modify("-$minutesAgo minutes")->format('Y-m-d H:i:s');

        // Note: Assumes the maintenance table has a `modified_datetime` column
        // from `current_timestamp() ON UPDATE current_timestamp()`.
        // Also assumes a mechanism to prevent re-notifying the customer for the same update.
        // This could be a flag `customer_notified_at` in the `maintenance` table,
        // or a check against the `notifications` table.

        $query = "SELECT m.id, m.generator_serial_number, m.maintenance_title,
                         m.maintenance_type, m.maintence_date, m.modified_time, cp.customer_id
                  FROM " . $this->table_name . " m
                  JOIN customer_products cp ON m.generator_serial_number = cp.generator_serial_number
                  WHERE m.modified_time >= ?
                  ORDER BY m.modified_time DESC";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed for recently updated maintenances: (" . $this->conn->errno . ") " . $this->conn->error);
            return [];
        }

        $stmt->bind_param("s", $timeThreshold);

        if ($stmt->execute()) {
            $result = $stmt->get_result();
            $maintenances = [];
            while ($row = $result->fetch_assoc()) {
                $maintenances[] = $row;
            }
            return $maintenances;
        }
        error_log("Get recently updated maintenances failed: (" . $stmt->errno . ") " . $stmt->error);
        return [];
    }


    /**
     * Updates an existing maintenance record.
     *
     * @return bool True on success, false on failure.
     */
    public function update() {
        $query = "UPDATE " . $this->table_name . "
                  SET generator_serial_number=?, maintenance_title=?,
                      maintenance_description=?, maintenance_type=?,
                      maintenance_check_user=?, maintence_date=?
                  WHERE id=?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize inputs
        $this->generator_serial_number = htmlspecialchars(strip_tags($this->generator_serial_number));
        $this->maintenance_title = htmlspecialchars(strip_tags($this->maintenance_title));
        $this->maintenance_description = htmlspecialchars(strip_tags($this->maintenance_description));
        $this->maintenance_type = htmlspecialchars(strip_tags($this->maintenance_type));
        $this->maintenance_check_user = htmlspecialchars(strip_tags($this->maintenance_check_user));
        $this->maintence_date = htmlspecialchars(strip_tags($this->maintence_date));
        $this->id = htmlspecialchars(strip_tags($this->id));

        $stmt->bind_param("ssssisi",
            $this->generator_serial_number,
            $this->maintenance_title,
            $this->maintenance_description,
            $this->maintenance_type,
            $this->maintenance_check_user,
            $this->maintence_date,
            $this->id
        );

        if ($stmt->execute()) {
            return true;
        }
        error_log("Maintenance update failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Deletes a maintenance record by ID.
     *
     * @return bool True on success, false on failure.
     */
    public function delete() {
        $query = "DELETE FROM " . $this->table_name . " WHERE id = ?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $this->id = htmlspecialchars(strip_tags($this->id));
        $stmt->bind_param("i", $this->id);

        if ($stmt->execute()) {
            return true;
        }
        error_log("Maintenance deletion failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Marks a maintenance record as having a reminder sent.
     * This would typically involve updating a flag/timestamp in the database.
     * For now, this is a conceptual placeholder as the schema doesn't have such columns.
     *
     * @param int $maintenanceId The ID of the maintenance record.
     * @return bool True on success, false otherwise.
     */
    public function markMaintenanceReminderSent($maintenanceId) {
        // Implement database update (e.g., UPDATE maintenance SET last_reminder_sent = NOW() WHERE id = ?)
        // For demonstration, just log. You should add a column in `maintenance` table for this.
        error_log("Conceptual: Marked maintenance ID $maintenanceId as reminder sent.");
        return true;
    }

    /**
     * Marks a maintenance record as having the customer notified.
     * This would typically involve updating a flag/timestamp in the database.
     * For now, this is a conceptual placeholder.
     *
     * @param int $maintenanceId The ID of the maintenance record.
     * @return bool True on success, false otherwise.
     */
    public function markCustomerNotified($maintenanceId) {
        // Implement database update (e.g., UPDATE maintenance SET last_customer_notified = NOW() WHERE id = ?)
        // For demonstration, just log. You should add a column in `maintenance` table for this.
        error_log("Conceptual: Marked maintenance ID $maintenanceId as customer notified.");
        return true;
    }
}
?>
