<?php
// models/Report.php

/**
 * Report Model for Rehlko Customer Care application.
 * Manages CRUD operations and data interaction for the 'reports' table.
 */
class Report {
    private $conn;
    private $table_name = "reports";

    // Object properties
    public $id;
    public $report_index;
    public $report_type;
    public $generator_serial_number;
    public $customer_id;
    public $problem_issue;
    public $running_hours;
    public $load_test;
    public $load_hour;
    public $load_amount;
    public $used_for;
    public $error_code;
    public $test_run;
    public $location;
    public $layer;
    public $remarks;
    public $status;
    public $is_pick_up;
    public $engineer_id;
    public $supervisor_id;
    public $created_datetime;
    public $modified_datetime;

    /**
     * Constructor for the Report model.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
    }

    /**
     * Creates a new report record.
     *
     * @return bool True on success, false on failure.
     */
    public function create() {
        $query = "INSERT INTO " . $this->table_name . "
                  SET report_index=?, report_type=?, generator_serial_number=?, customer_id=?,
                      problem_issue=?, running_hours=?, load_test=?, load_hour=?, load_amount=?,
                      used_for=?, error_code=?, test_run=?, location=?, layer=?, remarks=?,
                      status='pending', is_pick_up=0"; // Default status and pick_up status

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Generate unique report index
        // Assuming generateUniqueIndex() is a globally available helper function
        $this->report_index = generateUniqueIndex('RPT-');

        // Sanitize inputs
        $this->report_type = htmlspecialchars(strip_tags($this->report_type));
        $this->generator_serial_number = htmlspecialchars(strip_tags($this->generator_serial_number));
        $this->customer_id = htmlspecialchars(strip_tags($this->customer_id));
        $this->problem_issue = htmlspecialchars(strip_tags($this->problem_issue));
        $this->running_hours = htmlspecialchars(strip_tags($this->running_hours));
        $this->load_test = (int)$this->load_test; // Cast to int for tinyint
        $this->load_hour = htmlspecialchars(strip_tags($this->load_hour));
        $this->load_amount = htmlspecialchars(strip_tags($this->load_amount));
        $this->used_for = htmlspecialchars(strip_tags($this->used_for));
        $this->error_code = htmlspecialchars(strip_tags($this->error_code));
        $this->test_run = (int)$this->test_run; // Cast to int for tinyint
        $this->location = htmlspecialchars(strip_tags($this->location));
        $this->layer = htmlspecialchars(strip_tags($this->layer));
        $this->remarks = htmlspecialchars(strip_tags($this->remarks));
        

        $stmt->bind_param("sssisiiiissssss",
            $this->report_index,
            $this->report_type,
            $this->generator_serial_number,
            $this->customer_id,
            $this->problem_issue,
            $this->running_hours,
            $this->load_test,
            $this->load_hour,
            $this->load_amount,
            $this->used_for,
            $this->error_code,
            $this->test_run,
            $this->location,
            $this->layer,
            $this->remarks
        );

        if ($stmt->execute()) {
            $this->id = $this->conn->insert_id;
            return true;
        }
        error_log("Report creation failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }
    public function generateUniqueReportIndex(): string
    {
        // Example format: REHLKO-YYYYMMDD-HHMMSS-RANDOM (e.g., REHLKO-20240715-143000-A1B2)
        $timestamp = date('Ymd-His');
        $randomPart = substr(md5(uniqid(rand(), true)), 0, 6); // Generate a short random string
        return 'REHLKO-' . $timestamp . '-' . strtoupper($randomPart);
    }

    /**
     * Reads all report records with role-based filtering.
     *
     * @return mysqli_result|false The result set on success, false on failure.
     */
    public function readAll() {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        $query = "SELECT r.id, r.report_index, r.report_type, r.generator_serial_number, r.customer_id,
                            r.problem_issue, r.running_hours, r.load_test, r.load_hour, r.load_amount,
                            r.used_for, r.error_code, r.test_run, r.location, r.layer, r.remarks,
                            r.status, r.is_pick_up, r.engineer_id, r.supervisor_id,
                            r.created_datetime, r.modified_datetime,
                            uc.fullname AS customer_name, ue.fullname AS engineer_name,
                            us.fullname AS supervisor_name, p.generator_model_number
                     FROM " . $this->table_name . " r
                     LEFT JOIN users uc ON r.customer_id = uc.id
                     LEFT JOIN users ue ON r.engineer_id = ue.id
                     LEFT JOIN users us ON r.supervisor_id = us.id
                     LEFT JOIN products p ON r.generator_serial_number = p.generator_serial_number";

        $whereClauses = [];
        $bindParams = [];
        $bindParamTypes = '';

        if (in_array($currentUserRole, ['superadmin', 'admin', 'supervisor'])) {
            // Superadmin, Admin, Supervisor can read all reports.
        } elseif (in_array($currentUserRole, ['engineer'])) {
            // Engineers can pick up and solve, and read reports.
            // They can read all reports, but will only act on unpicked or assigned ones.
            // More specifically, they should read reports relevant to their support teams.
            $query .= " LEFT JOIN support_team st ON r.customer_id = st.customer_id";
            $whereClauses[] = "(r.engineer_id = ? OR r.status = 'pending' OR (st.engineer_id = ? AND r.status != 'completed'))";
            $bindParams[] = $currentUserId;
            $bindParams[] = $currentUserId;
            $bindParamTypes .= 'ii';
        } elseif (in_array($currentUserRole, ['member', 'champion'])) {
            // Members and Champions only read reports of customers who are in their support team.
            $query .= " JOIN support_team st ON r.customer_id = st.customer_id";
            $whereClauses[] = "(st.champion_id = ? OR st.member_id = ?)";
            $bindParams[] = $currentUserId;
            $bindParams[] = $currentUserId;
            $bindParamTypes .= 'ii';
        } elseif (in_array($currentUserRole, ['localcustomer', 'globalcustomer'])) {
            // Customers create reports and read their own.
            $whereClauses[] = "r.customer_id = ?";
            $bindParams[] = $currentUserId;
            $bindParamTypes .= 'i';
        } else {
            // If current user role is not authorized, send an error response immediately.
            // This case should ideally be handled by the controller's initial authorization check.
            sendJsonResponse(['message' => 'Unauthorized to read reports.'], 403);
            return false;
        }

        if (!empty($whereClauses)) {
            $query .= " WHERE " . implode(" AND ", $whereClauses);
        }

        $query .= " ORDER BY r.modified_datetime DESC";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        if (!empty($bindParams)) {
            $stmt->bind_param($bindParamTypes, ...$bindParams);
        }

        if ($stmt->execute()) {
            return $stmt->get_result();
        }
        error_log("Read all reports failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }
    
    public function recents() {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        $query = "SELECT r.id, r.report_index, r.report_type, r.generator_serial_number, r.customer_id,
                            r.problem_issue, r.running_hours, r.load_test, r.load_hour, r.load_amount,
                            r.used_for, r.error_code, r.test_run, r.location, r.layer, r.remarks,
                            r.status, r.is_pick_up, r.engineer_id, r.supervisor_id,
                            r.created_datetime, r.modified_datetime,
                            uc.fullname AS customer_name, ue.fullname AS engineer_name,
                            us.fullname AS supervisor_name, p.generator_model_number
                     FROM " . $this->table_name . " r
                     LEFT JOIN users uc ON r.customer_id = uc.id
                     LEFT JOIN users ue ON r.engineer_id = ue.id
                     LEFT JOIN users us ON r.supervisor_id = us.id
                     LEFT JOIN products p ON r.generator_serial_number = p.generator_serial_number";

        $whereClauses = [];
        $bindParams = [];
        $bindParamTypes = '';

        if (in_array($currentUserRole, ['superadmin', 'admin', 'supervisor'])) {
            // Superadmin, Admin, Supervisor can read all reports.
        } elseif (in_array($currentUserRole, ['engineer'])) {
            // Engineers can pick up and solve, and read reports.
            // They can read all reports, but will only act on unpicked or assigned ones.
            // More specifically, they should read reports relevant to their support teams.
            $query .= " LEFT JOIN support_team st ON r.customer_id = st.customer_id";
            $whereClauses[] = "(r.engineer_id = ? OR r.status = 'pending' OR (st.engineer_id = ? AND r.status != 'completed'))";
            $bindParams[] = $currentUserId;
            $bindParams[] = $currentUserId;
            $bindParamTypes .= 'ii';
        } elseif (in_array($currentUserRole, ['member', 'champion'])) {
            // Members and Champions only read reports of customers who are in their support team.
            $query .= " JOIN support_team st ON r.customer_id = st.customer_id";
            $whereClauses[] = "(st.champion_id = ? OR st.member_id = ?)";
            $bindParams[] = $currentUserId;
            $bindParams[] = $currentUserId;
            $bindParamTypes .= 'ii';
        } elseif (in_array($currentUserRole, ['localcustomer', 'globalcustomer'])) {
            // Customers create reports and read their own.
            $whereClauses[] = "r.customer_id = ?";
            $bindParams[] = $currentUserId;
            $bindParamTypes .= 'i';
        } else {
            // If current user role is not authorized, send an error response immediately.
            // This case should ideally be handled by the controller's initial authorization check.
            sendJsonResponse(['message' => 'Unauthorized to read reports.'], 403);
            return false;
        }

        if (!empty($whereClauses)) {
            $query .= " WHERE " . implode(" AND ", $whereClauses);
        }

        $query .= " ORDER BY r.modified_datetime DESC LIMIT 5";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        if (!empty($bindParams)) {
            $stmt->bind_param($bindParamTypes, ...$bindParams);
        }

        if ($stmt->execute()) {
            return $stmt->get_result();
        }
        error_log("Read all reports failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }
    /**
     * Reads a single report record by ID or report_index.
     *
     * @param string $identifier The ID or report_index of the report.
     * @param string $type 'id' or 'index' to specify identifier type.
     * @return array|false The report data as an associative array on success, false if not found.
     */
    public function readOne($identifier, $type = 'id') {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        $query = "SELECT r.id, r.report_index, r.report_type, r.generator_serial_number, r.customer_id,
                            r.problem_issue, r.running_hours, r.load_test, r.load_hour, r.load_amount,
                            r.used_for, r.error_code, r.test_run, r.location, r.layer, r.remarks,
                            r.status, r.is_pick_up, r.engineer_id, r.supervisor_id,
                            r.created_datetime, r.modified_datetime,
                            uc.fullname AS customer_name, ue.fullname AS engineer_name,
                            us.fullname AS supervisor_name, p.generator_model_number
                     FROM " . $this->table_name . " r
                     LEFT JOIN users uc ON r.customer_id = uc.id
                     LEFT JOIN users ue ON r.engineer_id = ue.id
                     LEFT JOIN users us ON r.supervisor_id = us.id
                     LEFT JOIN products p ON r.generator_serial_number = p.generator_serial_number
                     WHERE ";

        if ($type === 'id') {
            $query .= "r.id = ?";
        } else { // type === 'index'
            $query .= "r.report_index = ?";
        }
        $query .= " LIMIT 0,1";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize and bind
        $identifier = htmlspecialchars(strip_tags($identifier));
        if ($type === 'id') {
            $stmt->bind_param("i", $identifier);
        } else {
            $stmt->bind_param("s", $identifier);
        }

        if ($stmt->execute()) {
            $result = $stmt->get_result();
            $report = $result->fetch_assoc();

            if ($report) {
                // Detailed authorization for reading a single report
                if (in_array($currentUserRole, ['superadmin', 'admin', 'supervisor'])) {
                    return $report;
                } elseif (in_array($currentUserRole, ['localcustomer', 'globalcustomer'])) {
                    if ($report['customer_id'] == $currentUserId) {
                        return $report;
                    } else {
                        sendJsonResponse(['message' => 'Unauthorized to view this report.'], 403);
                        return false;
                    }
                } elseif (in_array($currentUserRole, ['engineer'])) {
                    // Engineer can read if assigned, or if it's pending and they are part of the support team
                    if ($report['engineer_id'] == $currentUserId ||
                        ($report['status'] === 'pending' && $this->isEngineerInSupportTeam($report['customer_id'], $currentUserId))) {
                        return $report;
                    } else {
                        sendJsonResponse(['message' => 'Unauthorized to view this report.'], 403);
                        return false;
                    }
                } elseif (in_array($currentUserRole, ['member', 'champion'])) {
                    // Members/Champions can only read if the customer is in their support team
                    if ($this->isMemberOrChampionInSupportTeam($report['customer_id'], $currentUserId)) {
                        return $report;
                    } else {
                        sendJsonResponse(['message' => 'Unauthorized to view this report.'], 403);
                        return false;
                    }
                } else {
                    sendJsonResponse(['message' => 'Unauthorized to view this report.'], 403);
                    return false;
                }
            }
        }
        error_log("Read one report failed or report not found: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Helper to check if engineer is in customer's support team.
     */
    private function isEngineerInSupportTeam($customerId, $engineerId) {
        $query = "SELECT COUNT(*) FROM support_team WHERE customer_id = ? AND engineer_id = ?";
        $stmt = $this->conn->prepare($query);
        if ($stmt === false) { error_log("isEngineerInSupportTeam prepare failed: " . $this->conn->error); return false; }
        $stmt->bind_param("ii", $customerId, $engineerId);
        $stmt->execute();
        $count = $stmt->get_result()->fetch_row()[0];
        return $count > 0;
    }

    /**
     * Helper to check if member/champion is in customer's support team.
     */
    private function isMemberOrChampionInSupportTeam($customerId, $userId) {
        $query = "SELECT COUNT(*) FROM support_team WHERE customer_id = ? AND (champion_id = ? OR member_id = ?)";
        $stmt = $this->conn->prepare($query);
        if ($stmt === false) { error_log("isMemberOrChampionInSupportTeam prepare failed: " . $this->conn->error); return false; }
        $stmt->bind_param("iii", $customerId, $userId, $userId);
        $stmt->execute();
        $count = $stmt->get_result()->fetch_row()[0];
        return $count > 0;
    }


    /**
     * Retrieves pending reports (status 'pending') for cron job.
     *
     * @return array An array of pending report records.
     */
    public function getPendingReports() {
        $query = "SELECT r.id, r.report_index, r.customer_id, r.created_datetime, r.problem_issue,
                            uc.role AS customer_role, uc.fullname AS customer_fullname, uc.email AS customer_email
                     FROM " . $this->table_name . " r
                     JOIN users uc ON r.customer_id = uc.id
                     WHERE r.status = 'pending' AND r.is_pick_up = 0
                     ORDER BY r.created_datetime ASC";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed for getPendingReports: (" . $this->conn->errno . ") " . $this->conn->error);
            return [];
        }

        if ($stmt->execute()) {
            $result = $stmt->get_result();
            $reports = [];
            while ($row = $result->fetch_assoc()) {
                $reports[] = $row;
            }
            return $reports;
        }
        error_log("Get pending reports failed: (" . $stmt->errno . ") " . $stmt->error);
        return [];
    }


    /**
     * Retrieves reports with status 'checking' for cron job.
     *
     * @return array An array of reports in 'checking' status.
     */
    public function getCheckingReports() {
        $query = "SELECT r.id, r.report_index, r.customer_id, r.engineer_id, r.created_datetime,
                            uc.fullname AS customer_fullname, uc.email AS customer_email,
                            ue.fullname AS engineer_fullname
                     FROM " . $this->table_name . " r
                     JOIN users uc ON r.customer_id = uc.id
                     LEFT JOIN users ue ON r.engineer_id = ue.id
                     WHERE r.status = 'checking'
                     ORDER BY r.modified_datetime ASC";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed for getCheckingReports: (" . $this->conn->errno . ") " . $this->conn->error);
            return [];
        }

        if ($stmt->execute()) {
            $result = $stmt->get_result();
            $reports = [];
            while ($row = $result->fetch_assoc()) {
                $reports[] = $row;
            }
            return $reports;
        }
        error_log("Get checking reports failed: (" . $stmt->errno . ") " . $stmt->error);
        return [];
    }


    /**
     * Updates the status of a report.
     *
     * @param int $id The ID of the report to update.
     * @param string $status The new status (e.g., 'solving', 'checking', 'completed').
     * @param int|null $engineerId Optional: The ID of the engineer picking up the report.
     * @param int|null $supervisorId Optional: The ID of the supervisor checking/completing the report.
     * @param bool $isPickedUp Optional: Set to 1 if report is picked up.
     * @return bool True on success, false on failure.
     */
    public function updateStatus($id, $status, $engineerId = null, $supervisorId = null, $isPickedUp = null) {
        $setClauses = ["status = ?"];
        $bindParams = [$status];
        $bindParamTypes = 's';

        if ($isPickedUp !== null) {
            $setClauses[] = "is_pick_up = ?";
            $bindParams[] = $isPickedUp;
            $bindParamTypes .= 'i';
        }
        if ($engineerId !== null) {
            $setClauses[] = "engineer_id = ?";
            $bindParams[] = $engineerId;
            $bindParamTypes .= 'i';
        }
        if ($supervisorId !== null) {
            $setClauses[] = "supervisor_id = ?";
            $bindParams[] = $supervisorId;
            $bindParamTypes .= 'i';
        }

        $query = "UPDATE " . $this->table_name . "
                  SET " . implode(", ", $setClauses) . "
                  WHERE id = ?";
        $bindParams[] = $id; // Add ID to bind parameters
        $bindParamTypes .= 'i'; // Add type for ID

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed for updateStatus: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $stmt->bind_param($bindParamTypes, ...$bindParams);

        if ($stmt->execute()) {
            return true;
        }
        error_log("Update report status failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Updates an existing report record (for general updates, not just status).
     *
     * @return bool True on success, false on failure.
     */
    public function update() {
        $query = "UPDATE " . $this->table_name . "
                  SET report_type=?, generator_serial_number=?, problem_issue=?,
                      running_hours=?, load_test=?, load_hour=?, load_amount=?,
                      used_for=?, error_code=?, test_run=?, location=?, layer=?, remarks=?
                  WHERE id=?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize inputs
        $this->report_type = htmlspecialchars(strip_tags($this->report_type));
        $this->generator_serial_number = htmlspecialchars(strip_tags($this->generator_serial_number));
        $this->problem_issue = htmlspecialchars(strip_tags($this->problem_issue));
        $this->running_hours = htmlspecialchars(strip_tags($this->running_hours));
        $this->load_test = (int)$this->load_test;
        $this->load_hour = htmlspecialchars(strip_tags($this->load_hour));
        $this->load_amount = htmlspecialchars(strip_tags($this->load_amount));
        $this->used_for = htmlspecialchars(strip_tags($this->used_for));
        $this->error_code = htmlspecialchars(strip_tags($this->error_code));
        $this->test_run = (int)$this->test_run;
        $this->location = htmlspecialchars(strip_tags($this->location));
        $this->layer = htmlspecialchars(strip_tags($this->layer));
        $this->remarks = htmlspecialchars(strip_tags($this->remarks));
        $this->id = htmlspecialchars(strip_tags($this->id));

        $stmt->bind_param("sssiissssssssi",
            $this->report_type,
            $this->generator_serial_number,
            $this->problem_issue,
            $this->running_hours,
            $this->load_test,
            $this->load_hour,
            $this->load_amount,
            $this->used_for,
            $this->error_code,
            $this->test_run,
            $this->location,
            $this->layer,
            $this->remarks,
            $this->id
        );

        if ($stmt->execute()) {
            return true;
        }
        error_log("Report update failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }


    /**
     * Deletes a report record by ID.
     *
     * @return bool True on success, false on failure.
     */
    public function delete() {
        $query = "DELETE FROM " . $this->table_name . " WHERE id = ?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $this->id = htmlspecialchars(strip_tags($this->id));
        $stmt->bind_param("i", $this->id);

        if ($stmt->execute()) {
            return true;
        }
        error_log("Report deletion failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Retrieves all necessary data for generating a PDF report.
     *
     * @param int $reportId The ID of the report.
     * @return array|false The comprehensive report data on success, false on failure.
     */
    public function getReportDetailsForPdf($reportIndex) {
        // Main report details
        $reportQuery = "SELECT r.*,
                                uc.fullname AS customer_name, uc.email AS customer_email, uc.phone AS customer_phone,
                                ue.fullname AS engineer_name, us.fullname AS supervisor_name,
                                p.generator_model_number
                         FROM " . $this->table_name . " r
                         LEFT JOIN users uc ON r.customer_id = uc.id
                         LEFT JOIN users ue ON r.engineer_id = ue.id
                         LEFT JOIN users us ON r.supervisor_id = us.id
                         LEFT JOIN products p ON r.generator_serial_number = p.generator_serial_number
                         WHERE r.report_index = ?";
        $reportStmt = $this->conn->prepare($reportQuery);
        if ($reportStmt === false) { error_log("PDF report query prepare failed: " . $this->conn->error); return false; }
        $reportStmt->bind_param("s", $reportIndex);
        $reportStmt->execute();
        $reportData = $reportStmt->get_result()->fetch_assoc();

        if (!$reportData) {
            return false;
        }

        $reportIndex = $reportData['report_index'];

        // Fetch related data: root_cause, corrective_action, effective_action, prevention, suggestion, report_media
        $relatedTables = [
            'root_cause' => ['root_cause_title', 'root_cause_media_type', 'root_cause_media_path', 'root_cause_media_name'],
            'corrective_action' => ['corrective_action_title', 'corrective_action_media_type', 'corrective_action_media_path', 'corrective_action_media_name'],
            'effective_action' => ['effective_action_title'],
            'prevention' => ['prevention_title'],
            'suggestion' => ['suggestion_title'],
            'report_media' => ['media_type', 'media_path', 'media_name']
        ];

        foreach ($relatedTables as $tableName => $columns) {
            $selectCols = implode(', ', $columns);
            $subQuery = "SELECT {$selectCols} FROM {$tableName} WHERE report_index = ?";
            $subStmt = $this->conn->prepare($subQuery);
            if ($subStmt === false) { error_log("PDF subquery prepare failed for $tableName: " . $this->conn->error); continue; }
            $subStmt->bind_param("s", $reportIndex);
            $subStmt->execute();
            $result = $subStmt->get_result();
            $reportData[$tableName] = [];
            while ($row = $result->fetch_assoc()) {
                $reportData[$tableName][] = $row;
            }
        }

        return $reportData;
    }

    /**
     * Reads reports assigned to a specific engineer OR reports with 'pending' status
     * that the engineer is authorized to pick up (based on support team assignment).
     *
     * @param int $engineerId The ID of the engineer.
     * @param string|null $filterStatus Optional: Filter reports by a specific status.
     * @param string|null $searchTerm Optional: Search term for report_index, problem_issue, or generator_serial_number.
     * @return mysqli_result|false The result set on success, false on failure.
     */
    public function readByEngineerOrPending($engineerId, $filterStatus = null, $searchTerm = null) {
        $query = "SELECT r.id, r.report_index, r.report_type, r.generator_serial_number, r.customer_id,
                            r.problem_issue, r.running_hours, r.load_test, r.load_hour, r.load_amount,
                            r.used_for, r.error_code, r.test_run, r.location, r.layer, r.remarks,
                            r.status, r.is_pick_up, r.engineer_id, r.supervisor_id,
                            r.created_datetime, r.modified_datetime,
                            uc.fullname AS customer_name, ue.fullname AS engineer_name,
                            us.fullname AS supervisor_name, p.generator_model_number
                     FROM " . $this->table_name . " r
                     LEFT JOIN users uc ON r.customer_id = uc.id
                     LEFT JOIN users ue ON r.engineer_id = ue.id
                     LEFT JOIN users us ON r.supervisor_id = us.id
                     LEFT JOIN products p ON r.generator_serial_number = p.generator_serial_number
                     LEFT JOIN support_team st ON r.customer_id = st.customer_id "; // Join support_team to check pending reports

        $whereClauses = [];
        $bindParams = [];
        $bindParamTypes = '';

        // Condition for reports assigned to the engineer OR pending reports in their support team
        $whereClauses[] = "(r.engineer_id = ? OR (r.status = 'pending' AND st.engineer_id = ?))";
        $bindParams[] = $engineerId;
        $bindParams[] = $engineerId;
        $bindParamTypes .= 'ii';

        if ($filterStatus) {
            $whereClauses[] = "r.status = ?";
            $bindParams[] = $filterStatus;
            $bindParamTypes .= 's';
        }

        if ($searchTerm) {
            $searchTerm = '%' . $searchTerm . '%';
            $whereClauses[] = "(r.report_index LIKE ? OR r.problem_issue LIKE ? OR r.generator_serial_number LIKE ?)";
            $bindParams[] = $searchTerm;
            $bindParams[] = $searchTerm;
            $bindParams[] = $searchTerm;
            $bindParamTypes .= 'sss';
        }

        if (!empty($whereClauses)) {
            $query .= " WHERE " . implode(" AND ", $whereClauses);
        }

        $query .= " ORDER BY r.created_datetime DESC";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed for readByEngineerOrPending: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        if (!empty($bindParams)) {
            $stmt->bind_param($bindParamTypes, ...$bindParams);
        }

        if ($stmt->execute()) {
            return $stmt->get_result();
        }
        error_log("Read reports by engineer or pending failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }
    
    /**
     * Reads all report records associated with a specific customer ID.
     *
     * @param int $customerId The ID of the customer.
     * @param string|null $filterStatus Optional: Filter reports by a specific status.
     * @param string|null $searchTerm Optional: Search term for report_index, problem_issue, or generator_serial_number.
     * @return mysqli_result|false The result set on success, false on failure.
     */
    public function readByCustomer($customerId, $filterStatus = null, $searchTerm = null) {
        $query = "SELECT r.id, r.report_index, r.report_type, r.generator_serial_number, r.customer_id,
                            r.problem_issue, r.running_hours, r.load_test, r.load_hour, r.load_amount,
                            r.used_for, r.error_code, r.test_run, r.location, r.layer, r.remarks,
                            r.status, r.is_pick_up, r.engineer_id, r.supervisor_id,
                            r.created_datetime, r.modified_datetime,
                            uc.fullname AS customer_name, ue.fullname AS engineer_name,
                            us.fullname AS supervisor_name, p.generator_model_number
                     FROM " . $this->table_name . " r
                     LEFT JOIN users uc ON r.customer_id = uc.id
                     LEFT JOIN users ue ON r.engineer_id = ue.id
                     LEFT JOIN users us ON r.supervisor_id = us.id
                     LEFT JOIN products p ON r.generator_serial_number = p.generator_serial_number
                     WHERE r.customer_id = ?";

        $whereClauses = [];
        $bindParams = [$customerId];
        $bindParamTypes = 'i';

        if ($filterStatus) {
            $whereClauses[] = "r.status = ?";
            $bindParams[] = $filterStatus;
            $bindParamTypes .= 's';
        }

        if ($searchTerm) {
            $searchTerm = '%' . $searchTerm . '%';
            $whereClauses[] = "(r.report_index LIKE ? OR r.problem_issue LIKE ? OR r.generator_serial_number LIKE ?)";
            $bindParams[] = $searchTerm;
            $bindParams[] = $searchTerm;
            $bindParams[] = $searchTerm;
            $bindParamTypes .= 'sss';
        }

        if (!empty($whereClauses)) {
            $query .= " AND " . implode(" AND ", $whereClauses);
        }

        $query .= " ORDER BY r.created_datetime DESC";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed for readByCustomer: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        if (!empty($bindParams)) {
            $stmt->bind_param($bindParamTypes, ...$bindParams);
        }

        if ($stmt->execute()) {
            return $stmt->get_result();
        }
        error_log("Read reports by customer failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }
}