<?php
// models/ReportReminder.php

/**
 * ReportReminder Model for Rehlko Customer Care application.
 * Manages data interaction for the 'report_reminders' table,
 * used to track sent warning notifications for reports.
 */
class ReportReminder {
    private $conn;
    private $table_name = "report_reminders";

    // Object properties
    public $id;
    public $report_id;
    public $reminder_type; // 'first_warning', 'last_warning', 'checking_notification', 'completed_notification'
    public $sent_at;
    public $recipients_count;

    /**
     * Constructor for the ReportReminder model.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
    }

    /**
     * Adds a new reminder record.
     *
     * @param int $reportId The ID of the report.
     * @param string $reminderType The type of reminder ('first_warning', 'last_warning', 'checking_notification', 'completed_notification').
     * @param int $recipientsCount The number of recipients the reminder was sent to.
     * @return bool True on success, false on failure.
     */
    public function addReminder($reportId, $reminderType, $recipientsCount = 0) {
        $query = "INSERT INTO " . $this->table_name . "
                  SET report_id=?, reminder_type=?, recipients_count=?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize inputs
        $reportId = htmlspecialchars(strip_tags($reportId));
        $reminderType = htmlspecialchars(strip_tags($reminderType));
        $recipientsCount = htmlspecialchars(strip_tags($recipientsCount));

        $stmt->bind_param("isi", $reportId, $reminderType, $recipientsCount);

        if ($stmt->execute()) {
            return true;
        }
        error_log("Adding report reminder failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Checks if a specific type of reminder has already been sent for a report.
     *
     * @param int $reportId The ID of the report.
     * @param string $reminderType The type of reminder ('first_warning', 'last_warning', etc.).
     * @return bool True if the reminder has been sent, false otherwise.
     */
    public function hasReminderBeenSent($reportId, $reminderType) {
        $query = "SELECT COUNT(*) FROM " . $this->table_name . "
                  WHERE report_id = ? AND reminder_type = ?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return true; // Assume sent on error to prevent duplicate attempts
        }

        $stmt->bind_param("is", $reportId, $reminderType);
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            $row = $result->fetch_row();
            return $row[0] > 0;
        }
        error_log("Check hasReminderBeenSent failed: (" . $stmt->errno . ") " . $stmt->error);
        return true; // Assume sent on error
    }

    /**
     * Marks a specific status notification as sent for a report.
     * This is useful for 'checking' and 'completed' status notifications.
     *
     * @param int $reportId The ID of the report.
     * @param string $statusType The status type (e.g., 'checking', 'completed').
     * @return bool True on success, false on failure.
     */
    public function markStatusNotificationSent($reportId, $statusType) {
        return $this->addReminder($reportId, $statusType . '_notification');
    }

    /**
     * Checks if a specific status notification has already been sent for a report.
     *
     * @param int $reportId The ID of the report.
     * @param string $statusType The status type (e.g., 'checking', 'completed').
     * @return bool True if the notification has been sent, false otherwise.
     */
    public function hasStatusNotificationBeenSent($reportId, $statusType) {
        return $this->hasReminderBeenSent($reportId, $statusType . '_notification');
    }
}
?>
