<?php
// models/RootCause.php

/**
 * RootCause Model for Rehlko Customer Care application.
 * Manages CRUD operations and data interaction for the 'root_cause' table.
 */
class RootCause {
    private $conn;
    private $table_name = "root_cause";

    // Object properties
    public $id;
    public $report_index;
    public $root_cause_title;
    public $root_cause_media_type;
    public $root_cause_media_path;
    public $root_cause_media_name;
    public $created_user;
    public $created_datetime;
    public $modified_datetime;

    /**
     * Constructor for the RootCause model.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
    }

    /**
     * Creates a new root cause record.
     *
     * @return bool True on success, false on failure.
     */
    public function create() {
        $query = "INSERT INTO " . $this->table_name . "
                  SET report_index=?, root_cause_title=?, root_cause_media_type=?,
                      root_cause_media_path=?, root_cause_media_name=?, created_user=?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize inputs
        $this->report_index = htmlspecialchars(strip_tags($this->report_index));
        $this->root_cause_title = htmlspecialchars(strip_tags($this->root_cause_title));
        $this->root_cause_media_type = htmlspecialchars(strip_tags($this->root_cause_media_type));
        $this->root_cause_media_path = htmlspecialchars(strip_tags($this->root_cause_media_path));
        $this->root_cause_media_name = htmlspecialchars(strip_tags($this->root_cause_media_name));
        $this->created_user = htmlspecialchars(strip_tags($this->created_user));


        $stmt->bind_param("sssssi",
            $this->report_index,
            $this->root_cause_title,
            $this->root_cause_media_type,
            $this->root_cause_media_path,
            $this->root_cause_media_name,
            $this->created_user
        );

        if ($stmt->execute()) {
            return true;
        }
        error_log("Root cause creation failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Reads root cause records for a specific report index.
     *
     * @param string $reportIndex The report index to filter by.
     * @return array An array of root cause records.
     */
    public function readAllByReportIndex($reportIndex) {
        $query = "SELECT id, report_index, root_cause_title, root_cause_media_type,
                         root_cause_media_path, root_cause_media_name, created_user,
                         created_datetime, modified_datetime
                  FROM " . $this->table_name . "
                  WHERE report_index = ?
                  ORDER BY created_datetime DESC";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed for readAllByReportIndex: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }
        $stmt->bind_param("s", $reportIndex);
        $stmt->execute();
        return $stmt->get_result();
    }

    /**
     * Reads a single root cause record by ID.
     *
     * @return array|false The root cause data as an associative array on success, false if not found.
     */
    public function readOne() {
        $query = "SELECT id, report_index, root_cause_title, root_cause_media_type,
                         root_cause_media_path, root_cause_media_name, created_user,
                         created_datetime, modified_datetime
                  FROM " . $this->table_name . "
                  WHERE id = ? LIMIT 0,1";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $this->id = htmlspecialchars(strip_tags($this->id));
        $stmt->bind_param("i", $this->id);

        if ($stmt->execute()) {
            $result = $stmt->get_result();
            return $result->fetch_assoc();
        }
        error_log("Read one root cause failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Updates an existing root cause record.
     *
     * @return bool True on success, false on failure.
     */
    public function update() {
        $query = "UPDATE " . $this->table_name . "
                  SET root_cause_title=?, root_cause_media_type=?,
                      root_cause_media_path=?, root_cause_media_name=?,
                      created_user=?, modified_datetime=CURRENT_TIMESTAMP
                  WHERE id = ?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed for update root cause: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize inputs
        $this->root_cause_title = htmlspecialchars(strip_tags($this->root_cause_title));
        $this->root_cause_media_type = htmlspecialchars(strip_tags($this->root_cause_media_type));
        $this->root_cause_media_path = htmlspecialchars(strip_tags($this->root_cause_media_path));
        $this->root_cause_media_name = htmlspecialchars(strip_tags($this->root_cause_media_name));
        $this->modified_user = htmlspecialchars(strip_tags($this->modified_user));
        $this->id = htmlspecialchars(strip_tags($this->id));

        $stmt->bind_param("ssssii",
            $this->root_cause_title,
            $this->root_cause_media_type,
            $this->root_cause_media_path,
            $this->root_cause_media_name,
            $this->modified_user,
            $this->id
        );

        if ($stmt->execute()) {
            return true;
        }
        error_log("Root cause update failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }
    
    public function uploadMedia($file, $reportIndex, $mediaType) {
        $uploadDir = __DIR__ . '/../public/uploads/rootcause';
        if (!is_dir($uploadDir)) {
            mkdir($uploadDir, 0777, true); // Create directory if it doesn't exist
        }

        $originalFileName = basename($file['name']);
        $fileExtension = pathinfo($originalFileName, PATHINFO_EXTENSION);
        $newFileName = uniqid('media_') . '.' . $fileExtension;
        $targetFilePath = $uploadDir . $newFileName;
        $relativeFilePath = 'public/uploads/rootcause' . $newFileName; // Path to store in DB

        if (move_uploaded_file($file['tmp_name'], $targetFilePath)) {
            return [
                'media_path' => $relativeFilePath,
                'media_name' => $originalFileName,
            ];
        } else {
            error_log("Failed to move uploaded file: " . $file['tmp_name'] . " to " . $targetFilePath);
            return false;
        }
    }

    /**
     * Deletes a root cause record by ID and its associated media file if any.
     *
     * @return bool True on success, false on failure.
     */
    public function delete() {
        // First, get the media path from the database
        $query = "SELECT root_cause_media_path FROM " . $this->table_name . " WHERE id = ?";
        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed for delete root cause (select path): (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }
        $this->id = htmlspecialchars(strip_tags($this->id));
        $stmt->bind_param("i", $this->id);
        $stmt->execute();
        $result = $stmt->get_result();
        $row = $result->fetch_assoc();

        $mediaPath = null;
        if ($row && !empty($row['root_cause_media_path'])) {
            $mediaPath = __DIR__ . '/../public/' . $row['root_cause_media_path']; // Full path
        }

        // Delete the database record
        $deleteQuery = "DELETE FROM " . $this->table_name . " WHERE id = ?";
        $deleteStmt = $this->conn->prepare($deleteQuery);
        if ($deleteStmt === false) {
            error_log("Prepare failed for delete root cause (delete record): (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }
        $deleteStmt->bind_param("i", $this->id);

        if ($deleteStmt->execute()) {
            // If DB record deleted, attempt to delete the file if it exists
            if ($mediaPath && file_exists($mediaPath)) {
                if (unlink($mediaPath)) {
                    return true;
                } else {
                    error_log("Failed to delete root cause media file: " . $mediaPath);
                    return false; // DB record deleted, but file failed
                }
            }
            return true; // DB record deleted, no file or file didn't exist
        } else {
            error_log("Root cause deletion failed: (" . $deleteStmt->errno . ") " . $deleteStmt->error);
            return false;
        }
    }
}
?>
