<?php
// models/Suggestion.php

/**
 * Suggestion Model for Rehlko Customer Care application.
 * Manages CRUD operations and data interaction for the 'suggestion' table.
 */
class Suggestion {
    private $conn;
    private $table_name = "suggestion";

    // Object properties
    public $id;
    public $report_index;
    public $suggestion_title;
    public $created_user;
    public $created_datetime;
    public $modified_datetime;

    /**
     * Constructor for the Suggestion model.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
    }

    /**
     * Creates a new suggestion record.
     *
     * @return bool True on success, false on failure.
     */
    public function create() {
        $query = "INSERT INTO " . $this->table_name . "
                  SET report_index=?, suggestion_title=?, created_user=?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize inputs
        $this->report_index = htmlspecialchars(strip_tags($this->report_index));
        $this->suggestion_title = htmlspecialchars(strip_tags($this->suggestion_title));
        $this->created_user = htmlspecialchars(strip_tags($this->created_user));

        $stmt->bind_param("ssi", $this->report_index, $this->suggestion_title, $this->created_user);

        if ($stmt->execute()) {
            $this->id = $this->conn->insert_id;
            return true;
        }
        error_log("Suggestion creation failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Reads suggestion records for a specific report index.
     *
     * @param string $reportIndex The report index to filter by.
     * @return array An array of suggestion records.
     */
    public function readByReportIndex($reportIndex) {
        $query = "SELECT id, report_index, suggestion_title, created_user, created_datetime, modified_datetime
                  FROM " . $this->table_name . "
                  WHERE report_index = ?
                  ORDER BY created_datetime ASC";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return [];
        }

        $reportIndex = htmlspecialchars(strip_tags($reportIndex));
        $stmt->bind_param("s", $reportIndex);

        if ($stmt->execute()) {
            $result = $stmt->get_result();
            $suggestions = [];
            while ($row = $result->fetch_assoc()) {
                $suggestions[] = $row;
            }
            return $suggestions;
        }
        error_log("Read suggestions by report index failed: (" . $stmt->errno . ") " . $stmt->error);
        return [];
    }

    /**
     * Reads a single suggestion record by ID.
     *
     * @return array|false The suggestion data as an associative array on success, false if not found.
     */
    public function readOne() {
        $query = "SELECT id, report_index, suggestion_title, created_user, created_datetime, modified_datetime
                  FROM " . $this->table_name . "
                  WHERE id = ? LIMIT 0,1";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $this->id = htmlspecialchars(strip_tags($this->id));
        $stmt->bind_param("i", $this->id);

        if ($stmt->execute()) {
            $result = $stmt->get_result();
            return $result->fetch_assoc();
        }
        error_log("Read one suggestion failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Updates an existing suggestion record.
     *
     * @return bool True on success, false on failure.
     */
    public function update() {
        $query = "UPDATE " . $this->table_name . "
                  SET suggestion_title=?
                  WHERE id=?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize inputs
        $this->suggestion_title = htmlspecialchars(strip_tags($this->suggestion_title));
        $this->id = htmlspecialchars(strip_tags($this->id));

        $stmt->bind_param("si", $this->suggestion_title, $this->id);

        if ($stmt->execute()) {
            return true;
        }
        error_log("Suggestion update failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Deletes a suggestion record by ID.
     *
     * @return bool True on success, false on failure.
     */
    public function delete() {
        $query = "DELETE FROM " . $this->table_name . " WHERE id = ?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $this->id = htmlspecialchars(strip_tags($this->id));
        $stmt->bind_param("i", $this->id);

        if ($stmt->execute()) {
            return true;
        }
        error_log("Suggestion deletion failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }
}
?>
