<?php
// models/SupportTeam.php

/**
 * SupportTeam Model for Rehlko Customer Care application.
 * Manages CRUD operations and data interaction for the 'support_team' table.
 */
class SupportTeam {
    private $conn;
    private $table_name = "support_team";

    // Object properties
    public $id;
    public $customer_id;
    public $engineer_id;
    public $champion_id;
    public $member_id;
    public $created_time;
    public $modified_time;

    /**
     * Constructor for the SupportTeam model.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
    }

    /**
     * Creates a new support team record.
     *
     * @return bool True on success, false on failure.
     */
    public function create() {
        $query = "INSERT INTO " . $this->table_name . "
                  SET customer_id=?, engineer_id=?, champion_id=?, member_id=?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize inputs
        $this->customer_id = htmlspecialchars(strip_tags($this->customer_id));
        $this->engineer_id = htmlspecialchars(strip_tags($this->engineer_id));
        $this->champion_id = htmlspecialchars(strip_tags($this->champion_id));
        $this->member_id = htmlspecialchars(strip_tags($this->member_id));

        $stmt->bind_param("iiii", $this->customer_id, $this->engineer_id, $this->champion_id, $this->member_id);

        if ($stmt->execute()) {
            $this->id = $this->conn->insert_id;
            return true;
        }
        error_log("Support team creation failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Reads all support team records, with role-based filtering.
     *
     * @return mysqli_result|false The result set on success, false on failure.
     */
    public function readAll() {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        $query = "SELECT st.id, st.customer_id, st.engineer_id, st.champion_id, st.member_id,
                         uc.fullname AS customer_name, ue.fullname AS engineer_name,
                         uch.fullname AS champion_name, um.fullname AS member_name,
                         st.created_time, st.modified_time
                  FROM " . $this->table_name . " st
                  LEFT JOIN users uc ON st.customer_id = uc.id
                  LEFT JOIN users ue ON st.engineer_id = ue.id
                  LEFT JOIN users uch ON st.champion_id = uch.id
                  LEFT JOIN users um ON st.member_id = um.id";

        $whereClauses = [];
        $bindParams = [];
        $bindParamTypes = '';

        if (in_array($currentUserRole, ['superadmin', 'admin', 'supervisor'])) {
            // Superadmin, Admin, Supervisor can read all support teams
            // No additional WHERE clause needed
        } elseif (in_array($currentUserRole, ['localcustomer', 'globalcustomer'])) {
            // Customers can only read the team they are included in
            $whereClauses[] = "st.customer_id = ?";
            $bindParams[] = $currentUserId;
            $bindParamTypes .= 'i';
        } elseif (in_array($currentUserRole, ['engineer', 'champion', 'member'])) {
            // Engineers, champions, members can only read teams they are included in
            $whereClauses[] = "(st.engineer_id = ? OR st.champion_id = ? OR st.member_id = ?)";
            $bindParams[] = $currentUserId;
            $bindParams[] = $currentUserId;
            $bindParams[] = $currentUserId;
            $bindParamTypes .= 'iii';
        } else {
            sendJsonResponse(['message' => 'Unauthorized to read support teams.'], 403);
        }

        if (!empty($whereClauses)) {
            $query .= " WHERE " . implode(" AND ", $whereClauses);
        }

        $query .= " ORDER BY st.created_time DESC";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        if (!empty($bindParams)) {
            $stmt->bind_param($bindParamTypes, ...$bindParams);
        }

        if ($stmt->execute()) {
            return $stmt->get_result();
        }
        error_log("Read all support teams failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Reads a single support team record by ID.
     *
     * @return array|false The support team data as an associative array on success, false if not found.
     */
    public function readOne() {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        $query = "SELECT st.id, st.customer_id, st.engineer_id, st.champion_id, st.member_id,
                         uc.fullname AS customer_name, ue.fullname AS engineer_name,
                         uch.fullname AS champion_name, um.fullname AS member_name,
                         st.created_time, st.modified_time
                  FROM " . $this->table_name . " st
                  LEFT JOIN users uc ON st.customer_id = uc.id
                  LEFT JOIN users ue ON st.engineer_id = ue.id
                  LEFT JOIN users uch ON st.champion_id = uch.id
                  LEFT JOIN users um ON st.member_id = um.id
                  WHERE st.id = ? LIMIT 0,1";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $this->id = htmlspecialchars(strip_tags($this->id));
        $stmt->bind_param("i", $this->id);

        if ($stmt->execute()) {
            $result = $stmt->get_result();
            $team = $result->fetch_assoc();

            if ($team) {
                // Apply specific authorization for reading a single team
                if (in_array($currentUserRole, ['superadmin', 'admin', 'supervisor']) ||
                    $team['customer_id'] == $currentUserId ||
                    $team['engineer_id'] == $currentUserId ||
                    $team['champion_id'] == $currentUserId ||
                    $team['member_id'] == $currentUserId) {
                    return $team;
                } else {
                    sendJsonResponse(['message' => 'Unauthorized to view this support team.'], 403);
                    return false;
                }
            }
        }
        error_log("Read one support team failed or team not found: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Retrieves a support team by customer ID.
     * Used internally by cron jobs or other models.
     *
     * @param int $customerId The customer ID.
     * @return array|false The support team data as an associative array on success, false if not found.
     */
    public function getByCustomerId($customerId) {
        $query = "SELECT st.id, st.customer_id, st.engineer_id, st.champion_id, st.member_id
                  FROM " . $this->table_name . " st
                  WHERE st.customer_id = ? LIMIT 0,1";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $stmt->bind_param("i", $customerId);

        if ($stmt->execute()) {
            $result = $stmt->get_result();
            return $result->fetch_assoc();
        }
        error_log("Get support team by customer ID failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Updates an existing support team record.
     *
     * @return bool True on success, false on failure.
     */
    public function update() {
        $query = "UPDATE " . $this->table_name . "
                  SET customer_id=?, engineer_id=?, champion_id=?, member_id=?
                  WHERE id=?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize inputs
        $this->customer_id = htmlspecialchars(strip_tags($this->customer_id));
        $this->engineer_id = htmlspecialchars(strip_tags($this->engineer_id));
        $this->champion_id = htmlspecialchars(strip_tags($this->champion_id));
        $this->member_id = htmlspecialchars(strip_tags($this->member_id));
        $this->id = htmlspecialchars(strip_tags($this->id));

        $stmt->bind_param("iiiii", $this->customer_id, $this->engineer_id, $this->champion_id, $this->member_id, $this->id);

        if ($stmt->execute()) {
            return true;
        }
        error_log("Support team update failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Deletes a support team record by ID.
     *
     * @return bool True on success, false on failure.
     */
    public function delete() {
        $query = "DELETE FROM " . $this->table_name . " WHERE id = ?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $this->id = htmlspecialchars(strip_tags($this->id));
        $stmt->bind_param("i", $this->id);

        if ($stmt->execute()) {
            return true;
        }
        error_log("Support team deletion failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }
}
?>
