<?php
class User {
    private $conn;
    private $table_name = "users";

    // Object properties (match database columns)
    public $id;
    public $username;
    public $fullname;
    public $email;
    public $password; // Hashed password
    public $role;
    public $country_id;
    public $phone;
    public $address;
    public $is_active;
    public $created_at;
    public $updated_at;

    /**
     * Constructor for the User model.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
    }

    /**
     * Creates a new user record in the database.
     *
     * @return bool True on success, false on failure.
     */
    public function create() {
        // SQL query to insert a new user record
        $query = "INSERT INTO " . $this->table_name . "
                  SET username=?, fullname=?, email=?, password=?,
                      role=?, country_id=?, phone=?, address=?,is_active=?";

        // Prepare the statement
        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize input values to prevent XSS and other attacks
        $this->username = htmlspecialchars(strip_tags($this->username));
        $this->fullname = htmlspecialchars(strip_tags($this->fullname));
        $this->email = htmlspecialchars(strip_tags($this->email));
        // Password is hashed using a helper function defined in utils/helpers.php
        $hashedPassword = hashPassword($this->password);
        $this->role = htmlspecialchars(strip_tags($this->role));
        $this->country_id = htmlspecialchars(strip_tags($this->country_id));
        $this->phone = htmlspecialchars(strip_tags($this->phone));
        $this->address = htmlspecialchars(strip_tags($this->address));
        $active = true;

        // Bind parameters to the prepared statement
        // 'ssssssis' corresponds to string, string, string, string, string, integer, string, string
        $bind_success = $stmt->bind_param("sssssissi",
            $this->username,
            $this->fullname,
            $this->email,
            $hashedPassword, // Use the hashed password
            $this->role,
            $this->country_id,
            $this->phone,
            $this->address,
            $active
        );

        if ($bind_success === false) {
            error_log("Bind_param failed: (" . $stmt->errno . ") " . $stmt->error);
            return false;
        }

        // Execute the statement
        if ($stmt->execute()) {
            // Set the new user's ID
            $this->id = $this->conn->insert_id;
            return true;
        }

        error_log("User creation failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Reads all user records, joining with the country table to get country name.
     * Applies filtering based on user role if authenticated.
     *
     * @return mysqli_result|false The result set on success, false on failure.
     */
    public function readAll() {
    // Get the current user's role and ID for authorization checks
    $currentUserRole = getCurrentUserRole();
    $currentUserId = getCurrentUserId();

    // Base query to select user details and country name - REMOVE THE INITIAL WHERE
    $query = "SELECT u.id, u.username, u.fullname, u.email, u.role, u.phone, u.address,u.country_id,
              c.country_name, u.created_at, u.updated_at
              FROM " . $this->table_name . " u
              LEFT JOIN country c ON u.country_id = c.id"; // Removed 'WHERE u.is_active = 1' here

    $whereClauses = [];
    $bindParams = [];
    $bindParamTypes = '';

    // ALWAYS include the is_active condition
    $whereClauses[] = "u.is_active = 1";

    // Apply role-based filtering
    if ($currentUserRole) {
        switch ($currentUserRole) {
            case 'superadmin':
            case 'admin':
                // Superadmin and Admin can see all users, no extra WHERE clause needed beyond is_active
                break;
            case 'localcustomer':
            case 'globalcustomer':
                // Customers can only read their own profile and support team members
                $whereClauses[] = "(u.id = ? OR u.id IN (SELECT engineer_id FROM support_team WHERE customer_id = ?) OR u.id IN (SELECT champion_id FROM support_team WHERE customer_id = ?) OR u.id IN (SELECT member_id FROM support_team WHERE customer_id = ?))";
                $bindParams[] = $currentUserId;
                $bindParams[] = $currentUserId;
                $bindParams[] = $currentUserId;
                $bindParams[] = $currentUserId;
                $bindParamTypes .= 'iiii';
                break;
            case 'engineer':
            case 'champion':
            case 'member':
                // Engineers, champions, members can read their own profile and users in their support teams
                $whereClauses[] = "(u.id = ? OR u.id IN (SELECT customer_id FROM support_team WHERE engineer_id = ? OR champion_id = ? OR member_id = ?) OR u.role IN ('engineer', 'champion', 'member'))"; // They can also see other support staff
                $bindParams[] = $currentUserId;
                $bindParams[] = $currentUserId;
                $bindParams[] = $currentUserId;
                $bindParams[] = $currentUserId;
                $bindParamTypes .= 'iiii';
                break;
            case 'supervisor':
                // Supervisors can see all users, no extra WHERE clause needed (similar to admin)
                break;
            default:
                // Default for unknown roles or no role: only own profile
                $whereClauses[] = "u.id = ?";
                $bindParams[] = $currentUserId;
                $bindParamTypes .= 'i';
                break;
        }
    } else {
        // If no user is authenticated, no data should be returned (or only public data if any)
        sendJsonResponse(['message' => 'Unauthorized access.'], 401);
        return false; // Added return false to stop execution here
    }

    // Now, correctly build the WHERE clause for all conditions
    if (!empty($whereClauses)) {
        $query .= " WHERE " . implode(" AND ", $whereClauses);
    }

    $query .= " ORDER BY u.created_at DESC";

    $stmt = $this->conn->prepare($query);
    if ($stmt === false) {
        error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
        return false;
    }

    if (!empty($bindParams)) {
        // The splice for bind_param is removed here, as it's not standard
        // Instead, the ...$bindParams syntax correctly unpacks the array for call_user_func_array for mysqli_stmt::bind_param.
        // Or directly passing ...$bindParams in PHP 5.6+
        $stmt->bind_param($bindParamTypes, ...$bindParams);
    }

    if ($stmt->execute()) {
        return $stmt->get_result();
    }

    error_log("Read all users failed: (" . $stmt->errno . ") " . $stmt->error);
    return false;
}


    /**
     * Reads a single user record by ID.
     *
     * @return array|false The user data as an associative array on success, false on failure.
     */
    public function readOne() {
        // Get the current user's ID and role for authorization
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        // SQL query to select a single user record
        // Make sure u.country_id is explicitly selected
        $query = "SELECT u.id, u.username, u.fullname, u.email, u.role, u.phone, u.address,
                           u.country_id, c.country_name, u.created_at, u.updated_at
                  FROM " . $this->table_name . " u
                  LEFT JOIN country c ON u.country_id = c.id
                  WHERE u.id = ? LIMIT 0,1";

        // Prepare the statement
        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed (readOne user): (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Bind the user ID parameter
        $this->id = htmlspecialchars(strip_tags($this->id));
        $stmt->bind_param("i", $this->id);

        // Execute the statement
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            $row = $result->fetch_assoc();

            if ($row) {
                // Authorization check:
                // Superadmin/Admin/Supervisor can read any user.
                // Other roles can only read their own profile, or support team members if applicable.
                if (in_array($currentUserRole, ['superadmin', 'admin', 'supervisor']) ||
                    $currentUserId == $this->id) { // User can read their own profile
                    // Populate object properties
                    $this->username = $row['username'];
                    $this->fullname = $row['fullname'];
                    $this->email = $row['email'];
                    $this->role = $row['role'];
                    // Use null coalescing operator to safely access country_id and country_name
                    // This handles cases where country_id might be NULL in the users table,
                    // or if for some reason the key is truly missing (less likely but safe)
                    $this->country_id = $row['country_id'] ?? null;
                    $this->phone = $row['phone'];
                    $this->address = $row['address'];
                    $this->created_at = $row['created_at'];
                    $this->updated_at = $row['updated_at'];
                    $this->country_name = $row['country_name'] ?? null; // For display

                    return $row; // Return the full row
                } else {
                    // Check if the current user is part of the support team of the user being viewed
                    // This involves querying the support_team table
                    $supportTeamQuery = "SELECT COUNT(*) FROM support_team
                                         WHERE customer_id = ? AND (engineer_id = ? OR champion_id = ? OR member_id = ?)
                                         OR (engineer_id = ? AND customer_id = ?)
                                         OR (champion_id = ? AND customer_id = ?)
                                         OR (member_id = ? AND customer_id = ?)";
                    $teamStmt = $this->conn->prepare($supportTeamQuery);
                    if ($teamStmt === false) {
                        error_log("Support team check prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
                        return false;
                    }
                    // Prepare parameters for bind_param
                    $bindParamsTeam = [
                        $this->id, $currentUserId, $currentUserId, $currentUserId, // If $this->id is a customer and $currentUserId is part of their team
                        $currentUserId, $this->id, // If $currentUserId is an engineer and $this->id is their customer (changed order to match query logic)
                        $currentUserId, $this->id, // If $currentUserId is a champion and $this->id is their customer
                        $currentUserId, $this->id   // If $currentUserId is a member and $this->id is their customer
                    ];
                    // Bind by reference for mysqli_stmt_bind_param
                    $refBindParamsTeam = [];
                    foreach ($bindParamsTeam as $key => $value) {
                        $refBindParamsTeam[$key] = &$bindParamsTeam[$key];
                    }

                    // The 'iiiiiiiiii' type string is correct for 10 integers
                    call_user_func_array([$teamStmt, 'bind_param'], array_merge(['iiiiiiiiii'], $refBindParamsTeam));

                    if ($teamStmt->execute()) {
                        $teamResult = $teamStmt->get_result();
                        $teamCount = $teamResult->fetch_row()[0];

                        if ($teamCount > 0) {
                            // User is part of a support team relevant to the requested user
                            return $row;
                        } else {
                            // Unauthorized - return false to let the API endpoint handle the 403
                            // sendJsonResponse(['message' => 'You are not authorized to view this user\'s profile.'], 403);
                            return false;
                        }
                    } else {
                        error_log("Support team query execution failed: (" . $teamStmt->errno . ") " . $teamStmt->error);
                        return false;
                    }
                }
            }
        }

        error_log("Read one user failed or user not found: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Reads a single user record by ID (internal use, without authorization check).
     * Used by cron jobs or other models.
     *
     * @param int $userId The ID of the user to read.
     * @return array|false The user data as an associative array on success, false on failure.
     */
    public function readById($userId) {
        $query = "SELECT u.id, u.username, u.fullname, u.email, u.role, u.phone, u.address,
                         c.country_name,u.password, u.created_at, u.updated_at
                  FROM " . $this->table_name . " u
                  LEFT JOIN country c ON u.country_id = c.id
                  WHERE u.id = ? LIMIT 0,1";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $stmt->bind_param("i", $userId);
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            return $result->fetch_assoc();
        }
        error_log("readById failed for user ID $userId: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Updates an existing user record in the database.
     *
     * @return bool True on success, false on failure.
     */
    public function reviveUser($id =null, $activate = false) {
        if ($id) {
            $this->id = $id;
        }
        
        // Build the SET clause dynamically for optional updates
        $setClauses = [];
        $bindParams = [];
        $bindParamTypes = '';

        if (isset($this->username)) {
            $setClauses[] = "username=?";
            $bindParams[] = htmlspecialchars(strip_tags($this->username));
            $bindParamTypes .= 's';
        }
        if (isset($this->fullname)) {
            $setClauses[] = "fullname=?";
            $bindParams[] = htmlspecialchars(strip_tags($this->fullname));
            $bindParamTypes .= 's';
        }
        if (isset($this->email)) {
            $setClauses[] = "email=?";
            $bindParams[] = htmlspecialchars(strip_tags($this->email));
            $bindParamTypes .= 's';
        }
        // Only update password if it's explicitly set (e.g., during reactivation or profile update)
        if (isset($this->password) && !empty($this->password)) {
            $setClauses[] = "password=?";
            $bindParams[] = hashPassword($this->password); // Hash the password here
            $bindParamTypes .= 's';
        }
        if (isset($this->role)) {
            $setClauses[] = "role=?";
            $bindParams[] = htmlspecialchars(strip_tags($this->role));
            $bindParamTypes .= 's';
        }
        if (isset($this->phone)) {
            $setClauses[] = "phone=?";
            $bindParams[] = htmlspecialchars(strip_tags($this->phone));
            $bindParamTypes .= 's';
        }
        if (isset($this->address)) {
            $setClauses[] = "address=?";
            $bindParams[] = htmlspecialchars(strip_tags($this->address));
            $bindParamTypes .= 's';
        }
        if (isset($this->country_id)) {
            $setClauses[] = "country_id=?";
            $bindParams[] = htmlspecialchars(strip_tags($this->country_id));
            $bindParamTypes .= 'i'; // Assuming country_id is an integer
        }

        // Always update updated_at timestamp
        $setClauses[] = "updated_at=CURRENT_TIMESTAMP";

        // Set is_active if $activate is true
        if ($activate) {
            $setClauses[] = "is_active=?";
            $bindParams[] = 1;
            $bindParamTypes .= 'i';
        }

        if (empty($setClauses)) {
            error_log("No fields to update for user ID: " . $this->id);
            return false; // No fields provided for update
        }

        $query = "UPDATE " . $this->table_name . " SET " . implode(", ", $setClauses) . " WHERE id=?";
        $bindParams[] = $this->id; // Add ID to the end for the WHERE clause
        $bindParamTypes .= 'i'; // Add 'i' for the ID

        // Prepare the statement
        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed (update user): (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Bind parameters
        // Need to use call_user_func_array for dynamic bind_param
        $refBindParams = [];
        foreach ($bindParams as $key => $value) {
            $refBindParams[$key] = &$bindParams[$key];
        }

        if (!call_user_func_array([$stmt, 'bind_param'], array_merge([$bindParamTypes], $refBindParams))) {
            error_log("Bind_param failed (update user): (" . $stmt->errno . ") " . $stmt->error);
            return false;
        }

        // Execute the statement
        if ($stmt->execute()) {
            return true;
        }

        error_log("User update failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }
    
    public function update() {
        // SQL query to update a user record
        $query = "UPDATE " . $this->table_name . "
                  SET username=?, fullname=?,  email=?,role=?, phone=?, address=?, country_id=?, is_active = ?
                  WHERE id=?";

        // Prepare the statement
        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize input values
        $this->username = htmlspecialchars(strip_tags($this->username));
        $this->fullname = htmlspecialchars(strip_tags($this->fullname));
        $this->email = htmlspecialchars(strip_tags($this->email));
        $this->role = htmlspecialchars(strip_tags($this->role));
        $this->phone = htmlspecialchars(strip_tags($this->phone));
        $this->address = htmlspecialchars(strip_tags($this->address));
        $this->country_id = htmlspecialchars(strip_tags($this->country_id));
        $revive = true;
        $this->id = htmlspecialchars(strip_tags($this->id));

        // Bind parameters
        $bind_success = $stmt->bind_param("ssssssiii",
            $this->username,
            $this->fullname,
            $this->email,
            $this->role,
            $this->phone,
            $this->address,
            $this->country_id,
            $revive,
            $this->id
        );

        if ($bind_success === false) {
            error_log("Bind_param failed: (" . $stmt->errno . ") " . $stmt->error);
            return false;
        }

        // Execute the statement
        if ($stmt->execute()) {
            return true;
        }

        error_log("User update failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }
    
    public function updateProfile() {
        // SQL query to update a user record
        $query = "UPDATE " . $this->table_name . "
                  SET username=?, fullname=?,  email=?, phone=?, address=?, country_id=?, is_active = ?
                  WHERE id=?";

        // Prepare the statement
        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize input values
        $this->username = htmlspecialchars(strip_tags($this->username));
        $this->fullname = htmlspecialchars(strip_tags($this->fullname));
        $this->email = htmlspecialchars(strip_tags($this->email));
        $this->phone = htmlspecialchars(strip_tags($this->phone));
        $this->address = htmlspecialchars(strip_tags($this->address));
        $this->country_id = htmlspecialchars(strip_tags($this->country_id));
        $revive = true;
        $this->id = htmlspecialchars(strip_tags($this->id));

        // Bind parameters
        $bind_success = $stmt->bind_param("sssssiii",
            $this->username,
            $this->fullname,
            $this->email,
            $this->phone,
            $this->address,
            $this->country_id,
            $revive,
            $this->id
        );

        if ($bind_success === false) {
            error_log("Bind_param failed: (" . $stmt->errno . ") " . $stmt->error);
            return false;
        }

        // Execute the statement
        if ($stmt->execute()) {
            return true;
        }

        error_log("User update failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Updates the password for a given user ID.
     *
     * @param int $id The ID of the user whose password to update.
     * @param string $newPasswordHash The new hashed password.
     * @return bool True on success, false on failure.
     */
    public function updatePassword($id, $newPasswordHash) {
        $query = "UPDATE " . $this->table_name . " SET password = ? WHERE id = ?";
        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed for password update: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }
        $stmt->bind_param("si", $newPasswordHash, $id);
        if ($stmt->execute()) {
            return true;
        }
        error_log("Password update failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Deletes a user record from the database by ID.
     *
     * @return bool True on success, false on failure.
     */
    public function delete() {
        $query = "UPDATE " . $this->table_name . " SET is_active = ? WHERE id = ?";
        $stmt = $this->conn->prepare($query);
        
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize and bind the ID
        $remove = false;
        $stmt->bind_param("ii", $remove,$this->id);

        // Execute the statement
        if ($stmt->execute()) {
            return true;
        }

        error_log("User deletion failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Finds a user by their email address.
     *
     * @param string $email The email address to search for.
     * @return array|false The user data as an associative array on success, false if not found.
     */
    public function findByEmailorUsername($unique) {
        // SQL query to select user by email
        $query = "SELECT id, username, fullname, email, password, role, country_id,is_active FROM " . $this->table_name . " WHERE email = ? OR username = ? LIMIT 0,1";

        // Prepare the statement
        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Bind the email or username parameter
        $unique = htmlspecialchars(strip_tags($unique));
        $stmt->bind_param("ss", $unique, $unique);

        // Execute the statement
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            return $result->fetch_assoc(); // Return the associative array if found
        }

        error_log("Find user by email failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Retrieves users by a specific role.
     *
     * @param string $role The role to filter by (e.g., 'engineer', 'admin').
     * @return array An array of user data (associative arrays).
     */
    public function getUsersByRole($role) {
        $query = "SELECT id, username, fullname, email FROM " . $this->table_name . " WHERE role = ?";
        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return [];
        }
        $stmt->bind_param("s", $role);
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            $users = [];
            while ($row = $result->fetch_assoc()) {
                $users[] = $row;
            }
            return $users;
        }
        error_log("Get users by role failed: (" . $stmt->errno . ") " . $stmt->error);
        return [];
    }

    /**
     * Checks if a user with the given email already exists.
     *
     * @param string $email The email to check.
     * @param int|null $excludeId Optional. If provided, excludes this user ID from the check (for updates).
     * @return bool True if email exists, false otherwise.
     */
    public function emailExists($email, $excludeId = null) {
        $query = "SELECT COUNT(*) FROM " . $this->table_name . " WHERE email = ?";
        $params = [$email];
        $types = 's';

        if ($excludeId !== null) {
            $query .= " AND id != ?";
            $params[] = $excludeId;
            $types .= 'i';
        }

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return true; // Assume exists on error to prevent duplicates
        }

        $stmt->bind_param($types, ...$params);
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            $row = $result->fetch_row();
            return $row[0] > 0;
        }
        error_log("Email exists check failed: (" . $stmt->errno . ") " . $stmt->error);
        return true; // Assume exists on error
    }
    public function usernameExists($username, $excludeId = null) {
        $query = "SELECT COUNT(*) FROM " . $this->table_name . " WHERE username = ?";
        $params = [$username];
        $types = 's';

        if ($excludeId !== null) {
            $query .= " AND id != ?";
            $params[] = $excludeId;
            $types .= 'i';
        }

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return true; // Assume exists on error to prevent duplicates
        }

        $stmt->bind_param($types, ...$params);
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            $row = $result->fetch_row();
            return $row[0] > 0;
        }
        error_log("Username exists check failed: (" . $stmt->errno . ") " . $stmt->error);
        return true; // Assume exists on error
    }
    public function countryExists($country_id) {
        $query = "SELECT id FROM  country WHERE id = ? LIMIT 0,1"; // Assuming 'country' table
        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed (countryExists): (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }
        $stmt->bind_param("i", $country_id);
        $stmt->execute();
        $stmt->store_result();
        $num = $stmt->num_rows;
        $stmt->close();
        return $num > 0;
    }
}
?>
