<?php

require_once __DIR__ . '/../vendor/FPDF/fpdf.php';

class PdfGenerator extends FPDF {

    /**
     * Constructor for the PDF Generator.
     * Sets up basic PDF properties.
     */
    public function __construct($orientation='P', $unit='mm', $size='A4') {
        parent::__construct($orientation, $unit, $size);
        $this->SetAutoPageBreak(true, 15); // Auto page break with a bottom margin of 15mm

        // Removed AddFont calls for 'Inter' as 'Courier' is a default FPDF font.
        $this->SetFont('Courier', '', 12); // Set initial default font to Courier Regular, size 12
    }

    /**
     * Header of the PDF document.
     * Adds logo and title.
     */
    function Header() {
        // Logo (assuming you have a 'logo.png' in public/assets/)
        // Check if the logo file exists before attempting to use it
        $logoPath = __DIR__ . '/../assets/blue.png'; // Double check this path based on your file structure
        if (file_exists($logoPath)) {
            $this->Image($logoPath, 10, 8, 30);
        } else {
            // Fallback for logo not found, just print text
            $this->SetFont('Courier', 'B', 15); // Changed to Courier Bold
            $this->Cell(40);
            $this->Cell(0, 10, 'Rehlko Customer Care', 0, 0, 'C');
            $this->Ln(5);
        }

        // Title
        $this->SetFont('Courier', 'B', 15); // Changed to Courier Bold
        $this->Cell(0, 10, 'Report Details', 0, 1, 'C');
        $this->Ln(5);
    }

    /**
     * Footer of the PDF document.
     * Adds page number.
     */
    function Footer() {
        $this->SetY(-15); // Position at 1.5 cm from bottom
        $this->SetFont('Courier', '', 8); // Changed to Courier Regular, 8pt
        $this->Cell(0, 10, 'Page ' . $this->PageNo() . '/{nb}', 0, 0, 'C'); // Page number
    }

    /**
     * Generates a comprehensive report PDF.
     *
     * @param string $reportIndex The unique index of the report.
     * @param array $reportData An associative array containing all report details.
     * @return string|false Path to the generated PDF file on success, false on failure.
     */
    public function generateReportPdf($id,$reportIndex, $reportData) {
        $this->AddPage();
        $this->AliasNbPages(); // Required for {nb} alias in Footer

        $this->SetFont('Courier', 'B', 12); // Changed to Courier Bold
        $this->Cell(0, 10, 'Report Index: ' . $reportIndex, 0, 1);
        $this->Ln(5);

        // General Report Information
        $this->SetFont('Courier', 'B', 10); // Changed to Courier Bold
        $this->Cell(0, 7, 'General Information', 'B', 1);
        $this->SetFont('Courier', '', 10); // Changed to Courier Regular
        foreach ([
            'Report Type' => 'report_type',
            'Generator Serial Number' => 'generator_serial_number',
            'Customer Name' => 'customer_name',
            'Problem Issue' => 'problem_issue',
            'Running Hours' => 'running_hours',
            'Load Test' => 'load_test',
            'Load Hour' => 'load_hour',
            'Load Amount' => 'load_amount',
            'Used For' => 'used_for',
            'Error Code' => 'error_code',
            'Test Run' => 'test_run',
            'Location' => 'location',
            'Layer' => 'layer',
            'Remarks' => 'remarks',
            'Status' => 'status',
            'Engineer' => 'engineer_name',
            'Supervisor' => 'supervisor_name',
            'Created Date' => 'created_datetime'
        ] as $label => $key) {
            if (isset($reportData[$key])) {
                $value = $reportData[$key];
                if (in_array($key, ['load_test', 'test_run'])) {
                    $value = $value ? 'Yes' : 'No';
                }
                $this->Cell(60, 7, $label . ':', 0);
                $this->MultiCell(0, 7, $value, 0);
            }
        }
        $this->Ln(5);

        // Root Cause
        $this->SetFont('Courier', 'B', 10); // Changed to Courier Bold
        $this->Cell(0, 7, 'Root Cause', 'B', 1);
        $this->SetFont('Courier', '', 10); // Changed to Courier Regular
        if (!empty($reportData['root_cause'])) {
            foreach ($reportData['root_cause'] as $rc) {
                $this->MultiCell(0, 7, '• ' . $rc['root_cause_title'], 0);
                if (!empty($rc['root_cause_media_path']) && file_exists(__DIR__ . '/../public/uploads/' . $rc['root_cause_media_path'])) {
                   $this->Image(__DIR__ . '/../public/uploads/' . $rc['root_cause_media_path'], null, null, 50);
                   $this->Ln(5);
                }
            }
        } else {
            $this->Cell(0, 7, 'No root cause identified.', 0, 1);
        }
        $this->Ln(5);

        // Corrective Actions
        $this->SetFont('Courier', 'B', 10); // Changed to Courier Bold
        $this->Cell(0, 7, 'Corrective Actions', 'B', 1);
        $this->SetFont('Courier', '', 10); // Changed to Courier Regular
        if (!empty($reportData['corrective_action'])) {
            foreach ($reportData['corrective_action'] as $ca) {
                $this->MultiCell(0, 7, '• ' . $ca['corrective_action_title'], 0);
                 if (!empty($ca['corrective_action_media_path']) && file_exists(__DIR__ . '/../public/uploads/' . $ca['corrective_action_media_path'])) {
                   $this->Image(__DIR__ . '/../public/uploads/' . $ca['corrective_action_media_path'], null, null, 50);
                   $this->Ln(5);
                 }
            }
        } else {
            $this->Cell(0, 7, 'No corrective actions taken.', 0, 1);
        }
        $this->Ln(5);

        // Effective Actions
        $this->SetFont('Courier', 'B', 10); // Changed to Courier Bold
        $this->Cell(0, 7, 'Effective Actions', 'B', 1);
        $this->SetFont('Courier', '', 10); // Changed to Courier Regular
        if (!empty($reportData['effective_action'])) {
            foreach ($reportData['effective_action'] as $ea) {
                $this->MultiCell(0, 7, '• ' . $ea['effective_action_title'], 0);
            }
        } else {
            $this->Cell(0, 7, 'No effective actions recorded.', 0, 1);
        }
        $this->Ln(5);

        // Prevention
        $this->SetFont('Courier', 'B', 10); // Changed to Courier Bold
        $this->Cell(0, 7, 'Prevention', 'B', 1);
        $this->SetFont('Courier', '', 10); // Changed to Courier Regular
        if (!empty($reportData['prevention'])) {
            foreach ($reportData['prevention'] as $prev) {
                $this->MultiCell(0, 7, '• ' . $prev['prevention_title'], 0);
            }
        } else {
            $this->Cell(0, 7, 'No prevention steps recorded.', 0, 1);
        }
        $this->Ln(5);

        // Suggestions
        $this->SetFont('Courier', 'B', 10); // Changed to Courier Bold
        $this->Cell(0, 7, 'Suggestions', 'B', 1);
        $this->SetFont('Courier', '', 10); // Changed to Courier Regular
        if (!empty($reportData['suggestion'])) {
            foreach ($reportData['suggestion'] as $sugg) {
                $this->MultiCell(0, 7, '• ' . $sugg['suggestion_title'], 0);
            }
        } else {
            $this->Cell(0, 7, 'No suggestions recorded.', 0, 1);
        }
        $this->Ln(5);

        // Define output path for the PDF
        $outputDir = __DIR__ . '/../public/reports/';
        if (!is_dir($outputDir)) {
            mkdir($outputDir, 0777, true);
        }
        $filename = "report_{$reportIndex}_" . date('Ymd_His') . ".pdf";
        $outputPath = $outputDir . $filename;

        try {
            $this->Output('F', $outputPath);
            return $outputPath;
        } catch (Exception $e) {
            error_log("Failed to generate PDF: " . $e->getMessage());
            return false;
        }
    }
}