<?php
// controllers/CompanyController.php

/**
 * CompanyController for Rehlko Customer Care application.
 * Manages CRUD operations for company records.
 * Access is restricted to 'superadmin' and 'admin' roles for create, update, delete.
 * Read access can be broader if needed for forms (e.g., user registration).
 */
class CompanyController {
    private $conn;
    private $companyModel;

    /**
     * Constructor for CountryController.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
        $this->companyModel = new Company($db); // Instantiate Company model
    }

    /**
     * Creates a new company.
     * Only 'superadmin' and 'admin' roles are authorized.
     */
    public function create() {
        // Check authorization
        $currentUserRole = getCurrentUserRole();
        if (!in_array($currentUserRole, ['superadmin', 'admin'])) {
            sendJsonResponse(['message' => 'Unauthorized. Only superadmin or admin can create companies.'], 403);
        }

        $data = json_decode(file_get_contents("php://input"));

        // Validate required fields
        if (empty($data->company_name)) {
            sendJsonResponse(['message' => 'Company Name must be required.'], 400);
        }

        // Set country properties
        $this->companyModel->company_name = $data->company_name;
        $this->companyModel->company_address = $data->company_address;
        $this->companyModel->company_phone = $data->company_phone;
        $this->companyModel->company_email = $data->company_email;
        $this->companyModel->country_id = $data->country_id;

        if ($this->companyModel->create()) {
            sendJsonResponse(['message' => 'Company created successfully.', 'companyId' => $this->companyModel->id], 201);
        } else {
            sendJsonResponse(['message' => 'Failed to create country.'], 500);
        }
    }

    /**
     * Reads all countries.
     * All authenticated users can read countries (e.g., for dropdowns in user forms).
     */
    public function readAll() {
        // No specific role check here for readAll, as countries are often needed for dropdowns.
        // If strict read access is needed, add:
        // $currentUserRole = getCurrentUserRole();
        // if (!$currentUserRole) { sendJsonResponse(['message' => 'Authentication required.'], 401); }

        $result = $this->companyModel->readAll();

        if ($result) {
            $countries_arr = [];
            while ($row = $result->fetch_assoc()) {
                $countries_arr[] = $row;
            }
            sendJsonResponse(['data'=>$countries_arr]);
        } else {
            sendJsonResponse(['message' => 'Failed to retrieve countries.'], 500);
        }
    }

    /**
     * Reads a single country by ID.
     * All authenticated users can read a specific country.
     *
     * @param int $id The ID of the country to retrieve.
     */
    public function read($id) {
        // No specific role check here for readOne, as countries are often needed for dropdowns.
        // If strict read access is needed, add:
        // $currentUserRole = getCurrentUserRole();
        // if (!$currentUserRole) { sendJsonResponse(['message' => 'Authentication required.'], 401); }

        $this->companyModel->id = $id;
        $country = $this->companyModel->readOne();

        if ($country) {
            sendJsonResponse($country);
        } else {
            sendJsonResponse(['message' => 'Country not found.'], 404);
        }
    }

    /**
     * Updates an existing country.
     * Only 'superadmin' and 'admin' roles are authorized.
     *
     * @param int $id The ID of the country to update.
     */
    public function update($id) {
        // Check authorization
        $currentUserRole = getCurrentUserRole();
        if (!in_array($currentUserRole, ['superadmin', 'admin'])) {
            sendJsonResponse(['message' => 'Unauthorized. Only superadmin or admin can update countries.'], 403);
        }

        $data = json_decode(file_get_contents("php://input"));
        
        // Validate required fields
        if (empty($data->company_name)) {
            sendJsonResponse(['message' => 'Missing required fields.'], 400);
        }

        // Set country properties
        $this->companyModel->company_name = $data->company_name;
        $this->companyModel->company_address = $data->company_address;
        $this->companyModel->company_phone = $data->company_phone;
        $this->companyModel->company_email = $data->company_email;
        $this->companyModel->country_id = $data->country_id;
        $this->companyModel->id = $id;

        if ($this->companyModel->update()) {
            sendJsonResponse(['message' => 'Company updated successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to update country.'], 500);
        }
    }

    /**
     * Deletes a country.
     * Only 'superadmin' and 'admin' roles are authorized.
     *
     * @param int $id The ID of the country to delete.
     */
    public function delete($id) {
        // Check authorization
        $currentUserRole = getCurrentUserRole();
        if (!in_array($currentUserRole, ['superadmin', 'admin'])) {
            sendJsonResponse(['message' => 'Unauthorized. Only superadmin or admin can delete countries.'], 403);
        }

        $this->companyModel->id = $id;

        if ($this->companyModel->delete()) {
            sendJsonResponse(['message' => 'Company deleted successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to delete country or country not found.'], 500);
        }
    }
}
?>
