<?php
// controllers/CorrectiveActionController.php

/**
 * CorrectiveActionController for Rehlko Customer Care application.
 * Manages CRUD operations for corrective action entries linked to reports.
 * Only engineers assigned to a report can perform CRUD on its corrective actions.
 */
class CorrectiveActionController {
    private $conn;
    private $correctiveActionModel;
    private $reportModel; // To validate report_index and engineer assignment

    /**
     * Constructor for CorrectiveActionController.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
        $this->correctiveActionModel = new CorrectiveAction($db);
        $this->reportModel = new Report($db);
    }

    /**
     * Creates a new corrective action for a report.
     * Only engineers assigned to the report can create.
     * Expects multipart/form-data for media upload.
     */
    public function create() {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        $authorized = false;
        
        // Use $_POST for text fields and $_FILES for file uploads
        $report_index = $_POST['report_index'] ?? null;
        $corrective_action_title = $_POST['corrective_action_title'] ?? null;
        $media_type = null;
        $media_path = null;
        $media_name = null;

        // Validate required fields
        if (empty($report_index) || empty($corrective_action_title)) {
            sendJsonResponse(['message' => 'Missing required fields: report_index, corrective_action_title.'], 400);
        }
        if (in_array($currentUserRole, ['superadmin', 'admin'])) {
            $report = $this->reportModel->readOne($report_index, 'index');
            $authorized = true;
        } else if($currentUserRole === 'engineer') {
            $report = $this->reportModel->readOneEngineer($report_index, 'index');
            error_log("hahah");
             $authorized = true;
        }
        

        if (!$authorized) {
            sendJsonResponse(['message' => 'Unauthorized. Only engineers can create root causes.'], 403);
        }
        if (empty($report)) { // Changed from ! to empty() for robustness
            
            sendJsonResponse(['message' => 'Report not found or invalid report_index.'], 404);
        }

        // Validate report_index and engineer assignment
        
        
        if ($report['engineer_id'] != $currentUserId && !in_array($currentUserRole, ['superadmin', 'admin'])) {
            sendJsonResponse(['message' => 'Unauthorized. You are not assigned to this report.'], 403);
        }

        // Handle file upload (images and video)
        $uploadDir = 'uploads/corrective_actions/';
        if (!is_dir(__DIR__ . '/../public/' . $uploadDir)) {
            mkdir(__DIR__ . '/../public/' . $uploadDir, 0777, true);
        }

        if (isset($_FILES['media']) && is_array($_FILES['media']['name'])) { // Multiple images
            $uploadedFiles = [];
            foreach ($_FILES['media']['name'] as $key => $name) {
                if ($_FILES['media']['error'][$key] == UPLOAD_ERR_OK) {
                    $fileTmpPath = $_FILES['media']['tmp_name'][$key];
                    $fileExtension = pathinfo($name, PATHINFO_EXTENSION);
                    $newFileName = uniqid() . '.' . $fileExtension;
                    $destPath = __DIR__ . '/../public/' . $uploadDir . $newFileName;

                    if (move_uploaded_file($fileTmpPath, $destPath)) {
                        $uploadedFiles[] = [
                            'path' => $uploadDir . $newFileName,
                            'name' => $name,
                            'type' => 'image',
                        ];
                    } else {
                        sendJsonResponse(['message' => 'Failed to move uploaded image file.'], 500);
                    }
                }
            }
            if (!empty($uploadedFiles)) {
                // For simplicity, let's store the first image info as the primary media.
                // A more robust solution would involve a separate media table.
                $media_path = $uploadedFiles[0]['path'];
                $media_name = $uploadedFiles[0]['name'];
                $media_type = 'image';
            }
        } elseif (isset($_FILES['media_video']) && $_FILES['media_video']['error'] == UPLOAD_ERR_OK) { // Single video
            $fileTmpPath = $_FILES['media_video']['tmp_name'];
            $fileName = $_FILES['media_video']['name'];
            $fileExtension = pathinfo($fileName, PATHINFO_EXTENSION);
            $newFileName = uniqid() . '.' . $fileExtension;
            $destPath = __DIR__ . '/../public/' . $uploadDir . $newFileName;

            if (move_uploaded_file($fileTmpPath, $destPath)) {
                $media_path = $uploadDir . $newFileName;
                $media_name = $fileName;
                $media_type = 'video';
            } else {
                sendJsonResponse(['message' => 'Failed to move uploaded video file.'], 500);
            }
        }

        $this->correctiveActionModel->report_index = $report_index;
        $this->correctiveActionModel->corrective_action_title = $corrective_action_title;
        $this->correctiveActionModel->corrective_action_media_type = $media_type;
        $this->correctiveActionModel->corrective_action_media_path = $media_path;
        $this->correctiveActionModel->corrective_action_media_name = $media_name;
        $this->correctiveActionModel->created_user = $currentUserId; // Assuming currentUserId is available

        if ($this->correctiveActionModel->create()) {
            sendJsonResponse(['success' => true,'message' => 'Corrective action created successfully.', 'id' => $this->correctiveActionModel->id], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to create corrective action.'], 500);
        }
    }

    /**
     * Updates an existing corrective action.
     * Only the engineer assigned to the report can update, or superadmin/admin.
     * Expects multipart/form-data for media upload.
     *
     * @param int $id The ID of the corrective action to update.
     */
    public function update($id) {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        // Fetch the existing corrective action to get its report_index and current media info
        $this->correctiveActionModel->id = $id;
        $existingCorrectiveAction = $this->correctiveActionModel->readOne();
        if (!$existingCorrectiveAction) {
            sendJsonResponse(['message' => 'Corrective action not found.'], 404);
        }

        // Authorize: Superadmin/Admin or the assigned engineer
        $report_index_to_check = $_POST['report_index'] ?? $existingCorrectiveAction['report_index'];
        
        
        
        $authorized = false;
        if (in_array($currentUserRole, ['superadmin', 'admin'])) {
            $report = $this->reportModel->readOne($report_index_to_check, 'index');
            $authorized = true;
        } else if($currentUserRole === 'engineer' && $report['engineer_id'] == $currentUserId) {
            $report = $this->reportModel->readOneEngineer($report_index_to_check, 'index');
             $authorized = true;
        }
        if (empty($report)) { // Changed from ! to empty() for robustness
            sendJsonResponse(['message' => 'Report not found or invalid report_index.'], 404);
        }

        if (!$authorized) {
            sendJsonResponse(['message' => 'Unauthorized. Only engineers can create root causes.'], 403);
        }
        
        // Use $_POST for text fields and $_FILES for file uploads
        $this->correctiveActionModel->id = $id;
        $this->correctiveActionModel->report_index = $_POST['report_index'] ?? $existingCorrectiveAction['report_index'];
        $this->correctiveActionModel->corrective_action_title = $_POST['corrective_action_title'] ?? $existingCorrectiveAction['corrective_action_title'];
        
        // Preserve existing media info by default
        $media_type = $existingCorrectiveAction['corrective_action_media_type'];
        $media_path = $existingCorrectiveAction['corrective_action_media_path'];
        $media_name = $existingCorrectiveAction['corrective_action_media_name'];

        // Check for media clearing request
        $clear_media = $_POST['clear_media'] ?? null;
        if ($clear_media === '1') {
            // Delete old file if it exists
            if ($existingCorrectiveAction['corrective_action_media_path'] &&
                file_exists(__DIR__ . '/../public/' . $existingCorrectiveAction['corrective_action_media_path'])) {
                unlink(__DIR__ . '/../public/' . $existingCorrectiveAction['corrective_action_media_path']);
            }
            $media_type = null;
            $media_path = null;
            $media_name = null;
        }

        // Handle new file upload (images and video)
        $uploadDir = 'uploads/corrective_actions/';
        if (!is_dir(__DIR__ . '/../public/' . $uploadDir)) {
            mkdir(__DIR__ . '/../public/' . $uploadDir, 0777, true);
        }

        // Prioritize new uploads over existing media or clear request
        $newMediaUploaded = false;
        if (isset($_FILES['media']) && is_array($_FILES['media']['name']) && !empty($_FILES['media']['name'][0])) { // Multiple images
            // Delete old file if a new one is being uploaded
            if ($existingCorrectiveAction['corrective_action_media_path'] &&
                file_exists(__DIR__ . '/../public/' . $existingCorrectiveAction['corrective_action_media_path'])) {
                unlink(__DIR__ . '/../public/' . $existingCorrectiveAction['corrective_action_media_path']);
            }

            $uploadedFiles = [];
            foreach ($_FILES['media']['name'] as $key => $name) {
                if ($_FILES['media']['error'][$key] == UPLOAD_ERR_OK) {
                    $fileTmpPath = $_FILES['media']['tmp_name'][$key];
                    $fileExtension = pathinfo($name, PATHINFO_EXTENSION);
                    $newFileName = uniqid() . '.' . $fileExtension;
                    $destPath = __DIR__ . '/../public/' . $uploadDir . $newFileName;

                    if (move_uploaded_file($fileTmpPath, $destPath)) {
                        $uploadedFiles[] = [
                            'path' => $uploadDir . $newFileName,
                            'name' => $name,
                            'type' => 'image',
                        ];
                    } else {
                        sendJsonResponse(['message' => 'Failed to move uploaded image file.'], 500);
                    }
                }
            }
            if (!empty($uploadedFiles)) {
                $media_path = $uploadedFiles[0]['path'];
                $media_name = $uploadedFiles[0]['name'];
                $media_type = 'image';
                $newMediaUploaded = true;
            }
        } elseif (isset($_FILES['media_video']) && $_FILES['media_video']['error'] == UPLOAD_ERR_OK) { // Single video
            // Delete old file if a new one is being uploaded
            if ($existingCorrectiveAction['corrective_action_media_path'] &&
                file_exists(__DIR__ . '/../public/' . $existingCorrectiveAction['corrective_action_media_path'])) {
                unlink(__DIR__ . '/../public/' . $existingCorrectiveAction['corrective_action_media_path']);
            }

            $fileTmpPath = $_FILES['media_video']['tmp_name'];
            $fileName = $_FILES['media_video']['name'];
            $fileExtension = pathinfo($fileName, PATHINFO_EXTENSION);
            $newFileName = uniqid() . '.' . $fileExtension;
            $destPath = __DIR__ . '/../public/' . $uploadDir . $newFileName;

            if (move_uploaded_file($fileTmpPath, $destPath)) {
                $media_path = $uploadDir . $newFileName;
                $media_name = $fileName;
                $media_type = 'video';
                $newMediaUploaded = true;
            } else {
                sendJsonResponse(['message' => 'Failed to move uploaded video file.'], 500);
            }
        }

        // If no new media uploaded and clear_media was not '1', retain original media info
        if (!$newMediaUploaded && $clear_media !== '1') {
            $this->correctiveActionModel->corrective_action_media_type = $existingCorrectiveAction['corrective_action_media_type'];
            $this->correctiveActionModel->corrective_action_media_path = $existingCorrectiveAction['corrective_action_media_path'];
            $this->correctiveActionModel->corrective_action_media_name = $existingCorrectiveAction['corrective_action_media_name'];
        } else {
            $this->correctiveActionModel->corrective_action_media_type = $media_type;
            $this->correctiveActionModel->corrective_action_media_path = $media_path;
            $this->correctiveActionModel->corrective_action_media_name = $media_name;
        }

        $this->correctiveActionModel->modified_user = $currentUserId; // Assuming currentUserId is available

        if ($this->correctiveActionModel->update()) {
            sendJsonResponse(['success' => true,'message' => 'Corrective action updated successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to update corrective action.'], 500);
        }
    }

    /**
     * Reads a single corrective action by its ID.
     *
     * @param int $id The ID of the corrective action to retrieve.
     */
    public function readOne($id) {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        $this->correctiveActionModel->id = $id;
        $correctiveAction = $this->correctiveActionModel->readOne();

        if (empty($correctiveAction)) { // Changed from ! to empty() for robustness
            sendJsonResponse(['message' => 'Corrective action not found.'], 404);
        }

        // Authorize: Superadmin/Admin or the assigned engineer to the report
        $report = $this->reportModel->readOne($correctiveAction['report_index'], 'index');
        if (empty($report)) { // Changed from ! to empty() for robustness
            sendJsonResponse(['message' => 'Report not found or invalid report_index for authorization.'], 404);
        }

        $authorized = false;
        if (in_array($currentUserRole, ['superadmin', 'admin'])) {
            $authorized = true;
        } else if ($report['engineer_id'] == $currentUserId) { // Removed '&& $report' as it's already checked by empty()
            $authorized = true;
        }

        if (!$authorized) {
            sendJsonResponse(['message' => 'Unauthorized to view this corrective action.'], 403);
        }

        sendJsonResponse(['message' => 'Corrective action retrieved.', 'data' => $correctiveAction], 200);
    }

    /**
     * Reads all corrective actions for a specific report_index.
     * Only accessible by superadmin/admin or the assigned engineer.
     *
     * @param int $report_index The index of the report.
     */
    public function readAllByReportIndex($report_index) {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        // Validate report_index and engineer assignment
        $report = $this->reportModel->readOne($report_index, 'index');
        
        

        $authorized = false;
        if (in_array($currentUserRole, ['superadmin', 'admin'])) {
            $authorized = true;
        } else if ($currentUserRole === 'engineer') {
            $report = $this->reportModel->readOneEngineer($report_index, 'index');
             $authorized = true;
        } else if (in_array($currentUserRole, ['localcustomer', 'globalcustomer'])&& $report['customer_id'] == $currentUserId) { // Allow customer to view their reports' root causes
             $authorized = true;
        }

        if (!$authorized) {
            sendJsonResponse(['message' => 'Unauthorized to view corrective actions for this report.'], 403);
        }
        if (empty($report)) { // Changed from ! to empty() for robustness
            sendJsonResponse(['message' => 'Report not found or invalid report_index.'], 404);
        }

        $stmt = $this->correctiveActionModel->readAllByReportIndex($report_index);
        $num = $stmt->num_rows;

        $corrective_actions_arr = [];
        $corrective_actions_arr['data'] = [];

        if ($num > 0) {
            while ($row = $stmt->fetch_assoc()) {
                extract($row);
                $corrective_action_item = [
                    'id' => $id,
                    'report_index' => $report_index,
                    'corrective_action_title' => $corrective_action_title,
                    'corrective_action_media_type' => $corrective_action_media_type,
                    'corrective_action_media_path' => $corrective_action_media_path,
                    'corrective_action_media_name' => $corrective_action_media_name,
                    'created_user' => $created_user,
                    'created_datetime' => $created_datetime,
                    'modified_datetime' => $modified_datetime,
                ];
                array_push($corrective_actions_arr['data'], $corrective_action_item);
            }
            sendJsonResponse(['message' => 'Corrective actions retrieved.', 'success' => true, 'data' => $corrective_actions_arr['data']], 200);
        } else {
            sendJsonResponse(['message' => 'No corrective actions found for this report.', 'success' => true, 'data' => []], 200);
        }
    }

    /**
     * Deletes a corrective action.
     * Only the engineer assigned to the report can delete, or superadmin/admin.
     *
     * @param int $id The ID of the corrective action to delete.
     */
    public function delete($id) {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        // Fetch the corrective action to get its report_index
        $this->correctiveActionModel->id = $id;
        $correctiveAction = $this->correctiveActionModel->readOne();
        if (!$correctiveAction) {
            sendJsonResponse(['message' => 'Corrective action not found.'], 404);
        }

        // Authorize: Superadmin/Admin or the assigned engineer
        $authorized = false;
        if (in_array($currentUserRole, ['superadmin', 'admin'])) {
            $authorized = true;
        } else {
            $report = $this->reportModel->readOne($correctiveAction['report_index'], 'index');
            if (empty($report)) { // Changed from ! to empty() for robustness
                sendJsonResponse(['message' => 'Report not found or invalid report_index for authorization.'], 404);
            }
            if ($report['engineer_id'] == $currentUserId) { // Removed '&& $report' as it's already checked by empty()
                 $authorized = true;
            }
        }

        if (!$authorized) {
            sendJsonResponse(['message' => 'Unauthorized to delete this corrective action.'], 403);
        }

        if ($this->correctiveActionModel->delete($id)) {
            sendJsonResponse(['success' => true,'message' => 'Corrective action deleted successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to delete corrective action.'], 500);
        }
    }
}
?>