<?php
// controllers/CustomerProductController.php

/**
 * CustomerProductController for Rehlko Customer Care application.
 * Manages operations related to assigning products to customers,
 * including auto-creation of maintenance records.
 * Access is restricted for assign, update, delete to 'superadmin', 'supervisor', 'admin'.
 * Read access is role-based and handled by the model.
 */
class CustomerProductController {
    private $conn;
    private $customerProductModel;
    private $productModel;// To verify product existence
    private $companyModel;// To verify company existence
    private $assignEngineerModel;
    private $warningConfigModel;

    /**
     * Constructor for CustomerProductController.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
        $this->customerProductModel = new CustomerProduct($db); // Instantiate CustomerProduct model
        $this->productModel = new Product($db); // For validation
        $this->companyModel = new Company($db); // For validation
        $this->assignEngineerModel = new AssignEngineer($db); 
        $this->warningConfigModel = new ReportWarningConfig($db);
    }   

    /**
     * Assigns a product to a customer, triggering auto-creation of maintenance records.
     * Only 'superadmin', 'supervisor', 'admin' roles are authorized.
     */
    public function assignProduct() {
        // Check authorization
        $currentUserRole = getCurrentUserRole();
        if (!in_array($currentUserRole, ['superadmin', 'supervisor', 'admin'])) {
            sendJsonResponse(['message' => 'Unauthorized. Only superadmin, supervisor, or admin can assign products.'], 403);
        }

        $data = json_decode(file_get_contents("php://input"));
       
        // Validate required fields
        if (empty($data->generator_serial_number) || empty($data->customer_id) || empty($data->level) || empty($data->gantry)) {
            sendJsonResponse(['message' => 'Missing required fields: generator_serial_number, customer_id, level, gantry.'], 400);
        }

        // Validate if generator_serial_number exists in products table
        if (!$this->productModel->readBySerialNumber($data->generator_serial_number)) {
            sendJsonResponse(['message' => 'Generator serial number does not exist in products.'], 404);
        }

        // Validate if customer_id exists and is a customer role
        $customerUser = $this->companyModel->readOne($data->customer_id);
       

        // Set properties for assignment
        $this->customerProductModel->generator_serial_number = $data->generator_serial_number;
        $this->customerProductModel->customer_id = $data->customer_id;
        $this->customerProductModel->level = $data->level;
        $this->customerProductModel->gantry = $data->gantry;

        if ($this->customerProductModel->assignProduct()) {
            $this->productModel->assignProduct($this->customerProductModel->generator_serial_number );
            sendJsonResponse([
        'message' => 'Product assigned to customer and maintenance auto-created successfully.',
        'assignmentId' => $this->customerProductModel->id,
        'generator_serial_number' => $this->customerProductModel->generator_serial_number // Add this line to return the generator serial number
    ], 201);
        } else {
            sendJsonResponse(['message' => 'Failed to assign product to customer.'], 500);
        }
    }

    public function readAll() {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        // For customer products, filterStatus and searchTerm might apply to the product itself
        // or the assignment status, if such fields exist in customer_products table.
        // For now, I'll keep them as parameters for consistency, but they might be unused
        // if your CustomerProduct model's read methods don't support them.
        $filterStatus = $_GET['status'] ?? null;
        $searchTerm = $_GET['search'] ?? null;

        $customerProducts = [];

        // Determine which customer products the current user can see based on their role
        if (in_array($currentUserRole, ['superadmin', 'admin', 'supervisor'])) {
            // Superadmins, Admins, Supervisors can see all customer products
            $customerProducts = $this->customerProductModel->readAll(); // This method should fetch all
        } elseif (in_array($currentUserRole, ['localcustomer', 'globalcustomer'])) {
            // Customers can only see their own assigned products
            $customerProducts = $this->customerProductModel->readByCustomer($currentUserId);
        } elseif (in_array($currentUserRole, ['engineer', 'champion', 'member'])) {
            $assignedEngineer = $this->assignEngineerModel->readEngineerByCurrentDate();
            
            if ($assignedEngineer) {
                if($assignedEngineer['engineer_id'] == $currentUserId){
                    error_log("Heheh");
                    $customerProducts = $this->customerProductModel->readAll();
                }
            }
            
        } else {
            sendJsonResponse(['message' => 'Unauthorized to view customer products.'], 403);
            return; // Exit to prevent further execution
        }

        // Check if the result from the model is valid (not false, which indicates an error)
        if ($customerProducts !== false) {
            $customer_products_arr = [];
            // If the model methods return a mysqli_result object
            if ($customerProducts instanceof mysqli_result) {
                while ($row = $customerProducts->fetch_assoc()) {
                    $customer_products_arr[] = $row;
                }
            } elseif (is_array($customerProducts)) {
                // If the model methods directly return an array of data
                $customer_products_arr = $customerProducts;
            }

            sendJsonResponse(['data' => $customer_products_arr, 'success' => true]);
        } else {
            // If the model method returned false due to an internal error
            sendJsonResponse(['message' => 'Failed to retrieve customer products.'], 500);
        }
    }

    /**
     * Reads a single customer product by ID.
     * Access is restricted based on user role and support team relationships, handled by the model.
     *
     * @param int $id The ID of the customer product assignment to retrieve.
     */
    public function read($id) {
        $currentUserRole = getCurrentUserRole();
        if (!$currentUserRole) {
            sendJsonResponse(['message' => 'Authentication required.'], 401);
        }

        $this->customerProductModel->id = $id;
        $customerProduct = $this->customerProductModel->readOne(); // Authorization logic is inside readOne model method

        if ($customerProduct) {
            sendJsonResponse(['data'=>$customerProduct]);
        } else {
            sendJsonResponse(['message' => 'Customer product not found or unauthorized.'], 404);
        }
    }

    /**
     * Updates an existing customer product assignment.
     * Only 'superadmin', 'supervisor', 'admin' roles are authorized.
     * Note: This assumes only 'level' and 'gantry' can be updated, not the product or customer.
     * If product/customer can be reassigned, the auto-maintenance logic needs careful adjustment.
     *
     * @param int $id The ID of the customer product assignment to update.
     */
    public function update($id) {
        // Check authorization
        $currentUserRole = getCurrentUserRole();
        if (!in_array($currentUserRole, ['superadmin', 'supervisor', 'admin'])) {
            sendJsonResponse(['message' => 'Unauthorized. Only superadmin, supervisor, or admin can update customer products.'], 403);
        }

        $data = json_decode(file_get_contents("php://input"));

        // Validate required fields
        if (empty($data->generator_serial_number) || empty($data->customer_id) || empty($data->level) || empty($data->gantry)) {
            sendJsonResponse(['message' => 'Missing required fields: generator_serial_number, customer_id, level, gantry.'], 400);
        }

        // Check if assignment exists
        $this->customerProductModel->id = $id;
        if (!$this->customerProductModel->readOne()) {
            sendJsonResponse(['message' => 'Customer product assignment not found.'], 404);
        }

        // Set properties for update
        $this->customerProductModel->generator_serial_number = $data->generator_serial_number;
        $this->customerProductModel->customer_id = $data->customer_id;
        $this->customerProductModel->level = $data->level;
        $this->customerProductModel->gantry = $data->gantry;
        $this->customerProductModel->id = $id;

        if ($this->customerProductModel->update()) {
            sendJsonResponse(['message' => 'Customer product assignment updated successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to update customer product assignment.'], 500);
        }
    }

    /**
     * Deletes a customer product assignment.
     * Only 'superadmin', 'supervisor', 'admin' roles are authorized.
     * Note: This does not delete related maintenance records automatically.
     * If maintenance should be cascaded, add logic here or in DB foreign key.
     *
     * @param int $id The ID of the customer product assignment to delete.
     */
    public function delete($id) {
        // Check authorization
        $currentUserRole = getCurrentUserRole();
        if (!in_array($currentUserRole, ['superadmin', 'supervisor', 'admin'])) {
            sendJsonResponse(['message' => 'Unauthorized. Only superadmin, supervisor, or admin can delete customer products.'], 403);
        }

        $this->customerProductModel->id = $id;

        if ($this->customerProductModel->delete()) {
            sendJsonResponse(['message' => 'Customer product assignment deleted successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to delete customer product assignment or not found.'], 500);
        }
    }
}
?>
