<?php
// controllers/MaintenanceController.php

/**
 * MaintenanceController for Rehlko Customer Care application.
 * Manages maintenance record operations.
 * Update access for all roles except customers.
 * Read access is role-based and handled by the model.
 */
class MaintenanceController {
    private $conn;
    private $maintenanceModel;
    private $maintenancePlanModel;
    private $companyModel;
    private $assignEngineerModel;
    private $notificationModel;
    private $userModel;
    private $pdfGenerator; // NEW: Add PdfGenerator property
    private $customerProductModel; // NEW: To get customer for a generator

    /**
     * Constructor for MaintenanceController.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
        $this->maintenanceModel = new Maintenance($db);
        $this->maintenancePlanModel = new MaintenancePlan($db);
        $this->assignEngineerModel = new AssignEngineer($db);
        $this->companyModel = new Company($db);
        $this->notificationModel = new Notification($db);
        $this->userModel = new User($db);
        $this->pdfGenerator = new PdfGenerator(); // NEW: Instantiate PdfGenerator
        $this->customerProductModel = new CustomerProduct($db); // NEW: Instantiate CustomerProductModel
    }

    /**
     * Creates a new maintenance record.
     * Only roles other than 'localcustomer' and 'globalcustomer' are authorized.
     */
    public function create() {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        if (in_array($currentUserRole, ['localcustomer', 'globalcustomer'])) {
            sendJsonResponse(['message' => 'Unauthorized. Customers cannot create maintenance records.'], 403);
            return;
        }

        $data = json_decode(file_get_contents("php://input"));

        if (empty($data->generator_serial_number) || empty($data->maintenance_type) || empty($data->plan_id) || empty($data->maintenance_title) || empty($data->maintenance_description)) {
            sendJsonResponse(['message' => 'Missing required fields (generator_serial_number, maintenance_title, maintenance_description, maintenance_type, or plan_id).'], 400);
            return;
        }

        $successCount = 0;
        $errors = [];

        $individualDescriptions = preg_split("/\r?\n/", $data->maintenance_description);

        if (empty($individualDescriptions)) {
            sendJsonResponse(['message' => 'No valid maintenance descriptions found to create records.'], 400);
            return;
        }
       // error_log("Individual Des: " . print_r($individualDescriptions, true));
        foreach ($individualDescriptions as $descriptionLine) {
            $this->maintenanceModel->generator_serial_number = $data->generator_serial_number;
            $this->maintenanceModel->maintenance_plan_id = $data->plan_id;
            $this->maintenanceModel->maintenance_title = $data->maintenance_title;
            $this->maintenanceModel->maintenance_description = trim($descriptionLine);
            $this->maintenanceModel->maintenance_type = $data->maintenance_type;
            $this->maintenanceModel->maintenance_check_user = $currentUserId;
            $this->maintenanceModel->maintenance_date = date('Y-m-d H:i:s'); // Corrected property name

            if ($this->maintenanceModel->create()) {
                $successCount++;
            } else {
                $errors[] = 'Failed to create record for description: ' . trim($descriptionLine);
            }
        }

        if ($successCount > 0) {
            $message = ($successCount == 1)
                ? 'Maintenance record created successfully.'
                : "$successCount maintenance records created successfully.";

            sendJsonResponse([
                'success' => true,
                'message' => $message,
                'maintenance_plan_id' => $data->plan_id,
                'errors' => $errors
            ], 201);
        } else {
            sendJsonResponse(['message' => 'Failed to create any maintenance records.', 'errors' => $errors], 500);
        }
    }

    /**
     * Reads all maintenance records.
     * Access is restricted based on user role and team assignments, handled by the model.
     */
    public function readAll() {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();
        if (!$currentUserRole) {
            sendJsonResponse(['message' => 'Authentication required.'], 401);
            return;
        }
        $customerId = isset($_GET['customer_id']) ? $_GET['customer_id'] : null;

        $maintenance = false; // Initialize as false to correctly handle query failures

        if (in_array($currentUserRole, ['superadmin', 'admin', 'supervisor'])) {
            $maintenance = $this->maintenancePlanModel->readAll($customerId);
        } elseif(in_array($currentUserRole, ['localcustomer', 'globalcustomer'])){
            $maintenance = $this->maintenancePlanModel->CustomerreadAll($currentUserId);
        } elseif (in_array($currentUserRole, ['engineer', 'champion', 'member'])) {
            $assignedEngineer = $this->assignEngineerModel->readEngineerByCurrentDate();
            if ($assignedEngineer) {
                $engineer = $this->userModel->readById($assignedEngineer['engineer_id']);
                if($engineer['id'] == $currentUserId){ // Check array value, not the entire array
                    $maintenance = $this->maintenancePlanModel->readAll($customerId);
                }
            }
        }

        // Check for a query failure first
        if ($maintenance === false) {
            sendJsonResponse(['message' => 'Unauthorized or failed to retrieve Maintenance.'], 403);
            return;
        }

        // Now, you can safely check the number of rows because $maintenance is a valid mysqli_result object.
        if ($maintenance->num_rows === 0) {
            sendJsonResponse(['success' => true, 'data' => [], 'message' => 'No maintenance records found.'], 200);
            return;
        }

        $maintenances_arr = [];
        while ($row = $maintenance->fetch_assoc()) {
            $maintenances_arr[] = $row;
        }

        sendJsonResponse(['success' => true, 'data' => $maintenances_arr], 200);
    }

    public function DetailsreadAll($plan_id) {
        $currentUserRole = getCurrentUserRole();
        if (!$currentUserRole) {
            sendJsonResponse(['message' => 'Authentication required.'], 401);
            return;
        }

        $result = $this->maintenanceModel->readAllByID($plan_id);
       // error_log($plan_id);
        if ($result) {
            $maintenance_arr = [];
            while ($row = $result->fetch_assoc()) {
                 // Fetch the username for maintenance_check_user
                if (!empty($row['maintenance_check_user'])) {
                    $checkedByUser = $this->userModel->readById($row['maintenance_check_user']);
                    $row['checked_by_user_name'] = $checkedByUser['fullname'] ?? 'N/A';
                } else {
                    $row['checked_by_user_name'] = 'N/A';
                }
                $maintenance_arr[] = $row;
            }
            //error_log(print_r($maintenances_arr,true));
            sendJsonResponse(['success' => true, 'data'=>$maintenance_arr], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to retrieve maintenances or unauthorized.'], 403);
        }
    }

    /**
     * Reads a single maintenance record by ID.
     * Access is restricted based on user role and team assignments, handled by the model.
     *
     * @param int $id The ID of the maintenance record to retrieve.
     */
    public function read($id) {
        $currentUserRole = getCurrentUserRole();
        if (!$currentUserRole) {
            sendJsonResponse(['message' => 'Authentication required.'], 401);
            return;
        }

        $this->maintenanceModel->id = $id;
        $maintenance = $this->maintenanceModel->readOne();

        if ($maintenance) {
            sendJsonResponse(['success' => true, 'data'=>$maintenance], 200);
        } else {
            sendJsonResponse(['message' => 'Maintenance record not found or unauthorized.'], 404);
        }
    }

    /**
     * Updates an existing maintenance record.
     * All roles except 'localcustomer' and 'globalcustomer' are authorized.
     * After update, notifies the product owner (customer).
     *
     * @param int $id The ID of the maintenance record to update.
     */
    public function update($id) {
        $currentUserRole = getCurrentUserRole();
        if (in_array($currentUserRole, ['localcustomer', 'globalcustomer'])) {
            sendJsonResponse(['message' => 'Unauthorized. Customers cannot update maintenance records.'], 403);
            return;
        }

        $data = json_decode(file_get_contents("php://input"));

        $this->maintenanceModel->id = $id;
        $existingMaintenance = $this->maintenanceModel->readOne();
        if (!$existingMaintenance) {
            sendJsonResponse(['message' => 'Maintenance record not found or unauthorized.'], 404);
            return;
        }

        $this->maintenanceModel->is_check = $data->is_check;
        $this->maintenanceModel->id = $id;

        if ($this->maintenanceModel->update()) {
            sendJsonResponse(['success' => true, 'message' => 'Maintenance record updated successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to update maintenance record.'], 500);
        }
    }

    public function updatecheck($id) {
        $currentUserRole = getCurrentUserRole();
        if (in_array($currentUserRole, ['localcustomer', 'globalcustomer'])) {
            sendJsonResponse(['message' => 'Unauthorized. Customers cannot update maintenance records.'], 403);
            return;
        }

        $data = json_decode(file_get_contents("php://input"));

       // error_log(print_r($data,true));
    }

    /**
     * Deletes a maintenance record.
     * All roles except 'localcustomer' and 'globalcustomer' are authorized.
     *
     * @param int $id The ID of the maintenance record to delete.
     */
    public function delete($id) {
        $currentUserRole = getCurrentUserRole();
        if (in_array($currentUserRole, ['localcustomer', 'globalcustomer'])) {
            sendJsonResponse(['message' => 'Unauthorized. Customers cannot delete maintenance records.'], 403);
            return;
        }

        $this->maintenanceModel->id = $id;

        if ($this->maintenanceModel->delete()) {
            sendJsonResponse(['success' => true, 'message' => 'Maintenance record deleted successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to delete maintenance record or not found.'], 500);
        }
    }

    /**
     * Creates a new maintenance plan.
     * Only roles other than 'localcustomer' and 'globalcustomer' are authorized.
     */
    public function createPlan() {
    $currentUserRole = getCurrentUserRole();
    if (in_array($currentUserRole, ['localcustomer', 'globalcustomer'])) {
        sendJsonResponse(['message' => 'Unauthorized. Customers cannot create maintenance plans.'], 403);
        return;
    }

    $data = json_decode(file_get_contents("php://input"));

    if (empty($data->generator_serial_number)) {
        sendJsonResponse(['message' => 'Missing required field: generator_serial_number.'], 400);
        return;
    }
    $customer = $this->customerProductModel->getByGeneratorSerialNumber($data->generator_serial_number);
    $customerId = $customer['customer_id'];
    // Add a check for the new required field
    if (empty($data->maintenance_end_date)) {
        sendJsonResponse(['message' => 'Missing required field: maintenance_end_date.'], 400);
        return;
    }

    $this->maintenancePlanModel->generator_serial_number = $data->generator_serial_number;
    $this->maintenancePlanModel->monthly = $data->monthly ?? false;
    $this->maintenancePlanModel->quarterly = $data->quarterly ?? false;
    $this->maintenancePlanModel->annually = $data->annually ?? false;
    // Assign the new field to the model
    $this->maintenancePlanModel->maintenance_end_date = $data->maintenance_end_date;
    $this->maintenancePlanModel->customer_id = $customerId;

    if ($this->maintenancePlanModel->create()) {
        sendJsonResponse(['success' => true, 'message' => 'Maintenance plan created successfully.', 'id' => $this->maintenancePlanModel->id], 201);
    } else {
        sendJsonResponse(['message' => 'Failed to create maintenance plan.'], 500);
    }
}

    /**
     * Retrieves the monthly, quarterly, and annually flags for a given maintenance plan ID.
     * Accessible by authorized users.
     *
     * @param int $planId The ID of the maintenance plan.
     */
    public function getPlanTypes($planId) {
        $currentUserRole = getCurrentUserRole();
        if (!$currentUserRole) {
            sendJsonResponse(['message' => 'Authentication required.'], 401);
            return;
        }

        $this->maintenancePlanModel->id = $planId;
        $plan = $this->maintenancePlanModel->readOne();

        if ($plan) {
            sendJsonResponse([
                'success' => true,
                'monthly' => (bool)$plan['monthly'],
                'quarterly' => (bool)$plan['quarterly'],
                'annually' => (bool)$plan['annually']
            ], 200);
        } else {
            sendJsonResponse(['message' => 'Maintenance plan not found.'], 404);
        }
    }

    /**
     * NEW METHOD: Generates a PDF for a specific maintenance plan.
     * @param int $planId The ID of the maintenance plan.
     */
    public function generatePdf($planId) {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        // Authorization check
        if (!in_array($currentUserRole, ['superadmin', 'admin', 'supervisor', 'engineer', 'localcustomer', 'globalcustomer'])) {
            sendJsonResponse(['message' => 'Unauthorized to generate PDF for this maintenance plan.'], 403);
            return;
        }
        
        // Fetch maintenance plan details
        $this->maintenancePlanModel->id = $planId;
        $maintenancePlan = $this->maintenancePlanModel->readOne();
        $type = $_GET['type'];

        if (!$maintenancePlan) {
            sendJsonResponse(['message' => 'Maintenance plan not found.'], 404);
            return;
        }

        // Fetch all maintenance records associated with this plan
        $maintenanceRecordsResult = $this->maintenanceModel->readAllByIDandType($planId,$type);
        $maintenanceRecords = [];
        if ($maintenanceRecordsResult) {
            while ($row = $maintenanceRecordsResult->fetch_assoc()) {
                // Fetch the fullname of the user who checked this maintenance
                if (!empty($row['maintenance_check_user'])) {
                    $checkedByUser = $this->userModel->readById($row['maintenance_check_user']);
                    $row['checked_by_user_name'] = $checkedByUser['fullname'] ?? 'N/A';
                } else {
                    $row['checked_by_user_name'] = 'N/A';
                }
                $maintenanceRecords[] = $row;
            }
        }

        // Fetch details of the user generating the report
        $engineerUser = $this->userModel->readById($currentUserId);
        $engineerFullName = $engineerUser['fullname'] ?? 'N/A';

        // Fetch customer name associated with the generator serial number
        $customerProduct = $this->customerProductModel->getByGeneratorSerialNumber($maintenancePlan['generator_serial_number']);
        $customerFullName = 'N/A';
        if ($customerProduct) {
            $this->companyModel->id=$customerProduct['customer_id'];
            $customer = $this->companyModel->readOne();
            $customerFullName = $customer['company_name'] ?? 'N/A';
        }


        $pdfData = [
            'maintenance_plan' => $maintenancePlan,
            'maintenance_records' => $maintenanceRecords,
            'engineer_fullname' => $engineerFullName,
            'customer_fullname' => $customerFullName,
            'generator_serial_number' => $maintenancePlan['generator_serial_number']
        ];
        
        $pdfPath = $this->pdfGenerator->generateMaintenancePdf($pdfData);

        if ($pdfPath && file_exists($pdfPath)) {
            header('Content-Type: application/pdf');
            header('Content-Disposition: attachment; filename="' . basename($pdfPath) . '"');
            readfile($pdfPath);
            exit();
        } else {
            sendJsonResponse(['message' => 'Maintenance PDF file not found or could not be generated.'], 500);
        }
    }
}
