<?php
// controllers/NotificationController.php

/**
 * NotificationController for Rehlko Customer Care application.
 * Manages operations related to in-app notifications for users.
 */
class NotificationController {
    private $conn;
    private $notificationModel;

    /**
     * Constructor for NotificationController.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
        $this->notificationModel = new Notification($db);
    }

    /**
     * Reads all notifications for the current authenticated user.
     *
     * @param int $userId The ID of the user whose notifications to retrieve.
     */
    public function readUserNotifications($userId) {
        $currentUserId = getCurrentUserId();
        // A user can only read their own notifications
        if ($currentUserId != $userId) {
            sendJsonResponse(['message' => 'Unauthorized. You can only view your own notifications.'], 403);
        }

        $result = $this->notificationModel->readUserNotifications($userId);

        if ($result) {
            $notifications_arr = [];
            while ($row = $result->fetch_assoc()) {
                $notifications_arr[] = $row;
            }
            sendJsonResponse(['data'=>$notifications_arr]);
        } else {
            sendJsonResponse(['message' => 'Failed to retrieve notifications.'], 500);
        }
    }

    /**
     * Marks a specific notification as read.
     * Only the owner of the notification can mark it as read.
     *
     * @param int $id The ID of the notification to mark as read.
     */
    public function markAsRead($id) {
        $currentUserId = getCurrentUserId();

        // Fetch the notification to verify ownership
        $this->notificationModel->id = $id; // Set ID for internal read
        $notification = $this->notificationModel->readOne(); // Requires a readOne method in Notification model

        if (!$notification) {
            sendJsonResponse(['message' => 'Notification not found.'], 404);
        }
        if ($notification['receiver_id'] != $currentUserId) {
            sendJsonResponse(['message' => 'Unauthorized. You cannot mark other users\' notifications as read.'], 403);
        }

        if ($this->notificationModel->markAsRead($id)) {
            sendJsonResponse(['message' => 'Notification marked as read successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to mark notification as read.'], 500);
        }
    }

    /**
     * Deletes a specific notification.
     * Only the owner of the notification can delete it, or superadmin/admin.
     *
     * @param int $id The ID of the notification to delete.
     */
    public function delete($id) {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        // Fetch the notification to verify ownership
        $this->notificationModel->id = $id;
        $notification = $this->notificationModel->readOne();

        if (!$notification) {
            sendJsonResponse(['message' => 'Notification not found.'], 404);
        }

        $authorized = false;
        if (in_array($currentUserRole, ['superadmin', 'admin'])) {
            $authorized = true;
        } elseif ($notification['receiver_id'] == $currentUserId) {
            $authorized = true;
        }

        if (!$authorized) {
            sendJsonResponse(['message' => 'Unauthorized to delete this notification.'], 403);
        }

        if ($this->notificationModel->delete()) {
            sendJsonResponse(['message' => 'Notification deleted successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to delete notification.'], 500);
        }
    }
}
?>
