<?php
// controllers/ProductController.php

/**
 * ProductController for Rehlko Customer Care application.
 * Manages CRUD operations for product records.
 * Access is restricted for create, update, delete to 'superadmin', 'supervisor', 'admin'.
 * Read access is role-based and handled by the model.
 */
class ProductController {
    private $conn;
    private $productModel;

    /**
     * Constructor for ProductController.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
        $this->productModel = new Product($db); // Instantiate Product model
    }

    /**
     * Creates a new product.
     * Only 'superadmin', 'supervisor', 'admin' roles are authorized.
     */
    public function create() {
        // Check authorization
        $currentUserRole = getCurrentUserRole();
        if (!in_array($currentUserRole, ['superadmin', 'supervisor', 'admin'])) {
            sendJsonResponse(['message' => 'Unauthorized. Only superadmin, supervisor, or admin can create products.'], 403);
        }

        $data = json_decode(file_get_contents("php://input"));

        // Validate required fields
        if (empty($data->generator_serial_number) || empty($data->generator_model_number)) {
            sendJsonResponse(['message' => 'Missing required fields: generator_serial_number, generator_model_number.'], 400);
        }
        error_log($data->engine_serial_number);
        error_log($data->engine_model_number);
        // Set product properties
        $this->productModel->generator_serial_number = $data->generator_serial_number;
        $this->productModel->generator_model_number = $data->generator_model_number;
        $this->productModel->enigne_serial_number = $data->engine_serial_number ?? null;
        $this->productModel->enigne_model_number = $data->engine_model_number ?? null;
        $this->productModel->alt_serial_number = $data->alt_serial_number ?? null;
        $this->productModel->alt_model_number = $data->alt_model_number ?? null;
        $this->productModel->controller_parts = $data->controller_parts ?? null;
        $this->productModel->controller_serial = $data->controller_serial_number ?? null;
        $this->productModel->remarks = $data->remarks ?? null;
            
            error_log( $this->productModel->enigne_serial_number);
        error_log($this->productModel->enigne_model_number);
        if ($this->productModel->create()) {
            sendJsonResponse(['message' => 'Product created successfully.', 'productId' => $this->productModel->id], 201);
        } else {
            sendJsonResponse(['message' => 'Failed to create product.'], 500);
        }
    }

    /**
     * Reads all products.
     * Access is restricted based on user role and team assignments, handled by the model.
     */
    public function readAll() {
        $currentUserRole = getCurrentUserRole();
        if (!$currentUserRole) {
            sendJsonResponse(['message' => 'Authentication required.'], 401);
        }

        // Authorization is handled within the Product model's readAll method
        $result = $this->productModel->readAll();

        if ($result && $result->num_rows > 0) { // Check for a valid result and if rows exist
            $products_arr = [];
            while ($row = $result->fetch_assoc()) {
                $products_arr[] = $row;
            }
            sendJsonResponse(['message' => 'Products fetched successfully.', 'data' => $products_arr], 200);
        } elseif ($result && $result->num_rows === 0) { // If query was successful but no rows
            sendJsonResponse(['message' => 'No products found.', 'data' => []], 200);
        } else { // If $result is false (query failed or unauthorized by model)
            sendJsonResponse(['message' => 'Failed to retrieve products or unauthorized.'], 403);
        }
    }
    
    public function UnassignreadAll() {
        $currentUserRole = getCurrentUserRole();
        if (!$currentUserRole) {
            sendJsonResponse(['message' => 'Authentication required.'], 401);
        }

        // Authorization is handled within the Product model's readAll method
        $result = $this->productModel->UnassignreadAll();

        if ($result && $result->num_rows > 0) { // Check for a valid result and if rows exist
            $products_arr = [];
            while ($row = $result->fetch_assoc()) {
                $products_arr[] = $row;
            }
            sendJsonResponse(['message' => 'Unassigned Products fetched successfully.', 'data' => $products_arr], 200);
        } elseif ($result && $result->num_rows === 0) { // If query was successful but no rows
            sendJsonResponse(['message' => 'All products are assigned.Please create new produt to assign.', 'data' => []], 200);
        } else { // If $result is false (query failed or unauthorized by model)
            sendJsonResponse(['message' => 'Failed to retrieve products or unauthorized.'], 403);
        }
    }

    /**
     * Reads a single product by ID.
     * Access is restricted based on user role and team assignments, handled by the model.
     *
     * @param int $id The ID of the product to retrieve.
     */
    public function read($id) {
        $currentUserRole = getCurrentUserRole();
        if (!$currentUserRole) {
            sendJsonResponse(['message' => 'Authentication required.'], 401);
        }

        $this->productModel->id = $id;
        $product = $this->productModel->readOne(); // Authorization logic is inside readOne model method

        if ($product) {
            sendJsonResponse(['message' => 'Product fetched successfully.', 'data' => $product], 200); // Wrap in 'data' key
        } else {
            // readOne will send 403 response if unauthorized, or return false if not found.
            sendJsonResponse(['message' => 'Product not found or unauthorized.'], 404);
        }
    }

    /**
     * Updates an existing product.
     * Only 'superadmin', 'supervisor', 'admin' roles are authorized.
     *
     * @param int $id The ID of the product to update.
     */
    public function update($id) {
        // Check authorization
        $currentUserRole = getCurrentUserRole();
        if (!in_array($currentUserRole, ['superadmin', 'supervisor', 'admin'])) {
            sendJsonResponse(['message' => 'Unauthorized. Only superadmin, supervisor, or admin can update products.'], 403);
        }

        $data = json_decode(file_get_contents("php://input"));

        // Basic validation
        if (empty($data->generator_serial_number) || empty($data->generator_model_number)) {
            sendJsonResponse(['message' => 'Missing required fields.'], 400);
        }

        // Check if product exists
        $this->productModel->id = $id;
        // The readOne in Product model should populate properties if found, or return false if not found/unauthorized
        $existingProduct = $this->productModel->readOne();
        if (!$existingProduct) {
             sendJsonResponse(['message' => 'Product not found or unauthorized to update.'], 404);
        }

        // Set properties for update
        $this->productModel->generator_serial_number = $data->generator_serial_number;
        $this->productModel->generator_model_number = $data->generator_model_number;
        $this->productModel->enigne_serial_number = $data->engine_serial_number ?? null;
        $this->productModel->enigne_model_number = $data->engine_model_number ?? null;
        $this->productModel->alt_serial_number = $data->alt_serial_number ?? null;
        $this->productModel->alt_model_number = $data->alt_model_number ?? null;
        $this->productModel->controller_parts = $data->controller_parts ?? null;
        $this->productModel->controller_serial = $data->controller_serial_number ?? null;
        $this->productModel->remarks = $data->remarks ?? null;
        $this->productModel->id = $id; // Ensure ID is set for the update query

        if ($this->productModel->update()) {
            sendJsonResponse(['message' => 'Product updated successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to update product.'], 500);
        }
    }

    /**
     * Deletes a product.
     * Only 'superadmin', 'supervisor', 'admin' roles are authorized.
     *
     * @param int $id The ID of the product to delete.
     */
    public function delete($id) {
        // Check authorization
        $currentUserRole = getCurrentUserRole();
        if (!in_array($currentUserRole, ['superadmin', 'supervisor', 'admin'])) {
            sendJsonResponse(['message' => 'Unauthorized. Only superadmin, supervisor, or admin can delete products.'], 403);
        }

        $this->productModel->id = $id;

        if ($this->productModel->delete()) {
            sendJsonResponse(['message' => 'Product deleted successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to delete product or product not found.'], 500);
        }
    }
}
