<?php
// controllers/ReportController.php

require_once __DIR__ . '/../utils/fcm_helper.php';
// All other required files like Report.php, User.php etc.


/**
 * ReportController for Rehlko Customer Care application.
 * Manages report lifecycle: creation, picking up, solving, checking, and completing.
 * Handles role-based access control for each action.
 */
class ReportController {
    private $conn;
    private $reportModel;
    private $userModel;
    private $companyModel;
    private $assignEngineerModel; // Changed from supportTeamModel
    private $notificationModel;
    private $reportMediaModel;
    private $mailer;
    private $pdfGenerator;
    private $fcmHelper;
    private $rootCauseModel;
    private $correctiveActionModel;
    private $effectiveActionModel;
    private $preventionModel;
    private $suggestionModel;

    /**
     * Constructor for ReportController.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
        $this->reportModel = new Report($db);
        $this->userModel = new User($db);
        $this->companyModel = new Company($db);
        $this->assignEngineerModel = new AssignEngineer($db); // Changed to AssignEngineer
        $this->notificationModel = new Notification($db);
        $this->mailer = new Mailer();
        $this->pdfGenerator = new PdfGenerator();
        $this->reportMediaModel = new ReportMedia($db);
        $this->fcmHelper = new FcmHelper();
        $this->rootCauseModel = new RootCause($db);
        $this->correctiveActionModel = new CorrectiveAction($db);
        $this->effectiveActionModel = new EffectiveAction($db);
        $this->preventionModel = new Prevention($db);
        $this->suggestionModel = new Suggestion($db);
    }

    /**
     * Creates a new report.
     * Only 'localcustomer' and 'globalcustomer' roles are authorized.
     */
    public function create(): void
    {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        if (!in_array($currentUserRole, ['localcustomer', 'globalcustomer'])) {
            sendJsonResponse(['message' => 'Unauthorized. Only customers can create reports.'], 403);
            return;
        }

        $postData = $_POST;
        $filesData = $_FILES;
        
        if (empty($postData['problem_issue'])) {
            sendJsonResponse(['message' => 'Missing required fields: problem_issue.'], 400);
            return;
        }

        $this->reportModel->customer_id = $currentUserId;
        $this->reportModel->report_index = $this->reportModel->generateUniqueReportIndex($currentUserId);
        $this->reportModel->problem_issue = $postData['problem_issue'];
        $this->reportModel->status = 'pending';

        // Set optional fields
        $this->reportModel->report_type = $postData['report_type'] ?? 'normal';
        $this->reportModel->generator_serial_number = $postData['generator_serial_number'] ?? null;
        $this->reportModel->used_for = $postData['used_for'] ?? 'stand by';
        $this->reportModel->location = $postData['location'] ?? null;
        $this->reportModel->layer = $postData['layer'] ?? null;
        $this->reportModel->remarks = $postData['remarks'] ?? null;
        $company=$this->companyModel->readByCustomer($currentUserId);
        
        $this->reportModel->company_id = $company['company_id'] ?? null;
        //error_log(print_r($filesData,true));
        error_log(print_r($this->reportModel->company_id,true));
        if ($this->reportModel->create()) {
            $createdReportId = $this->reportModel->id;
            $createdReportIndex = $this->reportModel->report_index;
            $uploadedImageUrls = [];
            $uploadedVideoUrl = null;
            
            // Handle file uploads
            $this->processFileUploads($createdReportIndex, $filesData, $uploadedImageUrls, $uploadedVideoUrl);

            // Send notifications to the customer and the assigned engineer
            $this->sendNewReportNotifications($createdReportIndex, $currentUserId, $postData['problem_issue']);

            // --- FCM Notification to all admins/supervisors ---
            $fcmTitle = "New Report Created: " . $createdReportIndex;
            $fcmBody = "A new report has been submitted by a customer.";
            $fcmTopic = "admin_notifications"; // Your target topic
            $fcmData = ["report_id" => $createdReportId];
            $this->fcmHelper->sendNotification($fcmTitle, $fcmBody, $fcmTopic, $fcmData);
            // ---------------------------------------------------

            sendJsonResponse([
                'message' => 'Report created successfully.',
                'report_index' => $createdReportIndex,
                'image_paths' => $uploadedImageUrls,
                'video_path' => $uploadedVideoUrl
            ], 201);
        } else {
            sendJsonResponse(['message' => 'Failed to create report.'], 500);
        }
    }

    /**
     * Allows an engineer to pick up a report.
     * Only 'engineer', 'admin', 'superadmin', 'supervisor' roles are authorized.
     * @param int $id The ID of the report.
     */
    public function pickupReport($id) {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        if (!in_array($currentUserRole, ['engineer', 'admin', 'superadmin', 'supervisor'])) {
            sendJsonResponse(['message' => 'Unauthorized.'], 403);
            return;
        }
            //error_log($id);
        $this->reportModel->id = $id;
        $report = $this->reportModel->readOneEngineer($this->reportModel->id, 'id');

        if (!$report) {
            sendJsonResponse(['message' => 'Report not found.'], 404);
            return;
        }

        if ($report['status'] !== 'pending') {
            sendJsonResponse(['message' => 'Report cannot be picked up. Current status is ' . $report['status'] . '.'], 400);
            return;
        }

        if ($this->reportModel->updateStatus($id, 'solving', $currentUserId, null, 1)) {
            $customer = $this->userModel->readById($report['customer_id']);
            $engineer = $this->userModel->readById($currentUserId);
            if ($customer && $engineer) {
                $this->notificationModel->createNotification(
                    $customer['id'],
                    'Report Picked Up',
                    'Your report (' . $report['report_index'] . ') has been picked up by ' . $engineer['fullname'] . '.',
                    'reports'
                );
            }
            sendJsonResponse(['message' => 'Report picked up successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to pick up report.'], 500);
        }
    }

    /**
     * Allows an engineer to solve a report.
     * Only 'engineer', 'admin', 'superadmin', 'supervisor' roles are authorized.
     * @param int $id The ID of the report.
     */
    public function solve($id) {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        if (!in_array($currentUserRole, ['engineer', 'admin', 'superadmin', 'supervisor'])) {
            sendJsonResponse(['message' => 'Unauthorized.'], 403);
            return;
        }

        $this->reportModel->id = $id;
        if (in_array($currentUserRole, ['superadmin', 'admin'])) {
            $report = $this->reportModel->readOne($this->reportModel->id, 'id');
            //$authorized = true;
        } else if($currentUserRole === 'engineer') {
            $report = $this->reportModel->readOneEngineer($this->reportModel->id, 'id');
             //$authorized = true;
        }
        

        if (!$report) {
            sendJsonResponse(['message' => 'Report not found.'], 404);
            return;
        }

        if ($report['status'] !== 'solving') {
            sendJsonResponse(['message' => 'Report can only be marked as solved when its status is "solving". Current status: ' . $report['status'] . '.'], 400);
            return;
        }
        
        if ($this->reportModel->updateStatus($id, 'checking', $report['engineer_id'], null, null)) {
            $customer = $this->userModel->readById($report['customer_id']);
            $engineer = $this->userModel->readById($currentUserId);
            if ($customer && $engineer) {
                $this->notificationModel->createNotification(
                    $customer['id'],
                    'Report Solved',
                    'Your report (' . $report['report_index'] . ') has been marked as solved by ' . $engineer['fullname'] . '. Please review the solution.',
                    'reports'
                );
            }
            sendJsonResponse(['message' => 'Report solved successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to solve report.'], 500);
        }
    }

    /**
     * Allows a customer to check a solved report.
     * Only 'localcustomer' and 'globalcustomer' roles are authorized.
     * @param int $id The ID of the report.
     */
    public function checkReport($id) {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();
        
        if (!in_array($currentUserRole, ['localcustomer', 'globalcustomer'])) {
            sendJsonResponse(['message' => 'Unauthorized. Only customers can check reports.'], 403);
            return;
        }
        
        $this->reportModel->id = $id;
        $report = $this->reportModel->readOne($id, 'id');
        
        if (!$report) {
            sendJsonResponse(['message' => 'Report not found.'], 404);
            return;
        }
        
        if ($report['customer_id'] != $currentUserId) {
            sendJsonResponse(['message' => 'Unauthorized. You can only check your own reports.'], 403);
            return;
        }
        
        if ($report['status'] !== 'checking') {
            sendJsonResponse(['message' => 'Report cannot be checked. Current status is ' . $report['status'] . '.'], 400);
            return;
        }
        
        sendJsonResponse(['message' => 'Report checked successfully.'], 200);
    }
    
    /**
     * Allows an engineer, supervisor, or admin to reject a report after a customer rejects it.
     * Only 'engineer', 'supervisor', 'admin', 'superadmin' roles are authorized.
     * @param int $id The ID of the report.
     */
    public function reject($id) {
        $currentUserRole = getCurrentUserRole();
        
        if (!in_array($currentUserRole, ['engineer', 'supervisor', 'admin', 'superadmin'])) {
            sendJsonResponse(['message' => 'Unauthorized. Only engineers, supervisors, admins, or superadmins can reject reports.'], 403);
            return;
        }
        
        $data = json_decode(file_get_contents("php://input"));
        
        if (empty($data->rejection_reason)) {
            sendJsonResponse(['message' => 'Rejection reason is required.'], 400);
            return;
        }
        
        $this->reportModel->id = $id;
        $report = $this->reportModel->readOne($id, 'id');
        
        if (!$report) {
            sendJsonResponse(['message' => 'Report not found.'], 404);
            return;
        }
        
        if (!in_array($report['status'], ['solving', 'checking'])) {
            sendJsonResponse(['message' => 'Report cannot be rejected. Current status is ' . $report['status'] . '.'], 400);
            return;
        }
        
        if ($this->reportModel->updateStatus($id, 'pending', null, null, null, $data->rejection_reason)) {
            $customer = $this->userModel->readById($report['customer_id']);
            $engineer = $this->userModel->readById($report['engineer_id']);
            if ($customer) {
                $this->notificationModel->createNotification(
                    $customer['id'],
                    'Report Rejected',
                    'Your report (' . $report['report_index'] . ') was rejected with the reason: ' . $data->rejection_reason,
                    'reports'
                );
            }
            if ($engineer) {
                $this->notificationModel->createNotification(
                    $engineer['id'],
                    'Report Rejected',
                    'Your report (' . $report['report_index'] . ') was rejected with the reason: ' . $data->rejection_reason,
                    'reports'
                );
            }
            
            sendJsonResponse(['message' => 'Report rejected and status reset to pending.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to reject report.'], 500);
        }
    }

    /**
     * Allows a supervisor or admin to complete a checked report.
     * Only 'supervisor', 'admin', 'superadmin' roles are authorized.
     * @param int $id The ID of the report.
     */
    public function complete($id) {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();
        if (!in_array($currentUserRole, ['supervisor', 'admin', 'superadmin'])) {
            sendJsonResponse(['message' => 'Unauthorized.'], 403);
            return;
        }

        $this->reportModel->id = $id;
        $report = $this->reportModel->readOne($this->reportModel->id, 'id');

        if (!$report) {
            sendJsonResponse(['message' => 'Report not found.'], 404);
            return;
        }

        if ($report['status'] !== 'checking') {
            sendJsonResponse(['message' => 'Report cannot be completed. Current status is ' . $report['status'] . '.'], 400);
            return;
        }

        if ($this->reportModel->updateStatus($id, 'completed', $report['engineer_id'], $currentUserId, null)) {
            $customer = $this->userModel->readById($report['customer_id']);
            if ($customer) {
                $this->notificationModel->createNotification(
                    $customer['id'],
                    'Report Completed',
                    'Your report (' . $report['report_index'] . ') has been completed. Thank you for your patience.',
                    'reports'
                );
                
                // Generate and email PDF
                $this->generateAndEmailPdf($report, $customer);
            }
            sendJsonResponse(['message' => 'Report completed successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to complete report.'], 500);
        }
    }

    /**
     * Reads a single report by its ID.
     * @param int $id The ID of the report to retrieve.
     */
    public function readSingle($id) {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();
        $this->reportModel->id = $id;
        $company=$this->companyModel->readByCustomer($currentUserId);
        $company_id = $company['company_id'] ?? null;
        $report = $this->reportModel->readOne($company_id,$this->reportModel->id, 'id');
        
        if (in_array($currentUserRole, ['engineer'])) {
            $assignedEngineer = $this->assignEngineerModel->readEngineerByCurrentDate();
            if ($assignedEngineer) {
                $engineer = $this->userModel->readById($assignedEngineer['engineer_id']);
                
                $status = "all";
                if($engineer['id'] == $currentUserId){
                    //error_log("YOu okay - Engineer ID: " . print_r($engineer['id'], true) . ", Current User ID: " . print_r($currentUserId, true));
                    $report = $this->reportModel->readOneEngineer($this->reportModel->id, 'id');
                    
                }else{
                    //error_log("YOu Not okay - Engineer ID: " . print_r($engineer['id'], true) . ", Current User ID: " . print_r($currentUserId, true));
                    $report = $this->reportModel->readOneSolve($this->reportModel->id, 'id');
                }
            }
            
        }
        
         if (in_array($currentUserRole, ['localcustomer','globalcustomer'])) {
            
            $company=$this->companyModel->readByCustomer($currentUserId);
            $company_id = $company['company_id'] ?? null;
            $report = $this->reportModel->readOneCompany($company_id,$this->reportModel->id, 'id');
            
        }
        
        if (!$report) {
            //sendJsonResponse(['message' => 'Report not found.'], 404);
            return;
        }
        
        $report_index = $report['report_index'];
        $report_media = $this->reportMediaModel->readByReportIndex($report_index);

        sendJsonResponse(['success'=>true,'data'=>$report,'media'=>$report_media], 200);
    }

    /**
     * Reads all reports with optional filtering.
     */
    public function readAll() {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();
       
        $customerId = $_GET['customer_id'] ?? null;
        $reports = [];
        if (in_array($currentUserRole, ['superadmin', 'admin', 'supervisor'])) {
            // Superadmins, Admins, Supervisors can see all customer products
            $reports = $this->reportModel->readAll($customerId); // This method should fetch all
        } elseif (in_array($currentUserRole, ['localcustomer','globalcustomer'])){
             $company=$this->companyModel->readByCustomer($currentUserId);
             $company_id = $company['company_id'] ?? null;
             $reports = $this->reportModel->readAll($company_id);
        } elseif (in_array($currentUserRole, ['engineer', 'champion', 'member'])) {
            $assignedEngineer = $this->assignEngineerModel->readEngineerByCurrentDate();
            if ($assignedEngineer) {
                $engineer = $this->userModel->readById($assignedEngineer['engineer_id']);
                
                $status = "all";
                if($engineer['id'] == $currentUserId){
                    //error_log("YOu okay - Engineer ID: " . print_r($engineer['id'], true) . ", Current User ID: " . print_r($currentUserId, true));
                    $reports = $this->reportModel->readAllAssign($status,$customerId);
                }else{
                    //error_log("YOu Not okay - Engineer ID: " . print_r($engineer['id'], true) . ", Current User ID: " . print_r($currentUserId, true));
                    $reports = $this->reportModel->readAllSolve($currentUserId,$status,$customerId);
                }
            }
            
        }
        
        
        if ($reports === false) {
            sendJsonResponse(['message' => 'Unauthorized or failed to retrieve reports.'], 403);
            return;
        }
        
        $reports_arr = [];
        while ($row = $reports->fetch_assoc()) {
            $reports_arr[] = $row;
        }
        
        sendJsonResponse(['data' => $reports_arr,'success'=>true]);
    }
    
    
    /**
     * Reads all uncompleted reports.
     */
    public function readUncompletedAll() {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();
        $searchTerm = $_GET['search'] ?? null;
        $reports = $this->reportModel->readUncompletedAll($searchTerm);
        
        $reports = [];
        if (in_array($currentUserRole, ['superadmin', 'admin', 'supervisor'])) {
            // Superadmins, Admins, Supervisors can see all customer products
            $reports = $this->reportModel->readUncompletedAll( $searchTerm); // This method should fetch all
        } elseif (in_array($currentUserRole, ['localcustomer','globalcustomer'])){
             $company=$this->companyModel->readByCustomer($currentUserId);
             $company_id = $company['company_id'] ?? null;
             $reports = $this->reportModel->readUncompletedAll($company_id);
        } elseif (in_array($currentUserRole, ['engineer', 'champion', 'member'])) {
            $assignedEngineer = $this->assignEngineerModel->readEngineerByCurrentDate();
            if ($assignedEngineer) {
                $engineer = $this->userModel->readById($assignedEngineer['engineer_id']);
                
                $status = "uncompleted";
                if($engineer['id'] == $currentUserId){
                    
                    $reports = $this->reportModel->readAllAssign($status);
                }else{
                    $reports = $this->reportModel->readAllSolve($currentUserId,$status);
                }
            }
            
        }
        
        $reports_arr = [];
        while ($row = $reports->fetch_assoc()) {
            $reports_arr[] = $row;
        }
        sendJsonResponse(['data' => $reports_arr, 'success' => true]);
    }

    /**
     * Reads all completed reports.
     */
    public function readCompletedAll() {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();
        $searchTerm = $_GET['search'] ?? null;
        $reports = $this->reportModel->readCompletedAll($searchTerm);
        
        $reports = [];
        if (in_array($currentUserRole, ['superadmin', 'admin', 'supervisor'])) {
            // Superadmins, Admins, Supervisors can see all customer products
            $reports = $this->reportModel->readCompletedAll( $searchTerm); // This method should fetch all
        } elseif (in_array($currentUserRole, ['localcustomer','globalcustomer'])){
             $company=$this->companyModel->readByCustomer($currentUserId);
             $company_id = $company['company_id'] ?? null;
             $reports = $this->reportModel->readCompletedAll($company_id);
        } elseif (in_array($currentUserRole, ['engineer', 'champion', 'member'])) {
            $assignedEngineer = $this->assignEngineerModel->readEngineerByCurrentDate();
            if ($assignedEngineer) {
                $engineer = $this->userModel->readById($assignedEngineer['engineer_id']);
                $status = "completed";
                if($engineer == $currentUserId){
                    
                    $reports = $this->reportModel->readAllAssign($status);
                }else{
                    $reports = $this->reportModel->readAllSolve($currentUserId,$status);
                }
            }
            
        }
        
        $reports_arr = [];
        while ($row = $reports->fetch_assoc()) {
            $reports_arr[] = $row;
        }
        sendJsonResponse(['data' => $reports_arr, 'success' => true]);
    }
    
    /**
     * Reads recent reports.
     */
    public function recents() {
        $filterStatus = $_GET['status'] ?? null;
        $searchTerm = $_GET['search'] ?? null;
        $reports = $this->reportModel->readAll($filterStatus, $searchTerm);
        
        if ($reports === false) {
            sendJsonResponse(['message' => 'Unauthorized to view recent reports or failed to retrieve.'], 403);
            return;
        }
        
        $reports_arr = [];
        while ($row = $reports->fetch_assoc()) {
            $reports_arr[] = $row;
        }
        sendJsonResponse(['data' => $reports_arr,'success'=>true]);
    }
    
    /**
     * Generates a PDF for a specific report.
     * @param int $id The ID of the report.
     */
    public function generatePdf($id) {
    $currentUserRole = getCurrentUserRole();
    $currentUserId = getCurrentUserId();

    $this->reportModel->id = $id;
    
    if(in_array($currentUserRole, ['superadmin', 'admin', 'supervisor', 'localcustomer','globalcustomer']) ){
        $report = $this->reportModel->readOne($this->reportModel->id, 'id');
    }elseif(in_array($currentUserRole, [ 'engineer']) ){
        $report = $this->reportModel->readOneEngineer($this->reportModel->id, 'id');
    }

    if (!$report) {
        sendJsonResponse(['message' => 'Report not found.'], 404);
        return;
    }
    
    $authorized = in_array($currentUserRole, ['superadmin', 'admin', 'supervisor', 'engineer']) || ($report['customer_id'] == $currentUserId);
    
    
    if (!$authorized) {
        sendJsonResponse(['message' => 'Unauthorized to generate PDF for this report.'], 403);
        return;
    }

    $reportIndex = $report['report_index'];
    // The previous call to readOne with 'index' is redundant and could overwrite data.
    // The initial $report variable already contains the necessary report data.
    // So we can remove the line `$reportData = $this->reportModel->readOne($reportIndex,'index');`

    // The line below overwrites the main report data with only root cause data.
    // We should combine the data instead.
    // Fetch related data
    $rootCauseData = $this->rootCauseModel->readAllByReportIndexPDF($reportIndex);
    $correctiveActionData = $this->correctiveActionModel->readAllByReportIndexPDF($reportIndex);
    $effectiveActionData = $this->effectiveActionModel->readByReportIndex($reportIndex);
    $preventionData = $this->preventionModel->readByReportIndex($reportIndex);
    $suggestionData = $this->suggestionModel->readByReportIndex($reportIndex);
    $report_media = $this->reportMediaModel->readByReportIndex($reportIndex); // Corrected variable name

    // Combine all data into a single array
    $combinedReportData = $report;
    $combinedReportData['root_causes'] = $rootCauseData;
    $combinedReportData['report_media'] = $report_media;
    $combinedReportData['corrective_actions'] = $correctiveActionData;
    $combinedReportData['effective_actions'] = $effectiveActionData;
     $combinedReportData['preventions'] = $preventionData;
      $combinedReportData['suggestions'] = $suggestionData;

    //error_log(print_r($combinedReportData, true));
    // The rest of the logic remains the same.
    $pdfPath = $this->pdfGenerator->generateReportPdf($combinedReportData);

    if ($pdfPath && file_exists($pdfPath)) {
        header('Content-Type: application/pdf');
        header('Content-Disposition: attachment; filename="' . basename($pdfPath) . '"');
        readfile($pdfPath);
        exit();
    } else {
        error_log("PDF generation failed or file not found for report ID: " . $id);
        sendJsonResponse(['message' => 'PDF file not found or could not be generated.'], 500);
    }
}
    
    /**
     * Deletes a report.
     * Only 'superadmin' and 'admin' roles are authorized.
     * @param int $id The ID of the report.
     */
    public function delete($id) {
        $currentUserRole = getCurrentUserRole();
        if (!in_array($currentUserRole, ['superadmin', 'admin'])) {
            sendJsonResponse(['message' => 'Unauthorized. Only superadmin or admin can delete reports.'], 403);
            return;
        }

        $this->reportModel->id = $id;

        if ($this->reportModel->delete()) {
            sendJsonResponse(['message' => 'Report deleted successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to delete report or report not found.'], 500);
        }
    }
    
    /**
     * Helper method to process file uploads.
     * @param string $reportIndex The report index.
     * @param array $filesData The $_FILES array.
     * @param array $uploadedImageUrls Array to store uploaded image URLs.
     * @param string|null $uploadedVideoUrl Variable to store the uploaded video URL.
     */
    private function processFileUploads(string $reportIndex, array $filesData, array &$uploadedImageUrls, ?string &$uploadedVideoUrl): void
    {
        // Handle multiple images (report_images[] from Flutter)
        if (isset($filesData['images']) && is_array($filesData['images']['name'])) {
            foreach ($filesData['images']['name'] as $key => $name) {
                if ($filesData['images']['error'][$key] === UPLOAD_ERR_OK) {
                    $fileEntry = [
                        'name' => $filesData['images']['name'][$key],
                        'type' => $filesData['images']['type'][$key],
                        'tmp_name' => $filesData['images']['tmp_name'][$key],
                        'error' => $filesData['images']['error'][$key],
                        'size' => $filesData['images']['size'][$key],
                    ];
                    $uploadedMedia = $this->reportMediaModel->uploadMedia($fileEntry, $reportIndex, 'image');
                    if ($uploadedMedia) {
                        $uploadedImageUrls[] = $uploadedMedia['media_path'];
                    } else {
                        error_log("Failed to upload image for report " . $reportIndex . " - File: " . $name);
                    }
                } else {
                    error_log("Image upload error for file " . $name . ": " . $filesData['images']['error'][$key]);
                }
            }
        }

        // Handle single video (report_video from Flutter)
        if (isset($filesData['report_video']) && $filesData['report_video']['error'] === UPLOAD_ERR_OK) {
            $uploadedMedia = $this->reportMediaModel->uploadMedia($filesData['report_video'], $reportIndex, 'video');
            if ($uploadedMedia) {
                $uploadedVideoUrl = $uploadedMedia['media_path'];
            } else {
                error_log("Failed to upload video for report " . $reportIndex . " - File: " . $filesData['report_video']['name']);
            }
        } else if (isset($filesData['report_video'])) {
            error_log("Video upload error: " . $filesData['report_video']['error']);
        }
    }
    
    /**
     * Helper method to send notifications for a new report.
     * @param string $reportIndex The report index.
     * @param int $currentUserId The ID of the user who created the report.
     * @param string $problemIssue The problem issue description.
     */
    private function sendNewReportNotifications(string $reportIndex, int $currentUserId, string $problemIssue): void
    {
        $customer = $this->userModel->readById($currentUserId);
        if ($customer) {
            $this->notificationModel->createNotification(
                $currentUserId,
                'Report Created',
                'Your report (' . $reportIndex . ') has been successfully created and is awaiting assignment.',
                'reports'
            );
        }
        
        $assignedEngineer = $this->assignEngineerModel->readEngineerByCurrentDate();
        if ($assignedEngineer) {
            $engineer = $this->userModel->readById($assignedEngineer['engineer_id']);
            if ($engineer) {
                $this->notificationModel->createNotification(
                    $engineer['id'],
                    'New Report Assigned',
                    'A new report (' . $reportIndex . ') has been assigned to you. Please pick it up.',
                    'reports'
                );
                $report_subject = 'New Customer Care Report: ' . $reportIndex;
                $emailBody = $this->mailer->getEmailTemplate('new_report', [
                    'engineer_name' => $engineer['fullname'],
                    'report_index' => $reportIndex,
                    'customer_name' => $customer['fullname'] ?? 'Customer',
                    'problem_issue' => $problemIssue
                ]);
                $this->mailer->sendMail($engineer['email'], $engineer['fullname'], $report_subject, $emailBody);
            }
        } else {
            error_log("No engineer assigned for the current date for report " . $reportIndex . ". Manual assignment may be needed.");
        }
    }

    /**
     * Helper method to generate and email the PDF for a completed report.
     * @param array $report The report data.
     * @param array $customer The customer's data.
     */
    private function generateAndEmailPdf(array $report, array $customer): void
    {
        $reportIndex = $report['report_index'];
        $reportData = $this->reportModel->readOne($reportIndex,'index');
        $rootCauseData = $this->rootCauseModel->readAllByReportIndexPDF($reportIndex);
    $correctiveActionData = $this->correctiveActionModel->readAllByReportIndexPDF($reportIndex);
    $effectiveActionData = $this->effectiveActionModel->readByReportIndex($reportIndex);
    $preventionData = $this->preventionModel->readByReportIndex($reportIndex);
    $suggestionData = $this->suggestionModel->readByReportIndex($reportIndex);
    $report_media = $this->reportMediaModel->readByReportIndex($reportIndex); // Corrected variable name

    // Combine all data into a single array
    $combinedReportData = $reportData;
    $combinedReportData['root_causes'] = $rootCauseData;
    $combinedReportData['report_media'] = $report_media;
    $combinedReportData['corrective_actions'] = $correctiveActionData;
    $combinedReportData['effective_actions'] = $effectiveActionData;
     $combinedReportData['preventions'] = $preventionData;
      $combinedReportData['suggestions'] = $suggestionData;
        
        if (!is_array($combinedReportData) || empty($combinedReportData)) {
            error_log("Failed to retrieve detailed report data for PDF generation for report ID: " . $report['id']);
            return;
        }
        error_log("Failed to retrieve detailed report data for PDF generation for report ID: " . $report['id']);
        
        //error_log(print_r($reportData,true));
        
        $pdfPath = $this->pdfGenerator->generateReportPdf($combinedReportData);
        if ($pdfPath && file_exists($pdfPath)) {
            $complete_subject = 'Your customer care report (' . $reportIndex . ') has been successfully completed.';
            $emailBody = $this->mailer->getEmailTemplate('report_completed_customer', [
                'customer_name' => $customer['fullname'],
                'report_index' => $reportIndex
            ]);
            $this->mailer->sendMail($customer['email'], $customer['fullname'], $complete_subject, $emailBody, $pdfPath);
        } else {
            error_log("Failed to generate or find PDF for completed report " . $reportIndex);
        }
    }
}