<?php
// controllers/ReportMediaController.php

/**
 * ReportMediaController for Rehlko Customer Care application.
 * Handles uploading and managing media files (images/videos) associated with reports.
 * Primarily used for root cause and corrective action media.
 */
class ReportMediaController {
    private $conn;
    private $reportMediaModel;
    private $reportModel; // To validate report_index

    /**
     * Constructor for ReportMediaController.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
        $this->reportMediaModel = new ReportMedia($db);
        $this->reportModel = new Report($db);
    }

    /**
     * Uploads a media file for a specific report.
     * Expects 'report_index' and 'media_type' (image/video) in POST fields,
     * and the file itself in $_FILES.
     * Only engineers assigned to the report can upload media.
     */
    public function uploadMedia() {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        if ($currentUserRole !== 'engineer') {
            sendJsonResponse(['message' => 'Unauthorized. Only engineers can upload report media.'], 403);
        }

        // Validate required POST fields
        if (empty($_POST['report_index']) || empty($_POST['media_type'])) {
            sendJsonResponse(['message' => 'Missing required fields: report_index, media_type.'], 400);
        }
        $reportIndex = $_POST['report_index'];
        $mediaType = $_POST['media_type'];

        // Validate uploaded file
        if (!isset($_FILES['media_file']) || $_FILES['media_file']['error'] !== UPLOAD_ERR_OK) {
            sendJsonResponse(['message' => 'No file uploaded or upload error.'], 400);
        }

        // Verify report existence and engineer authorization
        $report = $this->reportModel->readOne($reportIndex, 'index');
        if (!$report) {
            sendJsonResponse(['message' => 'Report not found.'], 404);
        }
        if ($report['engineer_id'] != $currentUserId) {
            sendJsonResponse(['message' => 'Unauthorized. You are not assigned to this report.'], 403);
        }

        $file = $_FILES['media_file'];

        $uploadedMedia = $this->reportMediaModel->uploadMedia($file, $reportIndex, $mediaType);

        if ($uploadedMedia) {
            sendJsonResponse(['message' => 'Media uploaded successfully.', 'media' => $uploadedMedia], 201);
        } else {
            sendJsonResponse(['message' => 'Failed to upload media. Check file type or server permissions.'], 500);
        }
    }

    /**
     * Reads all media associated with a given report index.
     * Access is restricted based on report view permissions.
     *
     * @param string $reportIndex The report index to retrieve media for.
     */
    public function readMediaByReport($reportIndex) {
        $currentUserRole = getCurrentUserRole();
        if (!$currentUserRole) {
            sendJsonResponse(['message' => 'Authentication required.'], 401);
        }

        // Verify report existence and user authorization to view this report
        // The Report model's readOne method already handles comprehensive authorization.
        $report = $this->reportModel->readOne($reportIndex, 'index');
        if (!$report) {
            sendJsonResponse(['message' => 'Report not found or unauthorized to view.'], 404); // readOne will send 403 if unauthorized
        }

        $media = $this->reportMediaModel->readByReportIndex($reportIndex);

        if ($media !== false) { // check for strict false, as it returns empty array on no results
            sendJsonResponse(['data'=>$media]);
        } else {
            sendJsonResponse(['message' => 'Failed to retrieve media for this report.'], 500);
        }
    }

    /**
     * Deletes a specific media record.
     * Only the engineer who uploaded it, or superadmin/admin, can delete.
     *
     * @param int $id The ID of the media record to delete.
     */
    public function delete($id) {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        // Fetch media record to check ownership
        $this->reportMediaModel->id = $id;
        $media = $this->reportMediaModel->readOne(); // Adjust readOne to fetch individual media if not exists

        if (!$media) {
            sendJsonResponse(['message' => 'Media not found.'], 404);
        }

        // Authorize: Only superadmin/admin or the original creator (engineer)
        $authorized = false;
        if (in_array($currentUserRole, ['superadmin', 'admin'])) {
            $authorized = true;
        } else {
            // Need to get the created_user for this media from the database.
            // Assuming the `report_media` table itself doesn't have `created_user`,
            // we'd need to fetch the associated report and check its engineer_id.
            // If the schema was structured so root_cause/corrective_action have media,
            // then the deletion would be handled by those controllers.
            // For simplicity, assuming report_media has a `created_user` column now for this check.
            // If not, you'd link back to the report's engineer_id.
            $report = $this->reportModel->readOne($media['report_index'], 'index');
            if ($report && $report['engineer_id'] == $currentUserId) {
                 $authorized = true;
            }
        }

        if (!$authorized) {
            sendJsonResponse(['message' => 'Unauthorized to delete this media.'], 403);
        }

        if ($this->reportMediaModel->delete($id)) { // Pass ID to delete method
            sendJsonResponse(['message' => 'Media deleted successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to delete media or media not found.'], 500);
        }
    }
}
?>
