<?php
// controllers/SuggestionController.php

/**
 * SuggestionController for Rehlko Customer Care application.
 * Manages CRUD operations for suggestion entries linked to reports.
 * Only engineers assigned to a report can perform CRUD on its suggestions.
 */
class SuggestionController {
    private $conn;
    private $suggestionModel;
    private $reportModel; // To validate report_index and engineer assignment

    /**
     * Constructor for SuggestionController.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
        $this->suggestionModel = new Suggestion($db);
        $this->reportModel = new Report($db);
    }

    /**
     * Creates a new suggestion for a report.
     * Only engineers assigned to the report can create.
     */
    public function create() {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();
        $report_index = $_POST['report_index'] ?? null;
        $suggestion_title = $_POST['suggestion_title'] ?? null;

        // Validate required fields
        if (empty($report_index) || empty($suggestion_title)) {
            sendJsonResponse(['message' => 'Missing required fields: report_index, suggestion_title.'], 400);
            return; // Added return
        }
        
        $authorized = false;
        if (in_array($currentUserRole, ['superadmin', 'admin'])) {
            $report = $this->reportModel->readOne($report_index, 'index');
            $authorized = true;
        } else if($currentUserRole === 'engineer') {
            $report = $this->reportModel->readOneEngineer($report_index, 'index');
             $authorized = true;
        }
        

        if (!$authorized) {
            sendJsonResponse(['message' => 'Unauthorized. Only engineers can create root causes.'], 403);
        }
        // For multipart/form-data, data is in $_POST
        

        // Verify report existence and engineer authorization
        
        if (!$report) {
            sendJsonResponse(['message' => 'Report not found.'], 404);
            return; // Added return
        }
        

        // Set model properties
        $this->suggestionModel->report_index = $report_index;
        $this->suggestionModel->suggestion_title = $suggestion_title;
        $this->suggestionModel->created_user = $currentUserId;

        if ($this->suggestionModel->create()) {
            sendJsonResponse(['success' => true, 'message' => 'Suggestion created successfully.', 'id' => $this->suggestionModel->id], 201);
        } else {
            sendJsonResponse(['message' => 'Failed to create suggestion.'], 500);
        }
    }

    /**
     * Reads suggestion entries for a given report index.
     * Access is based on report view permissions.
     *
     * @param string $reportIndex The report index to retrieve suggestions for.
     */
    public function readByReportIndex($reportIndex) {
        $currentUserRole = getCurrentUserRole();
        if (!$currentUserRole) {
            sendJsonResponse(['message' => 'Authentication required.'], 401);
            return; // Added return
        }
        
        if (in_array($currentUserRole, ['superadmin', 'admin'])) {
            $report = $this->reportModel->readOne($reportIndex, 'index');
            $authorized = true;
        } else if($currentUserRole === 'engineer') {
            $report = $this->reportModel->readOneEngineer($reportIndex, 'index');
             $authorized = true;
        }

        // Verify user authorization to view this report first
        
        

        $suggestions = $this->suggestionModel->readByReportIndex($reportIndex);

        if ($suggestions !== false) {
            sendJsonResponse(['success' => true, 'data' => $suggestions], 200); // Added success and data keys
        } else {
            sendJsonResponse(['message' => 'Failed to retrieve suggestions.'], 500);
        }
    }

    /**
     * Updates an existing suggestion.
     * Only the engineer assigned to the report can update.
     *
     * @param int $id The ID of the suggestion to update.
     */
    public function update($id) {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

       

        // For multipart/form-data (even for PUT-like behavior via POST), data is in $_POST
        $suggestion_title = $_POST['suggestion_title'] ?? null;

        if (empty($suggestion_title)) {
            sendJsonResponse(['message' => 'Missing required field: suggestion_title.'], 400);
            return; // Added return
        }

        // Fetch the suggestion to get its report_index
        $this->suggestionModel->id = $id;
        $suggestion = $this->suggestionModel->readOne();
        if (!$suggestion) {
            sendJsonResponse(['message' => 'Suggestion not found.'], 404);
            return; // Added return
        }

        // Verify engineer authorization for the associated report
        $report = $this->reportModel->readOne($suggestion['report_index'], 'index');
        $authorized = false;
        if (in_array($currentUserRole, ['superadmin', 'admin'])) {
            $authorized = true;
        } else if($currentUserRole === 'engineer' && $report['engineer_id'] == $currentUserId) {
             $authorized = true;
        }
        

        if (!$authorized) {
            sendJsonResponse(['message' => 'Unauthorized. Only engineers can create root causes.'], 403);
        }
        

        // Set model properties for update
        $this->suggestionModel->suggestion_title = $suggestion_title;
        $this->suggestionModel->id = $id;

        if ($this->suggestionModel->update()) {
            sendJsonResponse(['success' => true, 'message' => 'Suggestion updated successfully.'], 200); // Added success key
        } else {
            sendJsonResponse(['message' => 'Failed to update suggestion.'], 500);
        }
    }

    /**
     * Deletes a suggestion.
     * Only the engineer assigned to the report can delete, or superadmin/admin.
     *
     * @param int $id The ID of the suggestion to delete.
     */
    public function delete($id) {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        // Fetch the suggestion to get its report_index
        $this->suggestionModel->id = $id;
        $suggestion = $this->suggestionModel->readOne();
        if (!$suggestion) {
            sendJsonResponse(['message' => 'Suggestion not found.'], 404);
            return; // Added return
        }

        // Authorize: Superadmin/Admin or the assigned engineer
        $authorized = false;
        if (in_array($currentUserRole, ['superadmin', 'admin'])) {
            $authorized = true;
        } else {
            $report = $this->reportModel->readOne($suggestion['report_index'], 'index');
            if ($report && $report['engineer_id'] == $currentUserId) {
                 $authorized = true;
            }
        }

        if (!$authorized) {
            sendJsonResponse(['message' => 'Unauthorized to delete this suggestion.'], 403);
            return; // Added return
        }

        if ($this->suggestionModel->delete()) {
            sendJsonResponse(['success' => true, 'message' => 'Suggestion deleted successfully.'], 200); // Added success key
        } else {
            sendJsonResponse(['message' => 'Failed to delete suggestion.'], 500);
        }
    }
}
?>
