<?php
// controllers/WarningConfigController.php

/**
 * WarningConfigController for Rehlko Customer Care application.
 * Manages CRUD operations for report warning configurations.
 * Access is restricted to 'superadmin' and 'admin' roles.
 */
class WarningConfigController {
    private $conn;
    private $warningConfigModel;

    /**
     * Constructor for WarningConfigController.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
        $this->warningConfigModel = new ReportWarningConfig($db);
    }

    /**
     * Reads all warning configurations.
     * Only 'superadmin' and 'admin' roles are authorized.
     */
    public function readAll() {
        $currentUserRole = getCurrentUserRole();
        if (!in_array($currentUserRole, ['superadmin', 'admin'])) {
            sendJsonResponse(['message' => 'Unauthorized. Only superadmin or admin can view warning configurations.'], 403);
        }

        $result = $this->warningConfigModel->readAll();

        if ($result) {
            $configs_arr = [];
            while ($row = $result->fetch_assoc()) {
                $configs_arr[] = $row;
            }
            sendJsonResponse(['data'=>$configs_arr]);
        } else {
            sendJsonResponse(['message' => 'Failed to retrieve warning configurations.'], 500);
        }
    }

    /**
     * Updates an existing warning configuration.
     * Only 'superadmin' and 'admin' roles are authorized.
     *
     * @param int $id The ID of the warning configuration to update.
     */
    public function update($id) {
        $currentUserRole = getCurrentUserRole();
        if (!in_array($currentUserRole, ['superadmin', 'admin'])) {
            sendJsonResponse(['message' => 'Unauthorized. Only superadmin or admin can update warning configurations.'], 403);
        }

        $data = json_decode(file_get_contents("php://input"));

        // Validate required fields
        if (empty($data->customer_type) || !isset($data->first_warning_hours) || !isset($data->last_warning_hours)) {
            sendJsonResponse(['message' => 'Missing required fields: customer_type, first_warning_hours, last_warning_hours.'], 400);
        }

        // Check if config exists
        $this->warningConfigModel->id = $id;
        if (!$this->warningConfigModel->readOne()) {
            sendJsonResponse(['message' => 'Warning configuration not found.'], 404);
        }

        // Set model properties for update
        $this->warningConfigModel->customer_type = $data->customer_type;
        $this->warningConfigModel->first_warning_hours = (int)$data->first_warning_hours;
        $this->warningConfigModel->last_warning_hours = (int)$data->last_warning_hours;
        $this->warningConfigModel->id = $id;

        if ($this->warningConfigModel->update()) {
            sendJsonResponse(['message' => 'Warning configuration updated successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to update warning configuration.'], 500);
        }
    }

    /**
     * Deletes a warning configuration.
     * Only 'superadmin' and 'admin' roles are authorized.
     *
     * @param int $id The ID of the warning configuration to delete.
     */
    public function delete($id) {
        $currentUserRole = getCurrentUserRole();
        if (!in_array($currentUserRole, ['superadmin', 'admin'])) {
            sendJsonResponse(['message' => 'Unauthorized. Only superadmin or admin can delete warning configurations.'], 403);
        }

        $this->warningConfigModel->id = $id;

        if ($this->warningConfigModel->delete()) {
            sendJsonResponse(['message' => 'Warning configuration deleted successfully.'], 200);
        } else {
            sendJsonResponse(['message' => 'Failed to delete warning configuration or not found.'], 500);
        }
    }
}
?>
