<?php
// cron/maintenance_reminders.php

/**
 * Cron Job: Maintenance Reminders
 * This script is designed to be run daily by a server cron scheduler.
 * It checks for upcoming maintenance records and sends reminders to engineers.
 * It also notifies product owners (customers) when maintenance is updated.
 *
 * Modifications:
 * - Sends reminders 2 weeks before the maintenance_check_date.
 * - Informs the engineer assigned to the maintenance, and additionally,
 * the engineer on duty for the current date.
 * - Resets `is_check` to 0 for maintenances that have reached their
 * maintenance_check_date but are still marked as checked (is_check = 1).
 */

// Define the root directory if not already defined
if (!defined('ROOT_DIR')) {
    define('ROOT_DIR', realpath(__DIR__ . '/..'));
}

// Include necessary configuration and utility files
require_once ROOT_DIR . '/config/database.php';
require_once ROOT_DIR . '/config/mail.php'; // For APP_BASE_URL
require_once ROOT_DIR . '/utils/helpers.php'; // For email templating functions
require_once ROOT_DIR . '/utils/Mailer.php';

// Include all required models
require_once ROOT_DIR . '/models/Maintenance.php';
require_once ROOT_DIR . '/models/User.php';
require_once ROOT_DIR . '/models/Product.php'; // Needed for product details
require_once ROOT_DIR . '/models/CustomerProduct.php'; // Needed to link product to customer
require_once ROOT_DIR . '/models/Notification.php';
require_once ROOT_DIR . '/models/AssignEngineer.php'; // NEW: Include AssignEngineer model

$maintenanceModel = new Maintenance($conn);
$userModel = new User($conn);
$productModel = new Product($conn);
$customerProductModel = new CustomerProduct($conn);
$notificationModel = new Notification($conn);
$assignEngineerModel = new AssignEngineer($conn); // NEW: Instantiate AssignEngineer model
$mailer = new Mailer();

echo "Starting Maintenance Reminders Cron Job...\n";

// --- Step 1: Send Reminders for Upcoming Maintenance to Engineers ---
// Fetch maintenances due in the next 14 days (2 weeks)
// This assumes your getUpcomingMaintenances method queries against maintenance_check_date
$upcomingMaintenances = $maintenanceModel->getUpcomingMaintenances(14); // Changed to 14 days

// Get the engineer currently on duty
$onDutyEngineer = $assignEngineerModel->readEngineerByCurrentDate();
$onDutyEngineerId = $onDutyEngineer ? $onDutyEngineer['engineer_id'] : null;

echo "On Duty Engineer ID is :".$onDutyEngineerId;

foreach ($upcomingMaintenances as $maintenance) {
    // Implement reminder sent logic if you have `last_engineer_reminder_sent_at` in `maintenance` table
    // $lastSent = strtotime($maintenance['last_engineer_reminder_sent_at'] ?? '1970-01-01');
    // if (time() - $lastSent < (24 * 60 * 60)) { // If sent within last 24 hours
    //    continue;
    // }

    echo "Processing upcoming maintenance for " . $maintenance['generator_serial_number'] . "\n";

    // Get customer details for the product owner
    $customerProduct = $customerProductModel->getByGeneratorSerialNumber($maintenance['generator_serial_number']);
    $customer = null;
    if ($customerProduct) {
        $customer = $userModel->readById($customerProduct['customer_id']);
    }

    $engineersToNotify = [];

    // 1. Add the specific engineer assigned to this maintenance (if available in the maintenance record)
    if (isset($maintenance['assigned_engineer_id']) && $maintenance['assigned_engineer_id']) {
        $assignedEngineer = $userModel->readById($maintenance['assigned_engineer_id']);
        if ($assignedEngineer) {
            $engineersToNotify[$assignedEngineer['id']] = $assignedEngineer; // Use ID as key to prevent duplicates
        }
    }

    // 2. Add the general on-duty engineer if there is one
    if ($onDutyEngineerId) {
        $onDutyUser = $userModel->readById($onDutyEngineerId);
        if ($onDutyUser && !isset($engineersToNotify[$onDutyUser['id']])) {
            $engineersToNotify[$onDutyUser['id']] = $onDutyUser;
        }
    }

    // If no specific engineers were identified, you might want to notify all engineers or skip.
    // Current logic: if $engineersToNotify is empty, no one gets notified for this maintenance.
    // Or, you could revert to notifying all engineers: $engineersToNotify = $userModel->getUsersByRole('engineer');

    foreach ($engineersToNotify as $engineer) {
        // Create in-app notification for engineer
        // Note: Using $this->notificationModel inside a cron script like this is incorrect if it's not a class method.
        // It should be $notificationModel->createNotification().
        $notificationModel->createNotification(
            $engineer['id'],
            'Upcoming Maintenance Reminder',
            'Maintenance for generator ' . $maintenance['generator_serial_number'] . ' (' . $maintenance['maintenance_title'] . ') is due (' . $maintenance['maintenance_type'] . ') on ' . date('Y-m-d', strtotime($maintenance['maintenance_check_date'])) . '.', // Use maintenance_check_date
            'maintenance'
        );

        // Send email to engineer
        $emailBody = $mailer->getEmailTemplate('engineer_maintenance_reminder', [
            'engineer_name' => $engineer['fullname'],
            'generator_serial_number' => $maintenance['generator_serial_number'],
            'maintenance_title' => $maintenance['maintenance_title'],
            'maintenance_type' => $maintenance['maintenance_type'],
            'due_date' => date('Y-m-d H:i', strtotime($maintenance['maintenance_check_date'])), // Use maintenance_check_date
            'customer_name' => $customer ? $customer['fullname'] : 'N/A',
            'app_link' => APP_BASE_URL . '/app/maintenance/' . $maintenance['id'] // Link to maintenance record
        ]);
        $mailer->sendMail(
            $engineer['email'],
            $engineer['fullname'],
            'Reminder: Upcoming Generator Maintenance',
            $emailBody
        );
    }
    // Mark that a reminder has been sent for this maintenance to engineers
    $maintenanceModel->markMaintenanceReminderSent($maintenance['id']);
}

echo "Finished sending upcoming maintenance reminders.\n";
echo "--- Step 2: Reset 'is_check' for due maintenances. ---\n";

// NEW: Step 2.5: Reset `is_check` for maintenances that are due today
// This logic will find maintenances where the `maintenance_check_date` is today,
// and `is_check` is currently 1 (meaning it was checked but now needs to be re-checked/reset).
// It sets `is_check` back to 0.
$maintenancesToReset = $maintenanceModel->getMaintenancesDueTodayAndChecked();

foreach ($maintenancesToReset as $maintenance) {
    echo "Resetting is_check for maintenance ID " . $maintenance['id'] . " (Generator: " . $maintenance['generator_serial_number'] . ").\n";
    // Assuming a method in Maintenance model to update only is_check status
    $maintenanceModel->resetMaintenanceCheckStatus($maintenance['id']);
}

echo "Finished resetting 'is_check' status.\n";

// --- Step 3: Notify Customer After Maintenance Update --- (Original Step 2, now Step 3)
// Fetch maintenances that were recently updated and customer hasn't been notified yet.
// Assumes `modified_time` is updated on record changes and `last_customer_notified_at` track.
$recentlyUpdatedMaintenances = $maintenanceModel->getRecentlyUpdatedMaintenances(60); // Check for updates in last 60 minutes

foreach ($recentlyUpdatedMaintenances as $maintenance) {
    // Prevent duplicate customer notifications for the same update
    // $lastNotified = strtotime($maintenance['last_customer_notified_at'] ?? '1970-01-01');
    // if (strtotime($maintenance['modified_time']) <= $lastNotified) {
    //    continue; // Already notified for this update
    // }

    echo "Processing recently updated maintenance for customer notification: " . $maintenance['generator_serial_number'] . "\n";

    $customerProduct = $customerProductModel->getByGeneratorSerialNumber($maintenance['generator_serial_number']);
    if ($customerProduct) {
        $customer = $userModel->readById($customerProduct['customer_id']);
        if ($customer) {
            // Create in-app notification for customer
            $notificationModel->createNotification( // Corrected call from $this->notificationModel
                $customer['id'],
                'Maintenance Update',
                'Maintenance for your generator ' . $maintenance['generator_serial_number'] . ' has been updated.',
                'maintenance'
            );

            // Send email to customer
            $emailBody = $mailer->getEmailTemplate('customer_maintenance_update', [
                'customer_name' => $customer['fullname'],
                'generator_serial_number' => $maintenance['generator_serial_number'],
                'maintenance_title' => $maintenance['maintenance_title'],
                'maintenance_date' => date('Y-m-d H:i', strtotime($maintenance['maintenance_date'])),
                'app_link' => APP_BASE_URL . '/app/maintenance/' . $maintenance['id']
            ]);
            $mailer->sendMail(
                $customer['email'],
                $customer['fullname'],
                'Maintenance Update for Your Generator',
                $emailBody
            );
            // Mark that customer has been notified for this update
            $maintenanceModel->markCustomerNotified($maintenance['id']);
        }
    }
}

echo "Maintenance Reminders Cron Job Finished.\n";

$conn->close(); // Close DB connection
?>