<?php
// cron/report_reminders.php

/**
 * Cron Job: Report Reminders
 * This script is designed to be run periodically (e.g., every 5-10 minutes)
 * by a server cron scheduler.
 * It checks for reports needing reminders based on warning configurations
 * and sends notifications via email (PHPMailer) and in-app notifications.
 */

// Define the root directory if not already defined (e.g., when run from cron)
// Adjust this path if your cron setup requires a different base directory.
if (!defined('ROOT_DIR')) {
    define('ROOT_DIR', realpath(__DIR__ . '/..'));
}

// Include necessary configuration and utility files
require_once ROOT_DIR . '/config/database.php';
require_once ROOT_DIR . '/config/mail.php'; // For APP_BASE_URL
require_once ROOT_DIR . '/utils/helpers.php';
require_once ROOT_DIR . '/utils/Mailer.php';
require_once ROOT_DIR . '/utils/PdfGenerator.php';

// Include all required models
require_once ROOT_DIR . '/models/Report.php';
require_once ROOT_DIR . '/models/User.php';
require_once ROOT_DIR . '/models/Notification.php';
require_once ROOT_DIR . '/models/ReportReminder.php';
require_once ROOT_DIR . '/models/ReportWarningConfig.php';
require_once ROOT_DIR . '/models/AssignEngineer.php';
// Include other related models if needed for PDF generation or detailed logs
require_once ROOT_DIR . '/models/RootCause.php';
require_once ROOT_DIR . '/models/CorrectiveAction.php';
require_once ROOT_DIR . '/models/EffectiveAction.php';
require_once ROOT_DIR . '/models/Prevention.php';
require_once ROOT_DIR . '/models/Suggestion.php';
require_once ROOT_DIR . '/models/ReportMedia.php'; // If media paths need resolving in PDF

$reportModel = new Report($conn);
$userModel = new User($conn);
$notificationModel = new Notification($conn);
$reportReminderModel = new ReportReminder($conn);
$warningConfigModel = new ReportWarningConfig($conn);
$assignEngineerModel = new AssignEngineer($conn);
$mailer = new Mailer();
$pdfGenerator = new PdfGenerator();


echo "Starting Report Reminders Cron Job...\n";

// --- Step 1: Fetch Warning Configurations ---
$localWarningConfig = $warningConfigModel->getWarningConfig('local'); // Ensure exact match to DB enum
$globalWarningConfig = $warningConfigModel->getWarningConfig('global'); // Ensure exact match to DB enum

if (!$localWarningConfig) {
    error_log("CRON ERROR: Local customer warning config not found. Please set it up.");
    $localWarningConfig = ['first_warning_hours' => 2, 'last_warning_hours' => 4]; // Default fallback
}
if (!$globalWarningConfig) {
    error_log("CRON ERROR: Global customer warning config not found. Please set it up.");
    $globalWarningConfig = ['first_warning_hours' => 3, 'last_warning_hours' => 6]; // Default fallback
}

// --- Step 2: Process Pending Reports for Pickup Reminders ---
$pendingReports = $reportModel->getPendingReports(); // Fetches reports with status 'pending' and is_pick_up = 0

foreach ($pendingReports as $report) {
    $reportId = $report['id'];
    $customerUser = $userModel->readById($report['customer_id']);

    if (!$customerUser) {
        error_log("CRON WARNING: Customer ID " . $report['customer_id'] . " not found for report " . $report['report_index']);
        continue;
    }

    $customerType = (strpos($customerUser['role'], 'localcustomer') !== false) ? 'local' : 'global'; // Match DB enum
    $config = ($customerType === 'local') ? $localWarningConfig : $globalWarningConfig;

    $firstWarningMinutes = $config['first_warning_hours'] * 60;
    $lastWarningMinutes = $config['last_warning_hours'] * 60;

    $createdTime = strtotime($report['created_datetime']);
    $minutesElapsed = round((time() - $createdTime) / 60);

    echo "Processing report " . $report['report_index'] . " (Minutes elapsed: $minutesElapsed)\n";
    echo $firstWarningMinutes;

    // Fetch the assigned engineer for this customer's support team
    $CurrentEngineer = $assignEngineerModel->readEngineerByCurrentDate();
    if (!empty($CurrentEngineer['engineer_id'])) {
        $assignedEngineer = $userModel->readById($CurrentEngineer['engineer_id']);
        echo $CurrentEngineer;
    }

    // --- First Warning Logic ---
    if ($minutesElapsed >= $firstWarningMinutes && !$reportReminderModel->hasReminderBeenSent($reportId, 'first_warning')) {
        echo "Sending first warning for report " . $report['report_index'] . "\n";
        if ($assignedEngineer) {
            // Send to assigned engineer
            var_dump($assignedEngineer);
            $notificationModel->createNotification(
                $assignedEngineer['id'],
                'Report Pickup Reminder',
                'Report ' . $report['report_index'] . ' from ' . $customerUser['fullname'] . ' needs to be picked up.',
                'reports'
            );
            $emailBody = $mailer->getEmailTemplate('new_report_engineer', [
                'engineer_name' => $assignedEngineer['fullname'],
                'report_index' => $report['report_index'],
                'customer_name' => $customerUser['fullname'],
                'problem_issue' => $report['problem_issue'],
                'app_link' => APP_BASE_URL . '/app/reports/' . $reportId // Link to report in app
            ]);
            $mailer->sendMail(
                $assignedEngineer['email'],
                $assignedEngineer['fullname'],
                'Reminder: Pick Up Report ' . $report['report_index'],
                $emailBody
            );
            $reportReminderModel->addReminder($reportId, 'first_warning', 1);
        } else {
            // If no assigned engineer, send to all engineers
            $allEngineers = $userModel->getUsersByRole('engineer');
            $recipientsCount = 0;
            foreach ($allEngineers as $engineer) {
                $notificationModel->createNotification(
                    $engineer['id'],
                    'Report Pickup Reminder',
                    'Report ' . $report['report_index'] . ' from ' . $customerUser['fullname'] . ' needs to be picked up. No specific engineer assigned yet.',
                    'reports'
                );
                $emailBody = $mailer->getEmailTemplate('new_report_engineer_general', [
                    'engineer_name' => $engineer['fullname'],
                    'report_index' => $report['report_index'],
                    'customer_name' => $customerUser['fullname'],
                    'problem_issue' => $report['problem_issue'],
                    'app_link' => APP_BASE_URL . '/app/reports/' . $reportId
                ]);
                $mailer->sendMail($engineer['email'], $engineer['fullname'], 'Reminder: Pick Up Report ' . $report['report_index'], $emailBody);
                $recipientsCount++;
            }
            if ($recipientsCount > 0) {
                 $reportReminderModel->addReminder($reportId, 'first_warning', $recipientsCount);
            }
        }
    }

    // --- Last Warning Logic ---
    if ($minutesElapsed >= $lastWarningMinutes && !$reportReminderModel->hasReminderBeenSent($reportId, 'last_warning')) {
        echo "Sending last warning for report " . $report['report_index'] . "\n";
        // Notify all engineers, admins, and supervisors
        $allEngineers = $userModel->getUsersByRole('engineer');
        $admins = $userModel->getUsersByRole('admin');
        $supervisors = $userModel->getUsersByRole('supervisor');

        $recipients = array_merge($allEngineers, $admins, $supervisors);
        $recipientCount = 0;
        foreach ($recipients as $recipient) {
            $notificationModel->createNotification(
                $recipient['id'],
                'CRITICAL: Report Not Picked Up',
                'Report ' . $report['report_index'] . ' has passed its last warning time and is still unpicked.',
                'reports'
            );
            $emailBody = $mailer->getEmailTemplate('critical_report_warning', [
                'recipient_name' => $recipient['fullname'],
                'report_index' => $report['report_index'],
                'customer_name' => $customerUser['fullname'],
                'problem_issue' => $report['problem_issue'],
                'app_link' => APP_BASE_URL . '/app/reports/' . $reportId
            ]);
            $mailer->sendMail(
                $recipient['email'],
                $recipient['fullname'],
                'CRITICAL: Report ' . $report['report_index'] . ' Overdue',
                $emailBody
            );
            $recipientCount++;
        }
        if ($recipientCount > 0) {
            $reportReminderModel->addReminder($reportId, 'last_warning', $recipientCount);
        }
    }
}

// --- Step 3: Process Reports changing to 'checking' status (for Supervisor/Admin notification) ---
$checkingReports = $reportModel->getCheckingReports(); // Fetch reports with status 'checking'

foreach ($checkingReports as $report) {
    $reportId = $report['id'];
    // Only send notification if it hasn't been sent for 'checking' status before
    if (!$reportReminderModel->hasStatusNotificationBeenSent($reportId, 'checking')) {
        echo "Sending checking notification for report " . $report['report_index'] . "\n";
        $supervisors = $userModel->getUsersByRole('supervisor');
        $admins = $userModel->getUsersByRole('admin');
        $recipients = array_merge($supervisors, $admins);

        foreach ($recipients as $recipient) {
            $notificationModel->createNotification(
                $recipient['id'],
                'Report Ready for Review',
                'Report ' . $report['report_index'] . ' is now in "checking" status and requires your review.',
                'reports'
            );
            $emailBody = $mailer->getEmailTemplate('report_ready_for_check_supervisor_admin', [
                'recipient_name' => $recipient['fullname'],
                'report_index' => $report['report_index'],
                'engineer_name' => $report['engineer_fullname'] ?? 'An Engineer',
                'app_link' => APP_BASE_URL . '/app/reports/' . $reportId
            ]);
            $mailer->sendMail(
                $recipient['email'],
                $recipient['fullname'],
                'Action Required: Report ' . $report['report_index'] . ' Check',
                $emailBody
            );
        }
        $reportReminderModel->markStatusNotificationSent($reportId, 'checking');
    }
}

// --- Step 4: Process Completed Reports (for Customer notification and PDF export) ---
// This part is generally handled within the `ReportController->completeReport()` method
// to ensure the PDF is generated and sent immediately upon completion.
// However, if there's a need to re-send or handle delayed PDF generation/mailing,
// this cron job could pick up reports that are 'completed' but customer_notified_at is null/old.

/*
// Example for handling completed reports in cron if needed:
$completedReports = $reportModel->getCompletedReportsNotYetNotified(); // You'd need to implement this method
foreach ($completedReports as $report) {
    $reportId = $report['id'];
    if (!$reportReminderModel->hasStatusNotificationBeenSent($reportId, 'completed')) {
        echo "Processing completed report " . $report['report_index'] . " for customer notification...\n";
        $reportDetails = $reportModel->getReportDetailsForPdf($reportId);
        if ($reportDetails) {
            $pdfPath = $pdfGenerator->generateReportPdf($reportDetails['report_index'], $reportDetails);

            if ($pdfPath && file_exists($pdfPath)) {
                $customer = $userModel->readById($reportDetails['customer_id']);
                if ($customer) {
                    $emailBody = $mailer->getEmailTemplate('report_completed_customer', [
                        'customer_name' => $customer['fullname'],
                        'report_index' => $reportDetails['report_index'],
                        'app_link' => APP_BASE_URL . '/app/reports/' . $reportId
                    ]);
                    $mailer->sendMail(
                        $customer['email'],
                        $customer['fullname'],
                        'Your Report ' . $reportDetails['report_index'] . ' is Completed',
                        $emailBody,
                        $pdfPath // Attach the generated PDF
                    );
                    $notificationModel->createNotification(
                        $customer['id'],
                        'Report Completed',
                        'Your report (' . $reportDetails['report_index'] . ') has been completed. A copy has been sent to your email.',
                        'reports'
                    );
                    $reportReminderModel->markStatusNotificationSent($reportId, 'completed');
                }
            } else {
                error_log("CRON ERROR: Failed to generate or find PDF for completed report " . $report['report_index']);
            }
        }
    }
}
*/

echo "Report Reminders Cron Job Finished.\n";

$conn->close(); // Close DB connection
?>
