<?php
// models/AssignEngineer.php

/**
 * AssignEngineer Model for Rehlko Customer Care application.
 * Manages data interaction for the 'assign_engineer' table.
 */
class AssignEngineer {
    private $conn;
    private $table_name = "assign_engineer";

    // Object properties
    public $id;
    public $engineer_id;
    public $start_date;
    public $end_date;

    /**
     * Constructor for the AssignEngineer model.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
    }

    //------------------------------------------------------------------------------------------------------------------

    /**
     * Creates a new engineer assignment record.
     *
     * @return bool True on success, false on failure.
     */
    public function create() {
        $query = "INSERT INTO " . $this->table_name . "
                  SET engineer_id=?, start_date=?, end_date=?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize inputs
        $this->engineer_id = htmlspecialchars(strip_tags($this->engineer_id));
        $this->start_date = htmlspecialchars(strip_tags($this->start_date));
        $this->end_date = htmlspecialchars(strip_tags($this->end_date));

        $stmt->bind_param("iss", $this->engineer_id, $this->start_date, $this->end_date);

        if ($stmt->execute()) {
            $this->id = $this->conn->insert_id;
            return true;
        }
        error_log("Engineer assignment creation failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    //------------------------------------------------------------------------------------------------------------------

    /**
     * Reads all engineer assignment records.
     *
     * @return mysqli_result|false The result set on success, false on failure.
     */
    public function readAll() {
        $query = "SELECT id, engineer_id, start_date, end_date FROM " . $this->table_name;
        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        if ($stmt->execute()) {
            return $stmt->get_result();
        }
        error_log("Read all assignments failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    //------------------------------------------------------------------------------------------------------------------

    /**
     * Reads a single engineer assignment record by ID.
     *
     * @return array|false The assignment data as an associative array on success, false if not found.
     */
    public function readOne() {
        $query = "SELECT id, engineer_id, start_date, end_date FROM " . $this->table_name . " WHERE id = ? LIMIT 0,1";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $this->id = htmlspecialchars(strip_tags($this->id));
        $stmt->bind_param("i", $this->id);

        if ($stmt->execute()) {
            $result = $stmt->get_result();
            return $result->fetch_assoc();
        }
        error_log("Read one assignment failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    //------------------------------------------------------------------------------------------------------------------

    /**
     * Updates an existing engineer assignment record.
     *
     * @return bool True on success, false on failure.
     */
    public function update() {
        $query = "UPDATE " . $this->table_name . "
                  SET engineer_id=?, start_date=?, end_date=?
                  WHERE id=?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize inputs
        $this->engineer_id = htmlspecialchars(strip_tags($this->engineer_id));
        $this->start_date = htmlspecialchars(strip_tags($this->start_date));
        $this->end_date = htmlspecialchars(strip_tags($this->end_date));
        $this->id = htmlspecialchars(strip_tags($this->id));

        $stmt->bind_param("issi", $this->engineer_id, $this->start_date, $this->end_date, $this->id);

        if ($stmt->execute()) {
            return true;
        }
        error_log("Assignment update failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    //------------------------------------------------------------------------------------------------------------------

    /**
     * Deletes an engineer assignment record by ID.
     *
     * @return bool True on success, false on failure.
     */
    public function delete() {
        $query = "DELETE FROM " . $this->table_name . " WHERE id = ?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $this->id = htmlspecialchars(strip_tags($this->id));
        $stmt->bind_param("i", $this->id);

        if ($stmt->execute()) {
            return true;
        }
        error_log("Assignment deletion failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    //------------------------------------------------------------------------------------------------------------------

    /**
     * Reads a single engineer assignment record where the current date falls between start and end dates.
     *
     * @return array|false The assignment data as an associative array on success, false if not found.
     */
    public function readEngineerByCurrentDate() {
        $query = "SELECT engineer_id FROM " . $this->table_name . " WHERE CURDATE() BETWEEN start_date AND end_date LIMIT 0,1";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        if ($stmt->execute()) {
            $result = $stmt->get_result();
            return $result->fetch_assoc();
        }
        error_log("Read engineer by current date failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }
}