<?php
// models/Company.php

/**
 * Company Model for Rehlko Customer Care application.
 * Manages CRUD operations and data interaction for the 'Company' table.
 */
class Company {
    private $conn;
    private $table_name = "company";

    // Object properties
    public $id;
    public $company_name;
    public $company_address;
    public $company_phone;
    public $created_time;
    public $modified_time;
   
    /**
     * Constructor for the Country model.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
    }

    /**
     * Creates a new country record.
     *
     * @return bool True on success, false on failure.
     */
    public function create() {
        $query = "INSERT INTO " . $this->table_name . "
                  SET company_name=?, company_address=?, company_phone=?,company_email=?,country_id=?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize inputs
        $this->company_name = htmlspecialchars(strip_tags($this->company_name));
        $this->company_address = htmlspecialchars(strip_tags($this->company_address));
        $this->company_phone = htmlspecialchars(strip_tags($this->company_phone));
        $this->company_email = htmlspecialchars(strip_tags($this->company_email));
        $this->country_id = htmlspecialchars(strip_tags($this->country_id));

        $stmt->bind_param("ssssi", $this->company_name, $this->company_address, $this->company_phone, $this->company_email, $this->country_id);

        if ($stmt->execute()) {
            $this->id = $this->conn->insert_id;
            return true;
        }
        error_log("Country creation failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Reads all country records.
     *
     * @return mysqli_result|false The result set on success, false on failure.
     */
    public function readAll() {
        $query = "SELECT id, company_name, company_address, company_phone,company_email,country_id, created_time, modified_time
                  FROM " . $this->table_name . "
                  ORDER BY company_name ASC";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        if ($stmt->execute()) {
            return $stmt->get_result();
        }
        error_log("Read all countries failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Reads a single country record by ID.
     *
     * @return array|false The country data as an associative array on success, false if not found.
     */
    public function readOne() {
        $query = "SELECT id, company_name, company_address, company_phone, created_time, modified_time
                  FROM " . $this->table_name . "
                  WHERE id = ? LIMIT 0,1";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $this->id = htmlspecialchars(strip_tags($this->id));
        $stmt->bind_param("i", $this->id);

        if ($stmt->execute()) {
            $result = $stmt->get_result();
            return $result->fetch_assoc();
        }
        error_log("Read one country failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Updates an existing country record.
     *
     * @return bool True on success, false on failure.
     */
    public function update() {
        $query = "UPDATE " . $this->table_name . "
                  SET company_name=?, company_address=?, company_phone=?,company_email=?,country_id=?
                  WHERE id=?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize inputs
        $this->company_name = htmlspecialchars(strip_tags($this->company_name));
        $this->company_address = htmlspecialchars(strip_tags($this->company_address));
        $this->company_phone = htmlspecialchars(strip_tags($this->company_phone));
        $this->company_email = htmlspecialchars(strip_tags($this->company_email));
        $this->country_id = htmlspecialchars(strip_tags($this->country_id));
        $this->id = htmlspecialchars(strip_tags($this->id));

        $stmt->bind_param("ssssii", $this->company_name, $this->company_address, $this->company_phone,$this->company_email,$this->country_id, $this->id);

        if ($stmt->execute()) {
            return true;
        }
        error_log("Country update failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }
    
    public function readByCustomer($customer_id) {
        $query = "SELECT *
                  FROM customer_relationship cr
                  LEFT JOIN company c ON cr.company_id=c.id
                  WHERE cr.customer_id = ? ";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }
       
        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

         $customer_id = htmlspecialchars(strip_tags($customer_id));
        $stmt->bind_param("i", $customer_id);

        if ($stmt->execute()) {
            $result = $stmt->get_result();
            return $result->fetch_assoc();
        }
        error_log("Read one company failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Deletes a country record by ID.
     *
     * @return bool True on success, false on failure.
     */
    public function delete() {
        $query = "DELETE FROM " . $this->table_name . " WHERE id = ?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $this->id = htmlspecialchars(strip_tags($this->id));
        $stmt->bind_param("i", $this->id);

        if ($stmt->execute()) {
            return true;
        }
        error_log("Country deletion failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }
}
?>
