<?php
// models/CorrectiveAction.php

/**
 * CorrectiveAction Model for Rehlko Customer Care application.
 * Manages CRUD operations and data interaction for the 'corrective_action' table.
 */
class CorrectiveAction {
    private $conn;
    private $table_name = "corrective_action";

    // Object properties
    public $id;
    public $report_index;
    public $corrective_action_title;
    public $corrective_action_media_type;
    public $corrective_action_media_path;
    public $corrective_action_media_name;
    public $created_user;
    public $created_datetime;
    public $modified_datetime;

    /**
     * Constructor for the CorrectiveAction model.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
    }

    /**
     * Creates a new corrective action record.
     *
     * @return bool True on success, false on failure.
     */
    public function create() {
        $query = "INSERT INTO " . $this->table_name . "
                  SET report_index=?, corrective_action_title=?, corrective_action_media_type=?,
                      corrective_action_media_path=?, corrective_action_media_name=?, created_user=?, created_datetime=CURRENT_TIMESTAMP()";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed for create corrective action: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize data
        $this->report_index = htmlspecialchars(strip_tags($this->report_index));
        $this->corrective_action_title = htmlspecialchars(strip_tags($this->corrective_action_title));
        $this->corrective_action_media_type = htmlspecialchars(strip_tags($this->corrective_action_media_type ?? ''));
        $this->corrective_action_media_path = htmlspecialchars(strip_tags($this->corrective_action_media_path ?? ''));
        $this->corrective_action_media_name = htmlspecialchars(strip_tags($this->corrective_action_media_name ?? ''));
        $this->created_user = htmlspecialchars(strip_tags($this->created_user));

        $stmt->bind_param("sssssi",
            $this->report_index,
            $this->corrective_action_title,
            $this->corrective_action_media_type,
            $this->corrective_action_media_path,
            $this->corrective_action_media_name,
            $this->created_user
        );

        if ($stmt->execute()) {
            $this->id = $this->conn->insert_id; // Get the ID of the newly created record
            return true;
        } else {
            error_log("Create corrective action failed: (" . $stmt->errno . ") " . $stmt->error);
            return false;
        }
    }

    /**
     * Reads a single corrective action record by ID.
     *
     * @return array|null The corrective action data if found, otherwise null.
     */
    public function readOne() {
        $query = "SELECT id, report_index, corrective_action_title, corrective_action_media_type,
                         corrective_action_media_path, corrective_action_media_name, created_user,
                         created_datetime, modified_datetime
                  FROM " . $this->table_name . "
                  WHERE id = ?
                  LIMIT 0,1";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed for readOne corrective action: (" . $this->conn->errno . ") " . $this->conn->error);
            return null;
        }

        $this->id = htmlspecialchars(strip_tags($this->id));
        $stmt->bind_param("i", $this->id);
        $stmt->execute();
        $result = $stmt->get_result();
        return $result->fetch_assoc();
    }

    /**
     * Reads all corrective actions for a given report index.
     *
     * @param int $report_index The index of the report.
     * @return mysqli_result The result set of corrective actions.
     */
    public function readAllByReportIndex($report_index) {
        $query = "SELECT ca.id, ca.report_index, ca.corrective_action_title, ca.corrective_action_media_type,
                         ca.corrective_action_media_path, ca.corrective_action_media_name, ca.created_user,u.fullname as created_user_name,
                         ca.created_datetime, ca.modified_datetime
                  FROM " . $this->table_name . " AS ca
                  INNER JOIN users AS u ON ca.created_user = u.id
                  WHERE ca.report_index = ?
                  ORDER BY ca.created_datetime DESC";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed for readAllByReportIndex corrective action: (" . $this->conn->errno . ") " . $this->conn->error);
            return null; // Or throw an exception
        }

        $report_index = htmlspecialchars(strip_tags($report_index));
        $stmt->bind_param("s", $report_index);
        $stmt->execute();
        return $stmt->get_result();
    }
    
    public function readAllByReportIndexPDF($reportIndex) {
    $query = "SELECT ca.id, ca.report_index, ca.corrective_action_title, ca.corrective_action_media_type,
                         ca.corrective_action_media_path, ca.corrective_action_media_name, ca.created_user,u.fullname as created_user_name,
                         ca.created_datetime, ca.modified_datetime
                  FROM " . $this->table_name . " AS ca
                  INNER JOIN users AS u ON ca.created_user = u.id
                  WHERE ca.report_index = ?
                  ORDER BY ca.created_datetime DESC";

    $stmt = $this->conn->prepare($query);
    if ($stmt === false) {
        error_log("Prepare failed for readAllByReportIndex: (" . $this->conn->errno . ") " . $this->conn->error);
        return false;
    }

    $stmt->bind_param("s", $reportIndex);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows > 0) {
        $correctiveActions = [];
        while ($row = $result->fetch_assoc()) {
            $correctiveActions[] = $row;
        }
        return $correctiveActions;
    } else {
        return [];
    }
}

    /**
     * Updates a corrective action record.
     *
     * @return bool True on success, false on failure.
     */
    public function update() {
        $query = "UPDATE " . $this->table_name . "
                  SET report_index=?, corrective_action_title=?, corrective_action_media_type=?,
                      corrective_action_media_path=?, corrective_action_media_name=?, created_user=?, modified_datetime=CURRENT_TIMESTAMP()
                  WHERE id = ?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed for update corrective action: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize data
        $this->report_index = htmlspecialchars(strip_tags($this->report_index));
        $this->corrective_action_title = htmlspecialchars(strip_tags($this->corrective_action_title));
        $this->corrective_action_media_type = htmlspecialchars(strip_tags($this->corrective_action_media_type ?? ''));
        $this->corrective_action_media_path = htmlspecialchars(strip_tags($this->corrective_action_media_path ?? ''));
        $this->corrective_action_media_name = htmlspecialchars(strip_tags($this->corrective_action_media_name ?? ''));
        $this->modified_user = htmlspecialchars(strip_tags($this->modified_user));
        $this->id = htmlspecialchars(strip_tags($this->id));


        $stmt->bind_param("sssssii",
            $this->report_index,
            $this->corrective_action_title,
            $this->corrective_action_media_type,
            $this->corrective_action_media_path,
            $this->corrective_action_media_name,
            $this->modified_user,
            $this->id
        );

        if ($stmt->execute()) {
            return true;
        } else {
            error_log("Update corrective action failed: (" . $stmt->errno . ") " . $stmt->error);
            return false;
        }
    }

    /**
     * Deletes a corrective action record and its associated media file.
     *
     * @param int $id The ID of the corrective action to delete.
     * @return bool True on success, false on failure.
     */
    public function delete($id) {
        // First, get the file path to delete the actual file
        $query = "SELECT corrective_action_media_path FROM " . $this->table_name . " WHERE id = ?";
        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed for delete corrective action (read media path): (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }
        $this->id = htmlspecialchars(strip_tags($id));
        $stmt->bind_param("i", $this->id);
        $stmt->execute();
        $result = $stmt->get_result();
        $row = $result->fetch_assoc();

        $mediaPath = null;
        if ($row && !empty($row['corrective_action_media_path'])) {
            $mediaPath = __DIR__ . '/../public/' . $row['corrective_action_media_path']; // Full path
        }

        // Delete the database record
        $deleteQuery = "DELETE FROM " . $this->table_name . " WHERE id = ?";
        $deleteStmt = $this->conn->prepare($deleteQuery);
        if ($deleteStmt === false) {
            error_log("Prepare failed for delete corrective action (delete record): (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }
        $deleteStmt->bind_param("i", $this->id);

        if ($deleteStmt->execute()) {
            // If DB record deleted, attempt to delete the file if it exists
            if ($mediaPath && file_exists($mediaPath)) {
                if (unlink($mediaPath)) {
                    return true;
                } else {
                    error_log("Failed to delete corrective action media file: " . $mediaPath);
                    return false; // DB record deleted, but file failed
                }
            }
            return true; // DB record deleted, no file or file didn't exist
        } else {
            error_log("Corrective action deletion failed: (" . $deleteStmt->errno . ") " . $deleteStmt->error);
            return false;
        }
    }
}
?>