<?php
// models/CustomerProduct.php

/**
 * CustomerProduct Model for Rehlko Customer Care application.
 * Manages CRUD operations and data interaction for the 'customer_products' table.
 */
class CustomerProduct {
    private $conn;
    private $table_name = "customer_products";

    // Object properties
    public $id;
    public $generator_serial_number;
    public $customer_id;
    public $level;
    public $gantry;
    public $created_time;
    public $modified_time;

    /**
     * Constructor for the CustomerProduct model.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
    }

    /**
     * Checks if a product with the given generator serial number is already assigned.
     *
     * @param string $generatorSerialNumber The generator serial number to check.
     * @return array|false The existing assigned product data if found, false otherwise.
     */
    public function getAssignedProductByGeneratorSerialNumber($generatorSerialNumber) {
        $query = "SELECT id, generator_serial_number, customer_id, level, gantry FROM " . $this->table_name . " WHERE generator_serial_number = ? LIMIT 0,1";
        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed (getAssignedProductByGeneratorSerialNumber): (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }
        $stmt->bind_param("s", $generatorSerialNumber);
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            return $result->fetch_assoc();
        }
        error_log("getAssignedProductByGeneratorSerialNumber failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Assigns a product to a customer or updates an existing assignment.
     *
     * @return bool True on success, false on failure.
     */
    public function assignProduct() {
        // Sanitize inputs first, as they are used in the pre-check
        $this->generator_serial_number = htmlspecialchars(strip_tags($this->generator_serial_number));
        $this->customer_id = htmlspecialchars(strip_tags($this->customer_id));
        $this->level = htmlspecialchars(strip_tags($this->level));
        $this->gantry = htmlspecialchars(strip_tags($this->gantry));

        // Check if this generator serial number is already assigned
        $existingAssignment = $this->getAssignedProductByGeneratorSerialNumber($this->generator_serial_number);

        // Start a transaction for atomicity
        $this->conn->begin_transaction();

        try {
            if ($existingAssignment) {
                // If the product is already assigned, update the existing record
                $this->id = $existingAssignment['id']; // Set the ID for the update method
                $query = "UPDATE " . $this->table_name . "
                          SET customer_id=?, level=?, gantry=?
                          WHERE generator_serial_number=?"; // Update based on serial number

                $stmt = $this->conn->prepare($query);
                if ($stmt === false) {
                    throw new Exception("Prepare statement failed for updating customer_products: (" . $this->conn->errno . ") " . $this->conn->error);
                }

                $stmt->bind_param("isss",
                    $this->customer_id,
                    $this->level,
                    $this->gantry,
                    $this->generator_serial_number // Bind generator_serial_number for WHERE clause
                );

                if (!$stmt->execute()) {
                    throw new Exception("Customer product update failed: (" . $stmt->errno . ") " . $stmt->error);
                }
            } else {
                // If the product is not yet assigned, insert a new record
                $query = "INSERT INTO " . $this->table_name . "
                          SET generator_serial_number=?, customer_id=?, level=?, gantry=?";

                $stmt = $this->conn->prepare($query);
                if ($stmt === false) {
                    throw new Exception("Prepare statement failed for inserting customer_products: (" . $this->conn->errno . ") " . $this->conn->error);
                }

                $stmt->bind_param("siss",
                    $this->generator_serial_number,
                    $this->customer_id,
                    $this->level,
                    $this->gantry
                );

                if (!$stmt->execute()) {
                    throw new Exception("Customer product assignment failed hh: (" . $stmt->errno . ") " . $stmt->error);
                }
                $this->id = $this->conn->insert_id; // Get the ID of the newly assigned product
            }

            // Commit transaction if all operations are successful
            $this->conn->commit();
            return true;

        } catch (Exception $e) {
            // Rollback transaction on any error
            $this->conn->rollback();
            error_log("Customer Product assignment failed jjj: " . $e->getMessage());
            return false;
        }
    }


    /**
     * Reads all customer product records, with role-based filtering.
     *
     * @return mysqli_result|false The result set on success, false on failure.
     */
    public function readAll() {
        // Ensure helper functions are available (e.g., via includes/auth_helpers.php)
        // If not, you'll need to pass current user role and ID as arguments
        if (!function_exists('getCurrentUserRole') || !function_exists('getCurrentUserId')) {
            error_log("Missing getCurrentUserRole or getCurrentUserId functions.");
            // Fallback or throw an error indicating misconfiguration
            return false;
        }

        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        // Base query to select assigned product details, customer username, and product model
        $query = "SELECT cp.id, cp.generator_serial_number, cp.customer_id, cp.level, cp.gantry,
                             c.company_name AS customer_username, p.generator_model_number
                     FROM " . $this->table_name . " cp
                     LEFT JOIN company c ON cp.customer_id = c.id
                     LEFT JOIN products p ON cp.generator_serial_number = p.generator_serial_number";

        $whereClauses = [];
        $bindParams = [];
        $bindParamTypes = '';

        if (in_array($currentUserRole, ['superadmin', 'admin', 'supervisor'])) {
            // Superadmin, Admin, Supervisor can read all customer products
            // No additional WHERE clause needed
        } elseif (in_array($currentUserRole, ['localcustomer', 'globalcustomer'])) {
            // Local/Global customers only read their assigned products
            $whereClauses[] = "cp.customer_id = ?";
            $bindParams[] = $currentUserId;
            $bindParamTypes .= 'i';
        } elseif (in_array($currentUserRole, ['engineer', 'champion', 'member'])) {
            // Engineers, champions, members only read products of customers who are in their support team
            // This requires joining with the support_team table
            
        } else {
            // Default: Unauthorized to read customer products
            // This should ideally be handled by the controller to send the JSON response
            // but returning false here will signal an issue.
            error_log("Unauthorized role attempted to read customer products: $currentUserRole");
            return false;
        }

        if (!empty($whereClauses)) {
            $query .= " WHERE " . implode(" AND ", $whereClauses);
        }

        $query .= " ORDER BY cp.created_time DESC"; // Assuming created_time column exists

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed (readAll customer products): (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        if (!empty($bindParams)) {
            // Use call_user_func_array for dynamic bind_param
            call_user_func_array([$stmt, 'bind_param'], array_merge([$bindParamTypes], $this->refValues($bindParams)));
        }

        if ($stmt->execute()) {
            return $stmt->get_result();
        }
        error_log("Read all customer products failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Reads a single customer product record by ID.
     * Includes authorization checks based on user role and support team relationships.
     *
     * @return array|false The customer product data as an associative array on success, false if not found or unauthorized.
     */
    public function readOne() {
        // Ensure helper functions are available
        if (!function_exists('getCurrentUserRole') || !function_exists('getCurrentUserId') || !function_exists('sendJsonResponse')) {
            error_log("Missing required helper functions for readOne customer product.");
            return false;
        }

        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        $query = "SELECT cp.id, cp.generator_serial_number, cp.customer_id, cp.level, cp.gantry,
                             u.username AS customer_username, p.generator_model_number
                     FROM " . $this->table_name . " cp
                     LEFT JOIN users u ON cp.customer_id = u.id
                     LEFT JOIN products p ON cp.generator_serial_number = p.generator_serial_number
                     WHERE cp.id = ? LIMIT 0,1";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed (readOne customer product): (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $this->id = htmlspecialchars(strip_tags($this->id));
        $stmt->bind_param("i", $this->id);

        if ($stmt->execute()) {
            $result = $stmt->get_result();
            $customerProduct = $result->fetch_assoc();

            if ($customerProduct) {
                // Authorization check:
                // Superadmin/Admin/Supervisor can read any customer product.
                // Customers can only read their own assigned products.
                // Support team members can only read products assigned to customers in their team.
                if (in_array($currentUserRole, ['superadmin', 'admin', 'supervisor']) ||
                    $customerProduct['customer_id'] == $currentUserId ||
                    (in_array($currentUserRole, ['engineer', 'champion', 'member']) &&
                     $this->isCustomerInSupportTeam($customerProduct['customer_id'], $currentUserId))) {
                    return $customerProduct;
                } else {
                    // Unauthorized access
                    sendJsonResponse(['message' => 'Unauthorized to view this customer product.'], 403);
                    return false;
                }
            } else {
                // Product not found
                return false;
            }
        }
        error_log("Read one customer product failed or not found: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Helper function to check if a customer is in the current user's support team.
     *
     * @param int $customerId The customer ID to check.
     * @param int $currentUserId The current logged-in user ID.
     * @return bool True if in support team, false otherwise.
     */
    private function isCustomerInSupportTeam($customerId, $currentUserId) {
        $query = "SELECT COUNT(*) FROM support_team
                  WHERE customer_id = ? AND (engineer_id = ? OR champion_id = ? OR member_id = ?)";
        $stmt = $this->conn->prepare($query);
        if ($stmt === false) { error_log("isCustomerInSupportTeam prepare failed: " . $this->conn->error); return false; }
        $stmt->bind_param("iiii", $customerId, $currentUserId, $currentUserId, $currentUserId);
        $stmt->execute();
        $count = $stmt->get_result()->fetch_row()[0];
        return $count > 0;
    }

    /**
     * Retrieves a customer product record by generator serial number.
     * Used internally by cron jobs or other models.
     *
     * @param string $generatorSerialNumber The generator serial number.
     * @return array|false The customer product data on success, false if not found.
     */
    public function getByGeneratorSerialNumber($generatorSerialNumber) {
        $query = "SELECT id, generator_serial_number, customer_id, level, gantry
                  FROM " . $this->table_name . "
                  WHERE generator_serial_number = ? LIMIT 0,1";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed (getByGeneratorSerialNumber): (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $stmt->bind_param("s", $generatorSerialNumber);
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            return $result->fetch_assoc();
        }
        error_log("Get by generator serial number failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }


    /**
     * Updates an existing customer product record.
     *
     * @return bool True on success, false on failure.
     */
    public function update() {
        $query = "UPDATE " . $this->table_name . "
                  SET generator_serial_number=?, customer_id=?, level=?, gantry=?
                  WHERE id=?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed (update customer product): (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize inputs
        $this->generator_serial_number = htmlspecialchars(strip_tags($this->generator_serial_number));
        $this->customer_id = htmlspecialchars(strip_tags($this->customer_id));
        $this->level = htmlspecialchars(strip_tags($this->level));
        $this->gantry = htmlspecialchars(strip_tags($this->gantry));
        $this->id = htmlspecialchars(strip_tags($this->id));

        $stmt->bind_param("sissi",
            $this->generator_serial_number,
            $this->customer_id,
            $this->level,
            $this->gantry,
            $this->id
        );

        if ($stmt->execute()) {
            return true;
        }
        error_log("Customer product update failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Deletes a customer product record by ID.
     *
     * @return bool True on success, false on failure.
     */
    public function delete() {
        // Step 1: Fetch the generator_serial_number for the given ID.
        $query_fetch = "SELECT generator_serial_number FROM " . $this->table_name . " WHERE id = ? LIMIT 1";
        $stmt_fetch = $this->conn->prepare($query_fetch);
        if ($stmt_fetch === false) {
            error_log("Prepare failed (fetch serial number): (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $id = htmlspecialchars(strip_tags($this->id));
        $stmt_fetch->bind_param("i", $id);
        $stmt_fetch->execute();
        $result = $stmt_fetch->get_result();

        $fetched_serial_number = null;
        if ($row = $result->fetch_assoc()) {
            $fetched_serial_number = $row['generator_serial_number'];
        }
        $stmt_fetch->close();

        // If no serial number was found, we cannot proceed.
        if ($fetched_serial_number === null) {
            error_log("No record found with ID: " . $id);
            return false;
        }

        // Step 2: Update the 'is_assigned' status in the 'products' table.
        $query_update = "UPDATE products SET is_assigned = 0 WHERE generator_serial_number = ?";
        $stmt_update = $this->conn->prepare($query_update);
        if ($stmt_update === false) {
            error_log("Prepare failed (update products table): (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $stmt_update->bind_param("s", $fetched_serial_number);
        if (!$stmt_update->execute()) {
            error_log("Product update failed: (" . $stmt_update->errno . ") " . $stmt_update->error);
            $stmt_update->close();
            return false;
        }
        $stmt_update->close();
        
        $query_delete = "DELETE FROM  maintenance_plan WHERE generator_serial_number = ?";
        $stmt_delete = $this->conn->prepare($query_delete);
        if ($stmt_delete === false) {
            error_log("Prepare failed (delete maintenance plan): (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $stmt_delete->bind_param("s", $fetched_serial_number);
        if (!$stmt_delete->execute()) {
            error_log("Maintenane Plan delete failed: (" . $stmt_update->errno . ") " . $stmt_update->error);
            $stmt_delete->close();
            return false;
        }
        $stmt_delete->close();
        
        // Step 3: Delete the original record from the 'customer_products' table.
        $query_delete = "DELETE FROM " . $this->table_name . " WHERE id = ?";
        $stmt_delete = $this->conn->prepare($query_delete);
        if ($stmt_delete === false) {
            error_log("Prepare failed (delete customer product): (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $stmt_delete->bind_param("i", $id);
        if ($stmt_delete->execute()) {
            $stmt_delete->close();
            return true;
        }

        error_log("Customer product deletion failed: (" . $stmt_delete->errno . ") " . $stmt_delete->error);
        $stmt_delete->close();
        return false;
    }

    /**
     * Helper function to pass parameters by reference for bind_param.
     *
     * @param array $arr The array of parameters.
     * @return array The array with values passed by reference.
     */
    private function refValues($arr) {
        if (strnatcmp(phpversion(),'5.3') >= 0) // PHP 5.3+
        {
            $refs = array();
            foreach($arr as $key => $value)
                $refs[$key] = &$arr[$key];
            return $refs;
        }
        return $arr;
    }
    public function readByCustomer($customerId, $filterStatus = null, $searchTerm = null) {
        $query = "SELECT cp.id, cp.generator_serial_number, cp.customer_id, cp.level, cp.gantry,
                                 c.company_name AS customer_username, p.generator_model_number
                           FROM customer_relationship cr
                           LEFT JOIN users u ON cr.customer_id = u.id
                           LEFT JOIN customer_products cp ON cp.customer_id = cr.company_id
                           LEFT JOIN company c ON cp.customer_id = c.id
                           LEFT JOIN products p ON cp.generator_serial_number = p.generator_serial_number
                           WHERE cr.customer_id = ?"; // Always filter by customer_id

        $whereClauses = [];
        $bindParams = [$customerId];
        $bindParamTypes = 'i';

        // Removed $filterStatus as it's not typical for customer_products.
        // If your customer_products table has a status, you'd add it here.

        if ($searchTerm) {
            $searchTerm = '%' . $searchTerm . '%';
            // Search against relevant columns in customer_products and joined products table
            $whereClauses[] = "(cp.generator_serial_number LIKE ? OR p.generator_model_number LIKE ?)";
            $bindParams[] = $searchTerm;
            $bindParams[] = $searchTerm;
            $bindParamTypes .= 'ss'; // Two 's' for the two string parameters
        }

        if (!empty($whereClauses)) {
            $query .= " AND " . implode(" AND ", $whereClauses);
        }

        // Order by a relevant timestamp from customer_products table
        $query .= " ORDER BY cp.created_time DESC"; // Assuming 'created_time' exists in customer_products

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed for readByCustomer (CustomerProduct): (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        if (!empty($bindParams)) {
            // Use call_user_func_array for dynamic bind_param if PHP version < 5.6
            // For PHP 5.6+ and above, ...$bindParams works directly.
            // If you are using PHP < 5.6, you might need a helper function like refValues from previous examples.
            $stmt->bind_param($bindParamTypes, ...$bindParams);
        }

        if ($stmt->execute()) {
            return $stmt->get_result();
        }
        error_log("Read customer products by customer failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }
    
    public function readBySupportTeam($engineerId, $filterStatus = null, $searchTerm = null) {
        // SELECT query to fetch customer product details along with customer username and product model number
        $query = "SELECT cp.id, cp.generator_serial_number, cp.customer_id, cp.level, cp.gantry,
                             u.username AS customer_username, p.generator_model_number
                        FROM " . $this->table_name . " cp
                        LEFT JOIN users u ON cp.customer_id = u.id
                        LEFT JOIN products p ON cp.generator_serial_number = p.generator_serial_number
                        LEFT JOIN support_team st ON cp.customer_id = st.customer_id "; // Link customer products to support team via customer_id

        $whereClauses = ["st.engineer_id = ?"]; // Filter by the engineer's user_id in the support_team table
        $bindParams = [$engineerId];
        $bindParamTypes = 'i'; // 'i' for integer (engineerId)

        if ($filterStatus) {
            // Assuming 'status' is a column in the 'customer_products' table.
            // If 'status' is in a 'reports' table, you would need to LEFT JOIN 'reports' and filter on 'r.status'.
            $bindParams[] = $filterStatus;
            $bindParamTypes .= 's'; // 's' for string (filterStatus)
        }

        if ($searchTerm) {
            $searchTerm = '%' . $searchTerm . '%';
            // Search against relevant columns in customer_products and joined products table
            $whereClauses[] = "(cp.generator_serial_number LIKE ? OR p.generator_model_number LIKE ?)";
            $bindParams[] = $searchTerm;
            $bindParams[] = $searchTerm;
            $bindParamTypes .= 'ss'; // Two 's' for the two string parameters
        }

        // Add WHERE keyword and concatenate all conditions
        if (!empty($whereClauses)) {
            $query .= " WHERE " . implode(" AND ", $whereClauses);
        }

        // Order by a relevant timestamp from customer_products table
        $query .= " ORDER BY cp.created_time DESC"; // Assuming 'created_datetime' exists in customer_products

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            // Log error if prepare fails
            error_log("Prepare failed for readBySupportTeam (CustomerProduct): (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Bind parameters dynamically
        if (!empty($bindParams)) {
            // Using the splat operator (...) for PHP 5.6+ for direct parameter binding.
            // If using older PHP, you might need call_user_func_array with references.
            $stmt->bind_param($bindParamTypes, ...$bindParams);
        }

        if ($stmt->execute()) {
            return $stmt->get_result();
        }

        // Log error if execute fails
        error_log("Read customer products by support team failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }
}