<?php
// models/Maintenance.php

/**
 * Maintenance Model for Rehlko Customer Care application.
 * Manages data interaction for the 'maintenance' table.
 */
class Maintenance {
    private $conn;
    private $table_name = "maintenance";

    // Object properties
    public $id;
    public $maintenance_plan_id;
    public $generator_serial_number;
    public $maintenance_title;
    public $maintenance_description;
    public $maintenance_type;
    public $maintenance_check_user;
    public $maintenance_date;
    public $is_check; // New property
    public $maintenance_check_date; // New property
    public $last_engineer_reminder_sent_at; // New property for tracking reminders
    public $modified_time; // New property for tracking updates
    public $last_customer_notified_at; // New property for tracking customer notifications

    /**
     * Constructor for the Maintenance model.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
    }

    //------------------------------------------------------------------------------------------------------------------

    /**
     * Creates a new maintenance record.
     *
     * @return bool True on success, false on failure.
     */
    public function create() {
        $query = "INSERT INTO " . $this->table_name . "
            SET maintenance_plan_id=?,
                generator_serial_number=?,
                maintenance_title=?,
                maintenance_description=?,
                maintenance_type=?,
                maintenance_check_user=?,
                maintenance_date=NOW(),
                maintenance_check_date = CASE maintenance_type
                                             WHEN 'monthly' THEN NOW() + INTERVAL 1 MONTH
                                             WHEN 'quarterly' THEN NOW() + INTERVAL 3 MONTH
                                             WHEN 'annually' THEN NOW() + INTERVAL 1 YEAR
                                             ELSE NULL
                                           END";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize inputs
        $this->maintenance_plan_id = htmlspecialchars(strip_tags($this->maintenance_plan_id));
        $this->generator_serial_number = htmlspecialchars(strip_tags($this->generator_serial_number));
        $this->maintenance_title = htmlspecialchars(strip_tags($this->maintenance_title));
        $this->maintenance_description = htmlspecialchars(strip_tags($this->maintenance_description));
        $this->maintenance_type = htmlspecialchars(strip_tags($this->maintenance_type));
        $this->maintenance_check_user = htmlspecialchars(strip_tags($this->maintenance_check_user));

        $stmt->bind_param("isssss",
            $this->maintenance_plan_id,
            $this->generator_serial_number,
            $this->maintenance_title,
            $this->maintenance_description,
            $this->maintenance_type,
            $this->maintenance_check_user
        );

        if ($stmt->execute()) {
            $this->id = $this->conn->insert_id;
            return true;
        }
        error_log("Maintenance record creation failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    //------------------------------------------------------------------------------------------------------------------

    /**
     * Reads all maintenance records for a given plan ID.
     *
     * @param int $plan_id The ID of the maintenance plan.
     * @return mysqli_result|false The result set on success, false on failure.
     */
    public function readAllByID($plan_id) {
        $query = "SELECT id, generator_serial_number, maintenance_title, maintenance_description, maintenance_type, maintenance_check_user, maintenance_date, maintenance_check_date, is_check, remarks FROM " . $this->table_name." WHERE maintenance_plan_id=?";
        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }
        $plan_id = htmlspecialchars(strip_tags($plan_id));
        $stmt->bind_param("i", $plan_id);
        if ($stmt->execute()) {
            return $stmt->get_result();
        }
        error_log("Read all maintenance records failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }
    
    public function readAllByIDandType($plan_id, $type) {
    $query = "SELECT id, generator_serial_number, maintenance_title, maintenance_description, maintenance_type, maintenance_check_user, maintenance_date, maintenance_check_date, is_check, remarks FROM " . $this->table_name . " WHERE maintenance_plan_id=? AND maintenance_type=?";
    $stmt = $this->conn->prepare($query);
    if ($stmt === false) {
        error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
        return false;
    }

    $plan_id = htmlspecialchars(strip_tags($plan_id));
    $type = htmlspecialchars(strip_tags($type));

    // Corrected line: Bind both parameters with their correct types ('i' for integer, 's' for string)
    $stmt->bind_param("is", $plan_id, $type);

    if ($stmt->execute()) {
        return $stmt->get_result();
    }

    error_log("Read all maintenance records failed: (" . $stmt->errno . ") " . $stmt->error);
    return false;
}

    //------------------------------------------------------------------------------------------------------------------

    /**
     * Reads a single maintenance record by ID.
     *
     * @return array|false The maintenance data as an associative array on success, false if not found.
     */
    public function readOne() {
        $query = "SELECT id, generator_serial_number, maintenance_title, maintenance_description, maintenance_type, maintenance_check_user, maintenance_date, maintenance_check_date, is_check FROM " . $this->table_name . " WHERE id = ? LIMIT 0,1";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $this->id = htmlspecialchars(strip_tags($this->id));
        $stmt->bind_param("i", $this->id);

        if ($stmt->execute()) {
            $result = $stmt->get_result();
            return $result->fetch_assoc();
        }
        error_log("Read one maintenance record failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    //------------------------------------------------------------------------------------------------------------------

    /**
     * Updates an existing maintenance record.
     *
     * @return bool True on success, false on failure.
     */
    public function update() {
        $query = "UPDATE " . $this->table_name . "
            SET is_check=?,
                maintenance_date=NOW(),
                maintenance_check_date =
                    CASE (SELECT maintenance_type FROM " . $this->table_name . " WHERE id=?)
                        WHEN 'monthly' THEN NOW() + INTERVAL 1 MONTH
                        WHEN 'quarterly' THEN NOW() + INTERVAL 3 MONTH
                        WHEN 'annually' THEN NOW() + INTERVAL 1 YEAR
                        ELSE NULL
                    END,
                modified_time=NOW() -- NEW: Update modified_time on any record update
            WHERE id=?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize inputs
        $this->is_check = htmlspecialchars(strip_tags($this->is_check));
        $this->id = htmlspecialchars(strip_tags($this->id));

        $stmt->bind_param("iii",
            $this->is_check,
            $this->id,
            $this->id
        );

        if ($stmt->execute()) {
            return true;
        }
        error_log("Maintenance record update failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    //------------------------------------------------------------------------------------------------------------------

    /**
     * Deletes a maintenance record by ID.
     *
     * @return bool True on success, false on failure.
     */
    public function delete() {
        $query = "DELETE FROM " . $this->table_name . " WHERE id = ?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $this->id = htmlspecialchars(strip_tags($this->id));
        $stmt->bind_param("i", $this->id);

        if ($stmt->execute()) {
            return true;
        }
        error_log("Maintenance record deletion failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    //------------------------------------------------------------------------------------------------------------------

    /**
     * Fetches maintenance records due within a specified number of days, checking against maintenance_check_date.
     *
     * @param int $days The number of days in the future to check for upcoming maintenance.
     * @return array An array of upcoming maintenance records.
     */
    public function getUpcomingMaintenances(int $days): array {
        // We only want to send reminders once per day for an upcoming maintenance.
        // Assuming `last_engineer_reminder_sent_at` column exists (TIMESTAMP or DATETIME, nullable).
        $query = "SELECT
                    id, generator_serial_number, maintenance_title, maintenance_description,
                    maintenance_type, maintenance_check_user, maintenance_date, maintenance_check_date, is_check,
                    (SELECT engineer_id FROM assign_engineer WHERE maintenance.maintenance_check_user = assign_engineer.engineer_id LIMIT 1) AS assigned_engineer_id
                  FROM
                    " . $this->table_name . "
                  WHERE
                    maintenance_check_date BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL ? DAY)
                    AND (last_engineer_reminder_sent_at IS NULL OR DATE(last_engineer_reminder_sent_at) < CURDATE())
                  ORDER BY
                    maintenance_check_date ASC";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return [];
        }

        $stmt->bind_param("i", $days);

        if ($stmt->execute()) {
            $result = $stmt->get_result();
            return $result->fetch_all(MYSQLI_ASSOC);
        }
        error_log("Get upcoming maintenances failed: (" . $stmt->errno . ") " . $stmt->error);
        return [];
    }

    //------------------------------------------------------------------------------------------------------------------

    /**
     * Marks a maintenance record as having had an engineer reminder sent.
     * Updates the 'last_engineer_reminder_sent_at' timestamp.
     *
     * @param int $maintenanceId The ID of the maintenance record.
     * @return bool True on success, false on failure.
     */
    public function markMaintenanceReminderSent(int $maintenanceId): bool {
        $query = "UPDATE " . $this->table_name . "
                  SET last_engineer_reminder_sent_at = NOW()
                  WHERE id = ?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $stmt->bind_param("i", $maintenanceId);

        if ($stmt->execute()) {
            return true;
        }
        error_log("Mark maintenance reminder sent failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    //------------------------------------------------------------------------------------------------------------------

    /**
     * Fetches maintenance records that are due today and are currently marked as checked (is_check = 1).
     * These records need to have their is_check status reset.
     *
     * @return array An array of maintenance records to be reset.
     */
    public function getMaintenancesDueTodayAndChecked(): array {
        $query = "SELECT
                    id, generator_serial_number, maintenance_title, maintenance_check_date, is_check
                  FROM
                    " . $this->table_name . "
                  WHERE
                    maintenance_check_date = CURDATE()
                    AND is_check = 1";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return [];
        }

        if ($stmt->execute()) {
            $result = $stmt->get_result();
            return $result->fetch_all(MYSQLI_ASSOC);
        }
        error_log("Get maintenances due today and checked failed: (" . $stmt->errno . ") " . $stmt->error);
        return [];
    }

    //------------------------------------------------------------------------------------------------------------------

    /**
     * Resets the 'is_check' status of a maintenance record to 0.
     *
     * @param int $maintenanceId The ID of the maintenance record.
     * @return bool True on success, false on failure.
     */
    public function resetMaintenanceCheckStatus(int $maintenanceId): bool {
        $query = "UPDATE " . $this->table_name . "
                  SET is_check = 0
                  WHERE id = ?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $stmt->bind_param("i", $maintenanceId);

        if ($stmt->execute()) {
            return true;
        }
        error_log("Reset maintenance check status failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    //------------------------------------------------------------------------------------------------------------------

    /**
     * Fetches maintenance records that were recently updated and for which the customer hasn't been notified for this update.
     * Assumes 'modified_time' column exists and is updated on record changes, and 'last_customer_notified_at' tracks customer notifications.
     *
     * @param int $minutes The number of minutes in the past to check for recently updated maintenances.
     * @return array An array of recently updated maintenance records.
     */
    public function getRecentlyUpdatedMaintenances(int $minutes): array {
        $query = "SELECT
                    id, generator_serial_number, maintenance_title, maintenance_description,
                    maintenance_type, maintenance_check_user, maintenance_date, modified_time,
                    last_customer_notified_at
                  FROM
                    " . $this->table_name . "
                  WHERE
                    modified_time BETWEEN DATE_SUB(NOW(), INTERVAL ? MINUTE) AND NOW()
                    AND (last_customer_notified_at IS NULL OR last_customer_notified_at < modified_time)
                  ORDER BY
                    modified_time DESC";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return [];
        }

        $stmt->bind_param("i", $minutes);

        if ($stmt->execute()) {
            $result = $stmt->get_result();
            return $result->fetch_all(MYSQLI_ASSOC);
        }
        error_log("Get recently updated maintenances failed: (" . $stmt->errno . ") " . $stmt->error);
        return [];
    }

    //------------------------------------------------------------------------------------------------------------------

    /**
     * Marks a maintenance record as having had a customer notification sent for its update.
     * Updates the 'last_customer_notified_at' timestamp.
     *
     * @param int $maintenanceId The ID of the maintenance record.
     * @return bool True on success, false on failure.
     */
    public function markCustomerNotified(int $maintenanceId): bool {
        $query = "UPDATE " . $this->table_name . "
                  SET last_customer_notified_at = NOW()
                  WHERE id = ?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $stmt->bind_param("i", $maintenanceId);

        if ($stmt->execute()) {
            return true;
        }
        error_log("Mark customer notified failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }
}
