<?php
// models/MaintenancePlan.php

/**
 * MaintenancePlan Model for Rehlko Customer Care application.
 * Manages data interaction for the 'maintenance_plan' table.
 */
class MaintenancePlan {
    private $conn;
    private $table_name = "maintenance_plan"; // Note: Table name from the prompt is "maintenance_plane"

    // Object properties
    public $id;
    public $generator_serial_number;
    public $monthly;
    public $quarterly;
    public $annually;
    public $maintenance_end_date;
    public $customer_id;

    /**
     * Constructor for the MaintenancePlan model.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
    }

    //------------------------------------------------------------------------------------------------------------------

    /**
     * Creates a new maintenance plan record.
     *
     * @return bool True on success, false on failure.
     */
    public function create() {
    // 1. Update the SQL query to include the maintenance_end_date column.
    $query = "INSERT INTO " . $this->table_name . "
              SET customer_id=?,generator_serial_number=?, monthly=?, quarterly=?, annually=?, maintenance_end_date=?";

    $stmt = $this->conn->prepare($query);
    if ($stmt === false) {
        error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
        return false;
    }

    // 2. Sanitize inputs, cast booleans to integers, and prepare the new date variable.
    $this->customer_id = (int)$this->customer_id;
    $this->generator_serial_number = htmlspecialchars(strip_tags($this->generator_serial_number));
    $monthly_val = (int)$this->monthly;
    $quarterly_val = (int)$this->quarterly;
    $annually_val = (int)$this->annually;
    // Prepare the maintenance_end_date value
    $maintenance_end_date_val = $this->maintenance_end_date;

    // 3. Update bind_param to include the new 's' for the date string.
    $stmt->bind_param("isiiis",
        $this->customer_id,
        $this->generator_serial_number,
        $monthly_val,
        $quarterly_val,
        $annually_val,
        $maintenance_end_date_val
    );

    if ($stmt->execute()) {
        $this->id = $this->conn->insert_id;
        return true;
    }
    error_log("Maintenance plan creation failed: (" . $stmt->errno . ") " . $stmt->error);
    return false;
}

    //------------------------------------------------------------------------------------------------------------------

    /**
     * Reads all maintenance plan records.
     *
     * @return mysqli_result|false The result set on success, false on failure.
     */
    public function readAll($customerId) {
        if($customerId == null){
            $query = "SELECT mp.id,mp.generator_serial_number,mp.monthly, mp.quarterly, mp.annually,mp.maintenance_end_date,uc.fullname AS customer_name
                   FROM " . $this->table_name . " mp
                   LEFT JOIN customer_products cs ON mp.generator_serial_number = cs.generator_serial_number
                   LEFT JOIN users uc ON cs.customer_id = uc.id";
        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        if ($stmt->execute()) {
            return $stmt->get_result();
        }
        error_log("Read all maintenance plans failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
        }else{
            $query = "SELECT mp.id, mp.generator_serial_number, mp.monthly, mp.quarterly, mp.annually,mp.maintenance_end_date, uc.fullname AS customer_name
              FROM " . $this->table_name . " mp
              LEFT JOIN customer_products cs ON mp.generator_serial_number = cs.generator_serial_number
              LEFT JOIN users uc ON cs.customer_id = uc.id
              WHERE cs.customer_id = ?"; // Use a placeholder for the customer ID
    
    // Prepare the statement
    $stmt = $this->conn->prepare($query);
    if ($stmt === false) {
        error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
        return false;
    }

    // Bind the customer ID parameter
    $stmt->bind_param("i", $customerId); // "i" indicates an integer type

    // Execute the statement
    if ($stmt->execute()) {
        return $stmt->get_result();
    }
    
    // Log the error if execution fails
    error_log("Read all maintenance plans failed: (" . $stmt->errno . ") " . $stmt->error);
    return false;
        }
    }
    
    public function CustomerreadAll($customer_id) {
    $query = "SELECT mp.id, mp.generator_serial_number, mp.monthly, mp.quarterly, mp.annually,mp.maintenance_end_date, uc.fullname AS customer_name
              FROM " . $this->table_name . " mp
              LEFT JOIN customer_products cs ON mp.generator_serial_number = cs.generator_serial_number
              LEFT JOIN users uc ON cs.customer_id = uc.id
              WHERE cs.customer_id = ?"; // Use a placeholder for the customer ID
    
    // Prepare the statement
    $stmt = $this->conn->prepare($query);
    if ($stmt === false) {
        error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
        return false;
    }

    // Bind the customer ID parameter
    $stmt->bind_param("i", $customer_id); // "i" indicates an integer type

    // Execute the statement
    if ($stmt->execute()) {
        return $stmt->get_result();
    }
    
    // Log the error if execution fails
    error_log("Read all maintenance plans failed: (" . $stmt->errno . ") " . $stmt->error);
    return false;
}

    //------------------------------------------------------------------------------------------------------------------

    /**
     * Reads a single maintenance plan record by ID.
     *
     * @return array|false The plan data as an associative array on success, false if not found.
     */
    public function readOne() {
        $query = "SELECT id, generator_serial_number, monthly, quarterly, annually,maintenance_end_date FROM " . $this->table_name . " WHERE id = ? LIMIT 0,1";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $this->id = htmlspecialchars(strip_tags($this->id));
        $stmt->bind_param("i", $this->id);

        if ($stmt->execute()) {
            $result = $stmt->get_result();
            return $result->fetch_assoc();
        }
        error_log("Read one maintenance plan failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    //------------------------------------------------------------------------------------------------------------------

    /**
     * Updates an existing maintenance plan record.
     *
     * @return bool True on success, false on failure.
     */
    public function update() {
        $query = "UPDATE " . $this->table_name . "
                  SET generator_serial_number=?, monthly=?, quarterly=?, annually=?
                  WHERE id=?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize inputs and cast booleans to integers
        $this->generator_serial_number = htmlspecialchars(strip_tags($this->generator_serial_number));
        $monthly_val = (int)$this->monthly;
        $quarterly_val = (int)$this->quarterly;
        $annually_val = (int)$this->annually;
        $this->id = htmlspecialchars(strip_tags($this->id));

        $stmt->bind_param("siiii",
            $this->generator_serial_number,
            $monthly_val,
            $quarterly_val,
            $annually_val,
            $this->id
        );

        if ($stmt->execute()) {
            return true;
        }
        error_log("Maintenance plan update failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    //------------------------------------------------------------------------------------------------------------------

    /**
     * Deletes a maintenance plan record by ID.
     *
     * @return bool True on success, false on failure.
     */
    public function delete() {
        $query = "DELETE FROM " . $this->table_name . " WHERE id = ?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $this->id = htmlspecialchars(strip_tags($this->id));
        $stmt->bind_param("i", $this->id);

        if ($stmt->execute()) {
            return true;
        }
        error_log("Maintenance plan deletion failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }
}
?>