<?php
// models/Notification.php

/**
 * Notification Model for Rehlko Customer Care application.
 * Manages CRUD operations and data interaction for the 'notifications' table.
 * Used for in-app notifications.
 */
class Notification {
    private $conn;
    private $table_name = "notifications";

    // Object properties
    public $id;
    public $receiver_id;
    public $notifications_title;
    public $notifications_content;
    public $notifications_category; // 'users', 'reports', 'generators', 'setting', 'maintenance'
    public $is_read; // 0 or 1
    public $created_time;
    public $modified_time;

    /**
     * Constructor for the Notification model.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
    }

    /**
     * Creates a new notification record.
     *
     * @param int $receiverId The ID of the user who will receive the notification.
     * @param string $title The title of the notification.
     * @param string $content The detailed content of the notification.
     * @param string $category The category of the notification (e.g., 'reports', 'maintenance').
     * @param int $isRead Initial read status (0 for unread, 1 for read).
     * @return bool True on success, false on failure.
     */
    public function createNotification($receiverId, $title, $content, $category, $isRead = 0) {
        $query = "INSERT INTO " . $this->table_name . "
                  SET receiver_id=?, notifications_title=?, notifications_content=?,
                      notifications_category=?, is_read=?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize inputs
        $receiverId = htmlspecialchars(strip_tags($receiverId));
        $title = htmlspecialchars(strip_tags($title));
        $content = htmlspecialchars(strip_tags($content));
        $category = htmlspecialchars(strip_tags($category));
        $isRead = htmlspecialchars(strip_tags($isRead));

        $stmt->bind_param("isssi", $receiverId, $title, $content, $category, $isRead);

        if ($stmt->execute()) {
            return true;
        }
        error_log("Notification creation failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Reads all notifications for a specific receiver, with an option to filter by read status.
     *
     * @param int $receiverId The ID of the user whose notifications to retrieve.
     * @param bool $unreadOnly If true, only retrieves unread notifications.
     * @return mysqli_result|false The result set on success, false on failure.
     */
    public function readUserNotifications($receiverId, $unreadOnly = false) {
        $query = "SELECT id, receiver_id, notifications_title, notifications_content,
                         notifications_category, is_read, created_time, modified_time
                  FROM " . $this->table_name . "
                  WHERE receiver_id = ?";

        if ($unreadOnly) {
            $query .= " AND is_read = 0";
        }
        $query .= " ORDER BY created_time DESC";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $receiverId = htmlspecialchars(strip_tags($receiverId));
        $stmt->bind_param("i", $receiverId);

        if ($stmt->execute()) {
            return $stmt->get_result();
        }
        error_log("Read user notifications failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }
    public function readOne() {
        // Query to read a single record
        $query = "SELECT id, receiver_id, notifications_title, notifications_content,
                          notifications_category, is_read, created_time, modified_time
                  FROM " . $this->table_name . "
                  WHERE id = ?
                  LIMIT 0,1"; // Limit to 1 row as we're looking for a single notification

        // Prepare query statement
        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return null;
        }

        // Sanitize ID
        $this->id = htmlspecialchars(strip_tags($this->id));

        // Bind ID of notification to be updated
        $stmt->bind_param("i", $this->id);

        // Execute query
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            if ($result->num_rows > 0) {
                $row = $result->fetch_assoc();

                // Set object properties to values of the retrieved row
                $this->receiver_id = $row['receiver_id'];
                $this->notifications_title = $row['notifications_title'];
                $this->notifications_content = $row['notifications_content'];
                $this->notifications_category = $row['notifications_category'];
                $this->is_read = $row['is_read'];
                $this->created_time = $row['created_time'];
                $this->modified_time = $row['modified_time'];

                return $row; // Return the associative array of data
            }
        }
        error_log("Read one notification failed: (" . $stmt->errno . ") " . $stmt->error);
        return null; // Return null if notification not found or an error occurred
    }
    /**
     * Marks a specific notification as read.
     *
     * @param int $notificationId The ID of the notification to mark as read.
     * @return bool True on success, false on failure.
     */
    public function markAsRead($notificationId) {
        $query = "UPDATE " . $this->table_name . " SET is_read = 1 WHERE id = ?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $notificationId = htmlspecialchars(strip_tags($notificationId));
        $stmt->bind_param("i", $notificationId);

        if ($stmt->execute()) {
            return true;
        }
        error_log("Mark notification as read failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Deletes a notification record by ID.
     *
     * @return bool True on success, false on failure.
     */
    public function delete() {
        $query = "DELETE FROM " . $this->table_name . " WHERE id = ?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $this->id = htmlspecialchars(strip_tags($this->id));
        $stmt->bind_param("i", $this->id);

        if ($stmt->execute()) {
            return true;
        }
        error_log("Notification deletion failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }
}
?>
