<?php
// models/Product.php

/**
 * Product Model for Rehlko Customer Care application.
 * Manages CRUD operations and data interaction for the 'products' table.
 */
class Product {
    private $conn;
    private $table_name = "products";

    // Object properties
    public $id;
    public $generator_serial_number;
    public $generator_model_number;
    public $enigne_serial_number; // Note: Typo 'enigne' from SQL schema
    public $enigne_model_number;  // Note: Typo 'enigne' from SQL schema
    public $alt_serial_number;
    public $alt_model_number;
    public $controller_parts;
    public $controller_serial;
    public $remarks;
    public $created_time;
    public $modified_time;

    /**
     * Constructor for the Product model.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
    }

    /**
     * Creates a new product record.
     *
     * @return bool True on success, false on failure.
     */
    public function create() {
        $query = "INSERT INTO " . $this->table_name . "
                  SET generator_serial_number=?, generator_model_number=?, enigne_serial_number=?,
                      enigne_model_number=?, alt_serial_number=?, alt_model_number=?,
                      controller_parts=?, controller_serial=?, remarks=?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize inputs
        $this->generator_serial_number = htmlspecialchars(strip_tags($this->generator_serial_number));
        $this->generator_model_number = htmlspecialchars(strip_tags($this->generator_model_number));
        $this->enigne_serial_number = htmlspecialchars(strip_tags($this->enigne_serial_number));
        $this->enigne_model_number = htmlspecialchars(strip_tags($this->enigne_model_number));
        $this->alt_serial_number = htmlspecialchars(strip_tags($this->alt_serial_number));
        $this->alt_model_number = htmlspecialchars(strip_tags($this->alt_model_number));
        $this->controller_parts = htmlspecialchars(strip_tags($this->controller_parts));
        $this->controller_serial = htmlspecialchars(strip_tags($this->controller_serial));
        $this->remarks = htmlspecialchars(strip_tags($this->remarks));
        
        error_log("Sanitize Inputs for eng serial no :".$this->enigne_serial_number);
        error_log("Sanitize Inputs for eng model no :".$this->enigne_model_number);

        $stmt->bind_param("sssssssss",
            $this->generator_serial_number,
            $this->generator_model_number,
            $this->enigne_serial_number,
            $this->enigne_model_number,
            $this->alt_serial_number,
            $this->alt_model_number,
            $this->controller_parts,
            $this->controller_serial,
            $this->remarks
        );

        if ($stmt->execute()) {
            $this->id = $this->conn->insert_id;
            return true;
        }
        error_log("Product creation failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Reads all product records, with role-based filtering.
     *
     * @return mysqli_result|false The result set on success, false on failure.
     */
    public function readAll() {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        $query = "SELECT p.id, p.generator_serial_number, p.generator_model_number, p.enigne_serial_number as engine_serial_number,
                         p.enigne_model_number as engine_model_number, p.alt_serial_number, p.alt_model_number,
                         p.controller_parts, p.controller_serial as controller_serial_number, p.remarks,
                         p.created_time, p.modified_time
                  FROM " . $this->table_name . " p";

        $whereClauses = [];
        $bindParams = [];
        $bindParamTypes = '';

        if (in_array($currentUserRole, ['superadmin', 'admin', 'supervisor'])) {
            // Superadmin, Admin, Supervisor can read all products
            // No additional WHERE clause needed
        } elseif (in_array($currentUserRole, ['localcustomer', 'globalcustomer'])) {
            // Local/Global customers only read their assigned products
            $query .= " JOIN customer_products cp ON p.generator_serial_number = cp.generator_serial_number";
            $whereClauses[] = "cp.customer_id = ?";
            $bindParams[] = $currentUserId;
            $bindParamTypes .= 'i';
        } elseif (in_array($currentUserRole, ['engineer', 'champion', 'member'])) {
            // Engineers, champions, members only read products of customers in their support team
            $query .= " JOIN customer_products cp ON p.generator_serial_number = cp.generator_serial_number";
            $query .= " JOIN support_team st ON cp.customer_id = st.customer_id";
            $whereClauses[] = "(st.engineer_id = ? OR st.champion_id = ? OR st.member_id = ?)";
            $bindParams[] = $currentUserId;
            $bindParams[] = $currentUserId;
            $bindParams[] = $currentUserId;
            $bindParamTypes .= 'iii';
        } else {
            sendJsonResponse(['message' => 'Unauthorized to read products.'], 403);
        }

        if (!empty($whereClauses)) {
            $query .= " WHERE " . implode(" AND ", $whereClauses);
        }

        $query .= " ORDER BY p.created_time DESC";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        if (!empty($bindParams)) {
            $stmt->bind_param($bindParamTypes, ...$bindParams);
        }

        if ($stmt->execute()) {
            return $stmt->get_result();
        }
        error_log("Read all products failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }
    
    public function UnassignreadAll() {
    $currentUserRole = getCurrentUserRole();
    $currentUserId = getCurrentUserId();

    $query = "SELECT p.id, p.generator_serial_number, p.generator_model_number, p.enigne_serial_number,
                     p.enigne_model_number, p.alt_serial_number, p.alt_model_number,
                     p.controller_parts, p.controller_serial, p.remarks,
                     p.created_time, p.modified_time
              FROM " . $this->table_name . " p";

    $whereClauses = ["p.is_assigned = false"]; // Add the new WHERE clause here
    $bindParams = [];
    $bindParamTypes = '';

    if (in_array($currentUserRole, ['superadmin', 'admin', 'supervisor'])) {
        // Superadmin, Admin, Supervisor can read all products
        // No additional WHERE clause needed
    } elseif (in_array($currentUserRole, ['localcustomer', 'globalcustomer'])) {
        // Local/Global customers only read their assigned products
        $query .= " JOIN customer_products cp ON p.generator_serial_number = cp.generator_serial_number";
        $whereClauses[] = "cp.customer_id = ?";
        $bindParams[] = $currentUserId;
        $bindParamTypes .= 'i';
    } elseif (in_array($currentUserRole, ['engineer', 'champion', 'member'])) {
        // Engineers, champions, members only read products of customers in their support team
        $query .= " JOIN customer_products cp ON p.generator_serial_number = cp.generator_serial_number";
        $query .= " JOIN support_team st ON cp.customer_id = st.customer_id";
        $whereClauses[] = "(st.engineer_id = ? OR st.champion_id = ? OR st.member_id = ?)";
        $bindParams[] = $currentUserId;
        $bindParams[] = $currentUserId;
        $bindParams[] = $currentUserId;
        $bindParamTypes .= 'iii';
    } else {
        sendJsonResponse(['message' => 'Unauthorized to read products.'], 403);
    }

    if (!empty($whereClauses)) {
        $query .= " WHERE " . implode(" AND ", $whereClauses);
    }

    $query .= " ORDER BY p.created_time DESC";

    $stmt = $this->conn->prepare($query);
    if ($stmt === false) {
        error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
        return false;
    }

    if (!empty($bindParams)) {
        $stmt->bind_param($bindParamTypes, ...$bindParams);
    }

    if ($stmt->execute()) {
        return $stmt->get_result();
    }
    error_log("Read all products failed: (" . $stmt->errno . ") " . $stmt->error);
    return false;
}

    /**
     * Reads a single product record by ID.
     *
     * @return array|false The product data as an associative array on success, false if not found.
     */
    public function readOne() {
        $currentUserRole = getCurrentUserRole();
        $currentUserId = getCurrentUserId();

        $query = "SELECT p.id, p.generator_serial_number, p.generator_model_number, p.enigne_serial_number,
                         p.enigne_model_number, p.alt_serial_number, p.alt_model_number,
                         p.controller_parts, p.controller_serial, p.remarks,
                         p.created_time, p.modified_time
                  FROM " . $this->table_name . " p
                  WHERE p.id = ? LIMIT 0,1";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $this->id = htmlspecialchars(strip_tags($this->id));
        $stmt->bind_param("i", $this->id);

        if ($stmt->execute()) {
            $result = $stmt->get_result();
            $product = $result->fetch_assoc();

            if ($product) {
                // Apply specific authorization for reading a single product
                if (in_array($currentUserRole, ['superadmin', 'admin', 'supervisor'])) {
                    return $product;
                } elseif (in_array($currentUserRole, ['localcustomer', 'globalcustomer'])) {
                    // Check if the product is assigned to this customer
                    $checkQuery = "SELECT COUNT(*) FROM customer_products WHERE customer_id = ? AND generator_serial_number = ?";
                    $checkStmt = $this->conn->prepare($checkQuery);
                    if ($checkStmt === false) { error_log("Check prepare failed: " . $this->conn->error); return false; }
                    $checkStmt->bind_param("is", $currentUserId, $product['generator_serial_number']);
                    $checkStmt->execute();
                    $count = $checkStmt->get_result()->fetch_row()[0];
                    if ($count > 0) {
                        return $product;
                    } else {
                        sendJsonResponse(['message' => 'Unauthorized to view this product.'], 403);
                        return false;
                    }
                } elseif (in_array($currentUserRole, ['engineer', 'champion', 'member'])) {
                    // Check if the product's customer is in this user's support team
                    $checkQuery = "SELECT COUNT(*) FROM customer_products cp
                                   JOIN support_team st ON cp.customer_id = st.customer_id
                                   WHERE cp.generator_serial_number = ? AND (st.engineer_id = ? OR st.champion_id = ? OR st.member_id = ?)";
                    $checkStmt = $this->conn->prepare($checkQuery);
                    if ($checkStmt === false) { error_log("Check prepare failed: " . $this->conn->error); return false; }
                    $checkStmt->bind_param("siii", $product['generator_serial_number'], $currentUserId, $currentUserId, $currentUserId);
                    $checkStmt->execute();
                    $count = $checkStmt->get_result()->fetch_row()[0];
                    if ($count > 0) {
                        return $product;
                    } else {
                        sendJsonResponse(['message' => 'Unauthorized to view this product.'], 403);
                        return false;
                    }
                } else {
                    sendJsonResponse(['message' => 'Unauthorized to view this product.'], 403);
                    return false;
                }
            }
        }
        error_log("Read one product failed or product not found: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Reads a single product record by its generator serial number.
     * Used internally by other models/cron jobs without authorization check.
     *
     * @param string $serialNumber The generator serial number.
     * @return array|false The product data as an associative array on success, false if not found.
     */
    public function readBySerialNumber($serialNumber) {
        $query = "SELECT id, generator_serial_number, generator_model_number, enigne_serial_number,
                         enigne_model_number, alt_serial_number, alt_model_number,
                         controller_parts, controller_serial, remarks,
                         created_time, modified_time
                  FROM " . $this->table_name . "
                  WHERE generator_serial_number = ? LIMIT 0,1";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $stmt->bind_param("s", $serialNumber);
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            return $result->fetch_assoc();
        }
        error_log("Read product by serial number failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Updates an existing product record.
     *
     * @return bool True on success, false on failure.
     */
    public function update() {
        $query = "UPDATE " . $this->table_name . "
                  SET generator_serial_number=?, generator_model_number=?, enigne_serial_number=?,
                      enigne_model_number=?, alt_serial_number=?, alt_model_number=?,
                      controller_parts=?, controller_serial=?, remarks=?
                  WHERE id=?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize inputs
        $this->generator_serial_number = htmlspecialchars(strip_tags($this->generator_serial_number));
        $this->generator_model_number = htmlspecialchars(strip_tags($this->generator_model_number));
        $this->enigne_serial_number = htmlspecialchars(strip_tags($this->enigne_serial_number));
        $this->enigne_model_number = htmlspecialchars(strip_tags($this->enigne_model_number));
        $this->alt_serial_number = htmlspecialchars(strip_tags($this->alt_serial_number));
        $this->alt_model_number = htmlspecialchars(strip_tags($this->alt_model_number));
        $this->controller_parts = htmlspecialchars(strip_tags($this->controller_parts));
        $this->controller_serial = htmlspecialchars(strip_tags($this->controller_serial));
        $this->remarks = htmlspecialchars(strip_tags($this->remarks));
        $this->id = htmlspecialchars(strip_tags($this->id));

        $stmt->bind_param("sssssssssi",
            $this->generator_serial_number,
            $this->generator_model_number,
            $this->enigne_serial_number,
            $this->enigne_model_number,
            $this->alt_serial_number,
            $this->alt_model_number,
            $this->controller_parts,
            $this->controller_serial,
            $this->remarks,
            $this->id
        );

        if ($stmt->execute()) {
            return true;
        }
        error_log("Product update failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }
    
    public function assignProduct($generator_serial_number) {
    // 1. Use a placeholder '?' for the value.
    $query = "UPDATE " . $this->table_name . "
              SET is_assigned = 1
              WHERE generator_serial_number = ?";

    // 2. Prepare the statement.
    $stmt = $this->conn->prepare($query);
    if ($stmt === false) {
        error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
        return false;
    }

    // 3. Bind the parameter. The 's' indicates the parameter is a string.
    $stmt->bind_param("s", $generator_serial_number);

    // 4. Execute the statement.
    if ($stmt->execute()) {
        return true;
    }
    
    error_log("Product update failed: (" . $stmt->errno . ") " . $stmt->error);
    return false;
}

    /**
     * Deletes a product record by ID.
     *
     * @return bool True on success, false on failure.
     */
    public function delete() {
        $query = "DELETE FROM " . $this->table_name . " WHERE id = ?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $this->id = htmlspecialchars(strip_tags($this->id));
        $stmt->bind_param("i", $this->id);

        if ($stmt->execute()) {
            return true;
        }
        error_log("Product deletion failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }
}
?>
