<?php
// models/ReportWarningConfig.php

/**
 * ReportWarningConfig Model for Rehlko Customer Care application.
 * Manages data interaction for the 'report_warning_config' table,
 * which stores first and last warning times for different customer types.
 */
class ReportWarningConfig {
    private $conn;
    private $table_name = "report_warning_config";

    // Object properties
    public $id;
    public $customer_type;
    public $first_warning_hours;
    public $last_warning_hours;
    public $created_at;
    public $updated_at;

    /**
     * Constructor for the ReportWarningConfig model.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
    }

    /**
     * Creates a new warning configuration record.
     *
     * @return bool True on success, false on failure.
     */
    public function create() {
        $query = "INSERT INTO " . $this->table_name . "
                  SET customer_type=?, first_warning_hours=?, last_warning_hours=?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize inputs
        $this->customer_type = htmlspecialchars(strip_tags($this->customer_type));
        $this->first_warning_hours = htmlspecialchars(strip_tags($this->first_warning_hours));
        $this->last_warning_hours = htmlspecialchars(strip_tags($this->last_warning_hours));

        $stmt->bind_param("sii", $this->customer_type, $this->first_warning_hours, $this->last_warning_hours);

        if ($stmt->execute()) {
            $this->id = $this->conn->insert_id;
            return true;
        }
        error_log("Warning config creation failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Reads all warning configuration records.
     *
     * @return mysqli_result|false The result set on success, false on failure.
     */
    public function readAll() {
        $query = "SELECT id, customer_type, first_warning_hours, last_warning_hours, created_at, updated_at
                  FROM " . $this->table_name . "
                  ORDER BY customer_type ASC";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        if ($stmt->execute()) {
            return $stmt->get_result();
        }
        error_log("Read all warning configs failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Reads a single warning configuration record by ID.
     *
     * @return array|false The config data as an associative array on success, false if not found.
     */
    public function readOne() {
        $query = "SELECT id, customer_type, first_warning_hours, last_warning_hours, created_at, updated_at
                  FROM " . $this->table_name . "
                  WHERE id = ? LIMIT 0,1";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $this->id = htmlspecialchars(strip_tags($this->id));
        $stmt->bind_param("i", $this->id);

        if ($stmt->execute()) {
            $result = $stmt->get_result();
            return $result->fetch_assoc();
        }
        error_log("Read one warning config failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Retrieves warning configuration by customer type (e.g., 'local' or 'global').
     * Used internally by cron jobs or other logic.
     *
     * @param string $customerType The type of customer ('local' or 'global').
     * @return array|false The config data as an associative array on success, false if not found.
     */
    public function getWarningConfig($customerType) {
        $query = "SELECT id, customer_type, first_warning_hours, last_warning_hours
                  FROM " . $this->table_name . "
                  WHERE customer_type = ? LIMIT 0,1";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed for getWarningConfig: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $stmt->bind_param("s", $customerType);
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            return $result->fetch_assoc();
        }
        error_log("Get warning config by type failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Updates an existing warning configuration record.
     *
     * @return bool True on success, false on failure.
     */
    public function update() {
        $query = "UPDATE " . $this->table_name . "
                  SET customer_type=?, first_warning_hours=?, last_warning_hours=?
                  WHERE id=?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize inputs
        $this->customer_type = htmlspecialchars(strip_tags($this->customer_type));
        $this->first_warning_hours = htmlspecialchars(strip_tags($this->first_warning_hours));
        $this->last_warning_hours = htmlspecialchars(strip_tags($this->last_warning_hours));
        $this->id = htmlspecialchars(strip_tags($this->id));

        $stmt->bind_param("siii", $this->customer_type, $this->first_warning_hours, $this->last_warning_hours, $this->id);

        if ($stmt->execute()) {
            return true;
        }
        error_log("Warning config update failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Deletes a warning configuration record by ID.
     *
     * @return bool True on success, false on failure.
     */
    public function delete() {
        $query = "DELETE FROM " . $this->table_name . " WHERE id = ?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $this->id = htmlspecialchars(strip_tags($this->id));
        $stmt->bind_param("i", $this->id);

        if ($stmt->execute()) {
            return true;
        }
        error_log("Warning config deletion failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }
}
?>
