<?php
// models/Terms.php

/**
 * Terms Model for Rehlko Customer Care application.
 * Manages CRUD operations and data interaction for the 'terms' table.
 * Used for terms and policies content.
 */
class Terms {
    private $conn;
    private $table_name = "terms";

    // Object properties
    public $id;
    public $terms_category; // 'service', 'privacy', 'support', 'security', 'disclaimer'
    public $terms_title;
    public $terms_content;
    public $created_date;
    public $modified_date;

    /**
     * Constructor for the Terms model.
     *
     * @param mysqli $db The database connection object.
     */
    public function __construct($db) {
        $this->conn = $db;
    }

    /**
     * Creates a new terms record.
     *
     * @return bool True on success, false on failure.
     */
    public function create() {
        $query = "INSERT INTO " . $this->table_name . "
                  SET terms_category=?, terms_title=?, terms_content=?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize inputs
        $this->terms_category = htmlspecialchars(strip_tags($this->terms_category));
        $this->terms_title = htmlspecialchars(strip_tags($this->terms_title));
        $this->terms_content = htmlspecialchars(strip_tags($this->terms_content));

        $stmt->bind_param("sss", $this->terms_category, $this->terms_title, $this->terms_content);

        if ($stmt->execute()) {
            $this->id = $this->conn->insert_id;
            return true;
        }
        error_log("Terms creation failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Reads all terms records.
     *
     * @return mysqli_result|false The result set on success, false on failure.
     */
    public function readAll() {
        $query = "SELECT id, terms_category, terms_title, terms_content, created_date, modified_date
                  FROM " . $this->table_name . "
                  ORDER BY terms_category, terms_title ASC";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        if ($stmt->execute()) {
            return $stmt->get_result();
        }
        error_log("Read all terms failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Reads a single terms record by ID.
     *
     * @return array|false The terms data as an associative array on success, false if not found.
     */
    public function readOne() {
        $query = "SELECT id, terms_category, terms_title, terms_content, created_date, modified_date
                  FROM " . $this->table_name . "
                  WHERE id = ? LIMIT 0,1";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $this->id = htmlspecialchars(strip_tags($this->id));
        $stmt->bind_param("i", $this->id);

        if ($stmt->execute()) {
            $result = $stmt->get_result();
            return $result->fetch_assoc();
        }
        error_log("Read one terms failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Updates an existing terms record.
     *
     * @return bool True on success, false on failure.
     */
    public function update() {
        $query = "UPDATE " . $this->table_name . "
                  SET terms_category=?, terms_title=?, terms_content=?
                  WHERE id=?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        // Sanitize inputs
        $this->terms_category = htmlspecialchars(strip_tags($this->terms_category));
        $this->terms_title = htmlspecialchars(strip_tags($this->terms_title));
        $this->terms_content = htmlspecialchars(strip_tags($this->terms_content));
        $this->id = htmlspecialchars(strip_tags($this->id));

        $stmt->bind_param("sssi", $this->terms_category, $this->terms_title, $this->terms_content, $this->id);

        if ($stmt->execute()) {
            return true;
        }
        error_log("Terms update failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }

    /**
     * Deletes a terms record by ID.
     *
     * @return bool True on success, false on failure.
     */
    public function delete() {
        $query = "DELETE FROM " . $this->table_name . " WHERE id = ?";

        $stmt = $this->conn->prepare($query);
        if ($stmt === false) {
            error_log("Prepare failed: (" . $this->conn->errno . ") " . $this->conn->error);
            return false;
        }

        $this->id = htmlspecialchars(strip_tags($this->id));
        $stmt->bind_param("i", $this->id);

        if ($stmt->execute()) {
            return true;
        }
        error_log("Terms deletion failed: (" . $stmt->errno . ") " . $stmt->error);
        return false;
    }
}
?>
