<?php
// public/index.php

/**
 * Main entry point and router for the Rehlko Customer Care API.
 * This script handles all incoming HTTP requests, routes them to
 * the appropriate controller methods, and manages CORS.
 *
 * It acts as a front controller, ensuring all API calls are processed
 * through a single point for consistency and security.
 */

// Set CORS headers to allow requests from any origin.
// In a production environment, you should replace '*' with specific origins (e.g., 'https://your-flutter-app.com').
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With"); // Added X-Requested-With for broader compatibility
header("Access-Control-Max-Age: 3600"); // Cache preflight requests for 1 hour

// Handle preflight OPTIONS requests, which are sent by browsers for CORS.
// These requests should respond with 200 OK and the CORS headers, then terminate.
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200); // Explicitly set 200 OK status for OPTIONS
    exit(0); // Respond with headers, then exit
}

// Important: Removed the global "Content-Type: application/json" header here.
// This header should be set by your `sendJsonResponse` function or by specific
// controllers when generating non-JSON responses (e.g., PDF files),
// to avoid conflicts.

// Include database connection and helper functions
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../utils/helpers.php'; // This file should contain sendJsonResponse()
require_once __DIR__ . '/../utils/Mailer.php';
require_once __DIR__ . '/../utils/firebase_auth.php'; // Included the new Firebase auth helper
require_once __DIR__ . '/../utils/PdfGenerator.php'; // Required for cron jobs and report completion
// autoloader is typically for classes, if models/controllers are included manually, it might not be strictly needed here,
// but keep it if other parts of your app use it.
// require_once __DIR__ . '/../utils/autoloader.php'; 


// Manually include all model files
// This approach is used because Composer is not allowed.
// Ensure all model files exist in the 'models' directory.
foreach (glob(__DIR__ . '/../models/*.php') as $filename) {
    require_once $filename;
}

// Manually include all controller files
// This approach is used because Composer is not allowed.
// Ensure all controller files exist in the 'controllers' directory.
foreach (glob(__DIR__ . '/../controllers/*.php') as $filename) {
    require_once $filename;
}

// Parse the request URI and method
$requestUri = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
$requestMethod = $_SERVER['REQUEST_METHOD'];

// Remove the '/api/' prefix from the URI to get clean segments
// Example: '/api/users/1' becomes 'users/1'
$uriSegments = explode('/', trim($requestUri, '/'));
$apiBaseIndex = array_search('new', $uriSegments); // Find 'api' segment
if ($apiBaseIndex !== false) {
    $uriSegments = array_slice($uriSegments, $apiBaseIndex + 1);
}

// Determine controller and method based on URI segments
$controllerSegment = $uriSegments[0] ?? 'NotFound';
$methodSegment = $uriSegments[1] ?? 'index'; // Default to 'index' if no method specified
$id = $uriSegments[2] ?? null; // For resource IDs (e.g., /users/1)
$subId = $uriSegments[3] ?? null; // For nested resources (e.g., /reports/RPT-123/media/1)


// Simple Routing Logic
// This switch statement maps URI segments to specific controller methods.
// Extend this to include all your API routes.
switch ($controllerSegment) {
    case 'auth':
        $controller = new AuthController($conn);
        if ($methodSegment === 'login' && $requestMethod === 'POST') {
            $controller->login(); // Already Check
        } elseif ($methodSegment === 'forgot-password' && $requestMethod === 'POST') {
            $controller->forgotPassword();
        } else {
            sendJsonResponse(['message' => 'Auth endpoint not found or method not allowed.'], 404);//Already Check
        }
        break;
    
    case 'quickstats':
        $controller = new QuickStatsController($conn);
        if($methodSegment === 'admin' && $requestMethod === 'GET'){
            $controller->AdminQuickStats();
        } elseif ($methodSegment === 'engineer'&& $requestMethod === 'GET') {
            $controller->EngineerQuickStats();
        } elseif ($methodSegment === 'customer' && $requestMethod === 'GET') {
            $controller->CustomerQuickStats();
        } else {
            sendJsonResponse(['message' => 'QuickStats endpoint not found or method not allowed.'], 404);//Already Check
        }
        break;

    case 'users':
        $controller = new UserController($conn);
        if ($methodSegment === 'create' && $requestMethod === 'POST') {
            $controller->create();// Already Check
        } elseif ($methodSegment === 'readall' && $requestMethod === 'GET') {
            $controller->readAll();// Already Check
        } elseif ($methodSegment === 'read' && $id && $requestMethod === 'GET') {
            $controller->read($id);// Already Check
        } elseif ($methodSegment === 'update' && $id && $requestMethod === 'PUT') {
            $controller->update($id);// Already Check
        } elseif ($methodSegment === 'delete' && $id && $requestMethod === 'DELETE') {
            $controller->delete($id); // Already Check & Still need to Soft delete
        } elseif ($methodSegment === 'active' && $id && $requestMethod === 'DELETE') {
            $controller->active($id); // Already Check & Still need to Soft delete
        } elseif ($methodSegment === 'deactive' && $id && $requestMethod === 'DELETE') {
            $controller->deactive($id); // Already Check & Still need to Soft delete
        } elseif ($methodSegment === 'profile' && $id && $requestMethod === 'PUT') {
            // Special endpoint for profile update by user himself
            $controller->updateProfile($id);// Already Check
        } elseif ($methodSegment === 'save-token' && $id && $requestMethod === 'POST') { // NEW: Endpoint to save device token
            $controller->saveDeviceToken($id);
        }
        else {
            sendJsonResponse(['message' => 'User endpoint not found or method not allowed.'], 404);
        }
        break;

    case 'country':
        $controller = new CountryController($conn);
        if ($methodSegment === 'create' && $requestMethod === 'POST') {
            $controller->create();// Already Check
        } elseif ($methodSegment === 'readall' && $requestMethod === 'GET') {
            $controller->readAll(); // Already Check
        } elseif ($methodSegment === 'read' && $id && $requestMethod === 'GET') {
            $controller->read($id); // Already Check
        } elseif ($methodSegment === 'update' && $id && $requestMethod === 'PUT') {
            $controller->update($id); // Already Check
        } elseif ($methodSegment === 'delete' && $id && $requestMethod === 'DELETE') {
            $controller->delete($id); // Already Check & Still need to soft delete
        } else {
            sendJsonResponse(['message' => 'Country endpoint not found or method not allowed.'], 404);
        }
        break;

    case 'products':
        $controller = new ProductController($conn);
        if ($methodSegment === 'create' && $requestMethod === 'POST') {
            $controller->create();// Already Check
        } elseif ($methodSegment === 'readall' && $requestMethod === 'GET') {
            $controller->readAll(); // Already Check
        } elseif ($methodSegment === 'unassign-readall' && $requestMethod === 'GET') {
            $controller->UnassignreadAll(); // Already Check
        } elseif ($methodSegment === 'read' && $id && $requestMethod === 'GET') {
            $controller->read($id);// Already Check
        } elseif ($methodSegment === 'update' && $id && $requestMethod === 'PUT') {
            $controller->update($id); // Already Check
        } elseif ($methodSegment === 'delete' && $id && $requestMethod === 'DELETE') {
            $controller->delete($id); // Already Check & Still Need to soft delete
        } else { 
            sendJsonResponse(['message' => 'Product endpoint not found or method not allowed.'], 404);
        }
        break;

    case 'assign-engineer': // Changed endpoint name
    $controller = new AssignEngineerController($conn); // Changed controller class
    if ($methodSegment === 'create' && $requestMethod === 'POST') {
        $controller->create();
    } elseif ($methodSegment === 'readall' && $requestMethod === 'GET') {
        $controller->readAll();
    } elseif ($methodSegment === 'read' && $id && $requestMethod === 'GET') {
        $controller->read($id);
    } elseif ($methodSegment === 'update' && $id && $requestMethod === 'PUT') {
        $controller->update($id);
    } elseif ($methodSegment === 'delete' && $id && $requestMethod === 'DELETE') {
        $controller->delete($id);
    } else {
        sendJsonResponse(['message' => 'Assign Engineer endpoint not found or method not allowed.'], 404);
    }
    break;

    case 'customer-products':
        $controller = new CustomerProductController($conn);
        if ($methodSegment === 'assign' && $requestMethod === 'POST') {
            $controller->assignProduct();
        } elseif ($methodSegment === 'readall' && $requestMethod === 'GET') {
            $controller->readAll();
        } elseif ($methodSegment === 'read' && $id && $requestMethod === 'GET') {
            $controller->read($id);
        } elseif ($methodSegment === 'update' && $id && $requestMethod === 'PUT') {
            $controller->update($id);
        } elseif ($methodSegment === 'delete' && $id && $requestMethod === 'DELETE') {
            $controller->delete($id);
        } else {
            sendJsonResponse(['message' => 'Customer Products endpoint not found or method not allowed.'], 404);
        }
        break;

   case 'maintenance':
    $controller = new MaintenanceController($conn);
    if ($methodSegment === 'create' && $requestMethod === 'POST') {
        $controller->create();
    } elseif ($methodSegment === 'createPlan' && $requestMethod === 'POST') {
        $controller->createPlan();
    }  elseif ($methodSegment === 'delete' && $id && $requestMethod === 'DELETE') {
        $controller->delete($id);
    } elseif ($methodSegment === 'readall' && $requestMethod === 'GET') {
        $controller->readAll();
    } elseif ($methodSegment === 'update' && $id && $requestMethod === 'POST') {
        $controller->update($id);
    } elseif ($methodSegment === 'details-readall' && $id && $requestMethod === 'GET') {
        $controller->DetailsreadAll($id);
    } elseif ($methodSegment === 'read' && $id && $requestMethod === 'GET') {
        $controller->read($id);
    } elseif ($methodSegment === 'get-types' && $requestMethod === 'GET') {
        // Extract plan_id from query parameters (e.g., ?plan_id=X)
        $planId = $_GET['plan_id'] ?? null;
        if ($planId) {
            $controller->getPlanTypes($planId);
        } else {
            sendJsonResponse(['message' => 'Missing plan_id parameter.'], 400);
        }
    } elseif ($methodSegment === 'download-pdf' && $id && $requestMethod === 'GET') { // NEW: PDF download endpoint
        $controller->generatePdf($id);
    } else {
        sendJsonResponse(['message' => 'Maintenance endpoint not found or method not allowed.'], 404);
    }
    break;

    case 'reports':
        $controller = new ReportController($conn);
        if ($methodSegment === 'create' && $requestMethod === 'POST') {
            $controller->create(); // Already Check
        } elseif ($methodSegment === 'readall' && $requestMethod === 'GET') {
            $controller->readAll(); 
        }
        elseif ($methodSegment === 'complete-readall' && $requestMethod === 'GET') {
            $controller->readCompletedAll(); // Already Check (admin)
        }
        elseif ($methodSegment === 'uncomplete-readall' && $requestMethod === 'GET') {
            $controller->readUncompletedAll(); // Already Check (admin)
        }elseif ($methodSegment === 'recents' && $requestMethod === 'GET') {
            $controller->recents(); // Already Check (admin)
        } elseif ($methodSegment === 'read' && $id && $requestMethod === 'GET') {
            $controller->readSingle($id);// Already Check (admin,customer)
        } elseif ($methodSegment === 'pickup' && $id && $requestMethod === 'PUT') {
            $controller->pickupReport($id);// Already Check
        } elseif ($methodSegment === 'solve' && $id && $requestMethod === 'PUT') {
            $controller->solve($id); 
        } elseif ($methodSegment === 'complete' && $id && $requestMethod === 'PUT') {
            $controller->complete($id);
        } elseif ($methodSegment === 'download-pdf' && $id && $requestMethod === 'GET') {
             $controller->generatePdf($id);
        } elseif ($methodSegment === 'check-pdf-ready' && $id && $requestMethod === 'GET') {
            // This is a new endpoint to allow Flutter to check if PDF is ready.
            // Implement this method in your ReportController if needed,
            // otherwise, the Flutter app will directly attempt download.
            sendJsonResponse(['status' => 'ready', 'message' => 'PDF endpoint check is a placeholder.'], 200);
        }
        else {
            sendJsonResponse(['message' => 'Report endpoint not found or method not allowed.'], 404);
        }
        break;

    case 'report-media': // Assuming separate endpoint for media uploads for reports
        $controller = new ReportMediaController($conn);
        if ($methodSegment === 'upload' && $requestMethod === 'POST') {
            $controller->uploadMedia();
        } elseif ($methodSegment === 'read-by-report' && $id && $requestMethod === 'GET') {
            $controller->readMediaByReport($id);
        } elseif ($methodSegment === 'delete' && $id && $requestMethod === 'DELETE') {
             $controller->delete($id); // Assuming a delete method for media
        }
        else {
            sendJsonResponse(['message' => 'Report Media endpoint not found or method not allowed.'], 404);
        }
        break;

    case 'root-cause':
        $controller = new RootCauseController($conn);
        if ($methodSegment === 'create' && $requestMethod === 'POST') {
            $controller->create();
        } elseif ($methodSegment === 'readall' && $requestMethod === 'GET') {
            // Existing readAll, might not be needed if always filtered by report
            $controller->readAll();
        } elseif ($methodSegment === 'readall-by-report' && $id && $requestMethod === 'GET') {
            // New endpoint to read all root causes for a specific report_index
            $controller->readAllByReportIndex($id); // $id here is report_index
        } elseif ($methodSegment === 'read' && $id && $requestMethod === 'GET') {
            $controller->read($id);
        } elseif ($methodSegment === 'update' && $id && $requestMethod === 'POST') {
            $controller->update($id);
        } elseif ($methodSegment === 'delete' && $id && $requestMethod === 'DELETE') {
            $controller->delete($id);
        } else {
            sendJsonResponse(['message' => 'Root cause endpoint not found or method not allowed.'], 404);
        }
        break;

    case 'corrective-action':
        $controller = new CorrectiveActionController($conn);
        if ($methodSegment === 'create' && $requestMethod === 'POST') {
            $controller->create();
        } elseif ($methodSegment === 'readall' && $requestMethod === 'GET') {
            // Existing readAll, might not be needed if always filtered by report
            $controller->readAll();
        } elseif ($methodSegment === 'readall-by-report' && $id && $requestMethod === 'GET') {
            // New endpoint to read all root causes for a specific report_index
            $controller->readAllByReportIndex($id); // $id here is report_index
        } elseif ($methodSegment === 'read' && $id && $requestMethod === 'GET') {
            $controller->read($id);
        } elseif ($methodSegment === 'update' && $id && $requestMethod === 'POST') {
            $controller->update($id);
        } elseif ($methodSegment === 'delete' && $id && $requestMethod === 'DELETE') {
            $controller->delete($id);
        } else {
            sendJsonResponse(['message' => 'Corrective Action endpoint not found or method not allowed.'], 404);
        }
        break;

    case 'effective-action':
        $controller = new EffectiveActionController($conn);
        if ($methodSegment === 'create' && $requestMethod === 'POST') {
            $controller->create();
        } elseif ($methodSegment === 'readall' && $requestMethod === 'GET') {
            // Existing readAll, might not be needed if always filtered by report
            $controller->readAll();
        } elseif ($methodSegment === 'readall-by-report' && $id && $requestMethod === 'GET') {
            // New endpoint to read all root causes for a specific report_index
            $controller->readAllByReportIndex($id); // $id here is report_index
        } elseif ($methodSegment === 'read' && $id && $requestMethod === 'GET') {
            $controller->read($id);
        } elseif ($methodSegment === 'update' && $id && $requestMethod === 'POST') {
            $controller->update($id);
        } elseif ($methodSegment === 'delete' && $id && $requestMethod === 'DELETE') {
            $controller->delete($id);
        } else {
            sendJsonResponse(['message' => 'Effective Action endpoint not found or method not allowed.'], 404);
        }
        break;

    case 'prevention':
        $controller = new PreventionController($conn);
        if ($methodSegment === 'create' && $requestMethod === 'POST') {
            $controller->create();
        } elseif ($methodSegment === 'update' && $id && $requestMethod === 'POST') {
            $controller->update($id);
        } elseif ($methodSegment === 'delete' && $id && $requestMethod === 'DELETE') {
            $controller->delete($id);
        } elseif ($methodSegment === 'read-by-report' && $id && $requestMethod === 'GET') {
            $controller->readByReportIndex($id);
        } else {
            sendJsonResponse(['message' => 'Prevention endpoint not found or method not allowed.'], 404);
        }
        break;

    case 'suggestion':
        $controller = new SuggestionController($conn);
        if ($methodSegment === 'create' && $requestMethod === 'POST') {
            $controller->create();
        } elseif ($methodSegment === 'update' && $id && $requestMethod === 'POST') {
            $controller->update($id);
        } elseif ($methodSegment === 'delete' && $id && $requestMethod === 'DELETE') {
            $controller->delete($id);
        } elseif ($methodSegment === 'read-by-report' && $id && $requestMethod === 'GET') {
            $controller->readByReportIndex($id);
        } else {
            sendJsonResponse(['message' => 'Suggestion endpoint not found or method not allowed.'], 404);
        }
        break;

    case 'warning-config':
        $controller = new WarningConfigController($conn);
        if ($methodSegment === 'readall' && $requestMethod === 'GET') {
            $controller->readAll();
        } elseif ($methodSegment === 'update' && $id && $requestMethod === 'PUT') {
            $controller->update($id);
        } else {
            sendJsonResponse(['message' => 'Warning Config endpoint not found or method not allowed.'], 404);
        }
        break;

    case 'terms':
        $controller = new TermsController($conn);
        if ($methodSegment === 'create' && $requestMethod === 'POST') {
            $controller->create();
        } elseif ($methodSegment === 'readall' && $requestMethod === 'GET') {
            $controller->readAll();
        } elseif ($methodSegment === 'read' && $id && $requestMethod === 'GET') {
            $controller->read($id);
        } elseif ($methodSegment === 'update' && $id && $requestMethod === 'PUT') {
            $controller->update($id);
        } elseif ($methodSegment === 'delete' && $id && $requestMethod === 'DELETE') {
            $controller->delete($id);
        } else {
            sendJsonResponse(['message' => 'Terms endpoint not found or method not allowed.'], 404);
        }
        break;
    
    case 'company':
        $controller = new CompanyController($conn);
        if ($methodSegment === 'create' && $requestMethod === 'POST') {
            $controller->create();
        } elseif ($methodSegment === 'readall' && $requestMethod === 'GET') {
            $controller->readAll();
        } elseif ($methodSegment === 'read' && $id && $requestMethod === 'GET') {
            $controller->read($id);
        } elseif ($methodSegment === 'update' && $id && $requestMethod === 'PUT') {
            $controller->update($id);
        } elseif ($methodSegment === 'delete' && $id && $requestMethod === 'DELETE') {
            $controller->delete($id);
        } else {
            sendJsonResponse(['message' => 'Company endpoint not found or method not allowed.'], 404);
        }
        break;

    case 'notifications':
        $controller = new NotificationController($conn);
        if ($methodSegment === 'read-user-notifications' && $id && $requestMethod === 'GET') {
            $controller->readUserNotifications($id); //Already Check
        } elseif ($methodSegment === 'mark-as-read' && $id && $requestMethod === 'PUT') {
            $controller->markAsRead($id);
        } else {
            sendJsonResponse(['message' => 'Notification endpoint not found or method not allowed.'], 404);
        }
        break;

    default:
        // Fallback for unhandled routes
        sendJsonResponse(['message' => 'API Endpoint Not Found'], 404);
        break;
}

// Close the database connection at the end of the script execution
$conn->close();
?>