<?php
// helpers/FcmHelper.php

// Manually include required files from the downloaded libraries
// helpers/FcmHelper.php

// Use the absolute path to the autoload file
// In fcm_helper.php located at /test/utils/fcm_helper.php
require_once ('/home/rehlbfiy/testing.rehlkocustomercare.com/test/vendor/google-api-php-client/vendor/autoload.php');


// The path may vary depending on your organization, adjust as needed.

use Google\Client;
use GuzzleHttp\Client as GuzzleClient;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Handler\CurlHandler;

class FcmHelper
{
    private Client $client;
    private string $projectId;

    public function __construct()
    {
        // Path to your service account JSON file
        $serviceAccountFile = __DIR__ . '/../rehlko-dd990-firebase-adminsdk-fbsvc-31cbf10716.json';
        
        if (!file_exists($serviceAccountFile)) {
            throw new Exception("Firebase service account JSON file not found at: " . $serviceAccountFile);
        }

        // Initialize Google Client with a Guzzle HTTP client
        $handler = new CurlHandler();
        $stack = HandlerStack::create($handler);
        $guzzleClient = new GuzzleClient(['handler' => $stack, 'verify' => false]);

        $this->client = new Client();
        $this->client->setHttpClient($guzzleClient);
        $this->client->setAuthConfig($serviceAccountFile);
        $this->client->addScope('https://www.googleapis.com/auth/firebase.messaging');

        // Get project ID from the auth config
        $authConfig = json_decode(file_get_contents($serviceAccountFile), true);
        $this->projectId = $authConfig['project_id'];
    }

    /**
     * Sends a Firebase Cloud Messaging notification.
     *
     * @param string $title The notification title.
     * @param string $body The notification body.
     * @param string $target The target topic (e.g., "admin_notifications").
     * @param array $data An associative array of custom key-value pairs for the data payload.
     * @return mixed The response from the FCM API.
     */
    public function sendNotification(string $title, string $body, string $target, array $data = []): mixed
    {
        try {
            // This method automatically handles token generation and caching.
            $accessToken = $this->client->fetchAccessTokenWithAssertion();
            $accessToken = $accessToken['access_token'];

            $headers = [
                'Authorization' => 'Bearer ' . $accessToken,
                'Content-Type' => 'application/json',
            ];

            $url = "https://fcm.googleapis.com/v1/projects/{$this->projectId}/messages:send";
            
            // Fix the data payload by ensuring all values are strings
            $stringifiedData = [];
            foreach ($data as $key => $value) {
                $stringifiedData[$key] = (string)$value;
            }
            
            $stringifiedData['click_action'] = "FLUTTER_NOTIFICATION_CLICK";
            
            $payload = [
                "message" => [
                    "topic" => $target,
                    "notification" => [
                        "title" => $title,
                        "body" => $body,
                    ],
                    "data" => $stringifiedData,
                ]
            ];

            $ch = curl_init($url);
            curl_setopt($ch, CURLOPT_HTTPHEADER, array_map(function($key, $value) {
                return "$key: $value";
            }, array_keys($headers), $headers));
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
            $response = curl_exec($ch);
            
            if (curl_errno($ch)) {
                $error_msg = curl_error($ch);
                curl_close($ch);
                throw new Exception("cURL Error: " . $error_msg);
            }
            
            $httpStatus = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);

            // Decode the response to check for Firebase API errors
            $responseData = json_decode($response, true);
            
            // Check for success or specific error codes
            if ($httpStatus === 200) {
                // Success
                error_log("FCM Success: " . print_r($responseData, true));
                return $responseData;
                
            } else {
                // Firebase API Error
                $errorMessage = "Unknown Firebase Error";
                if (isset($responseData['error']['message'])) {
                    $errorMessage = $responseData['error']['message'];
                }
                
                // You can add more detailed logging based on specific codes
                if ($httpStatus === 401) {
                    $errorMessage = "401 Unauthorized: Invalid or expired access token. " . $errorMessage;
                } elseif ($httpStatus === 403) {
                    $errorMessage = "403 Forbidden: Permissions issue. " . $errorMessage;
                } elseif ($httpStatus === 404) {
                    $errorMessage = "404 Not Found: Invalid Project ID or endpoint URL. " . $errorMessage;
                } elseif ($httpStatus >= 400 && $httpStatus < 500) {
                    $errorMessage = "Client Error ({$httpStatus}): " . $errorMessage;
                } elseif ($httpStatus >= 500) {
                    $errorMessage = "Server Error ({$httpStatus}): " . $errorMessage;
                }
                
                throw new Exception("Firebase API Request Failed: " . $errorMessage);
            }

        } catch (Exception $e) {
            error_log("FCM Error: " . $e->getMessage());
            return ["error" => $e->getMessage()];
        }
    }
}