
<?php
// firebase_auth.php

// You will need to use the JWT library you already have.
require_once __DIR__ . '/../vendor/firebase/php-jwt/src/JWT.php';
use Firebase\JWT\JWT;

/**
 * Retrieves a Firebase access token for server-side authentication.
 *
 * This function generates a JSON Web Token (JWT) signed with the Firebase
 * service account's private key and exchanges it for a short-lived access token
 * from Google's OAuth 2.0 endpoint. This token is required to make authenticated
 * calls to the Firebase Cloud Messaging (FCM) API.
 *
 * @return string|null The access token on success, or null on failure.
 */
function getFirebaseAccessToken() {
    // IMPORTANT: Replace this with the secure, absolute path to your service account JSON file.
    // The file should be stored outside of the web root for security.
    $credentialsFilePath = __DIR__ . '/../../rehlko-dd990-firebase-adminsdk-fbsvc-0d8eff48ad.json';
    if (!file_exists($credentialsFilePath)) {
        error_log("Firebase service account file not found at: " . $credentialsFilePath);
        return null;
    }
    $credentials = json_decode(file_get_contents($credentialsFilePath), true);

    // Google's required JWT payload for OAuth 2.0 authentication.
    $issuedAt = time();
    $expirationTime = $issuedAt + 3600; // Token expires in 1 hour
    $payload = [
        'iss' => $credentials['client_email'],
        'scope' => 'https://www.googleapis.com/auth/firebase.messaging',
        'aud' => 'https://oauth2.googleapis.com/token',
        'iat' => $issuedAt,
        'exp' => $expirationTime,
    ];

    // Sign the JWT with the private key using the RS256 algorithm.
    // This is the correct method for authenticating with Google's services.
    try {
        $jwt = JWT::encode($payload, $credentials['private_key'], 'RS256');
    } catch (Exception $e) {
        error_log("Failed to encode JWT: " . $e->getMessage());
        return null;
    }

    // Exchange the JWT for a short-lived access token using cURL.
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, 'https://oauth2.googleapis.com/token');
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query([
        'grant_type' => 'urn:ietf:params:oauth:grant-type:jwt-bearer',
        'assertion' => $jwt,
    ]));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    $response = curl_exec($ch);
    curl_close($ch);

    $responseData = json_decode($response, true);
    if (isset($responseData['access_token'])) {
        return $responseData['access_token'];
    } else {
        error_log("Failed to get Firebase access token: " . print_r($responseData, true));
        return null;
    }
}

/**
 * Sends a push notification to a specific device using Firebase Cloud Messaging (FCM).
 *
 * This function uses the access token obtained from getFirebaseAccessToken() to
 * make a request to the FCM API. It sends a simple notification with a title
 * and body, along with optional custom data.
 *
 * @param string $deviceToken The Firebase device registration token for the recipient.
 * @param string $title The title of the notification.
 * @param string $body The body text of the notification.
 * @param array $data An optional array of custom key-value pairs to send with the notification.
 * @return bool True if the notification was sent successfully, false otherwise.
 */
function sendFirebaseNotification($deviceToken, $title, $body, $data = []) {
    $accessToken = getFirebaseAccessToken();
    if (!$accessToken) {
        return false;
    }

    // IMPORTANT: Replace 'your-firebase-project-id' with your actual Firebase project ID.
    $projectId = 'your-firebase-project-id'; 
    $fcmUrl = "https://fcm.googleapis.com/v1/projects/{$projectId}/messages:send";

    // Build the FCM message payload.
    $message = [
        'message' => [
            'token' => $deviceToken,
            'notification' => [
                'title' => $title,
                'body' => $body,
            ],
            // 'data' payload for custom data
            'data' => $data,
        ],
    ];

    // Use cURL to send the notification to the FCM API.
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $fcmUrl);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Authorization: Bearer ' . $accessToken,
        'Content-Type: application/json'
    ]);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($message));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

    $response = curl_exec($ch);
    curl_close($ch);

    $responseData = json_decode($response, true);

    if (isset($responseData['name'])) {
        // The message was sent successfully if a 'name' field is present.
        return true;
    } else {
        error_log("Failed to send Firebase notification: " . print_r($responseData, true));
        return false;
    }
}
?>